# R program for processing up to one month of processed ~2.5-minute output from aircoa_raw_day.r to produce calibrated data and diagnostics 
#
# Copyright 2009 University Corporation for Atmospheric Research
#
# Britton Stephens, National Center for Atmospheric Research, 1850 Table Mesa Dr., Boulder, CO 80305, USA, (303)497-1018, stephens@ucar.edu 
#

#####
# FEF 1212
#Warning messages:
#1: In rbind(rep(cylid, nrow(clumpaves)), strftime(clumpdt,  ... :
#  number of columns of result is not a multiple of vector length (arg 2)
#2: mean(<data.frame>) is deprecated.
# Use colMeans() or sapply(*, mean) instead.


# define function name, options, and default values:
accncdys<-function(loc='NWR',unit='A',startchr='050901',endchr='050907',inflag='',outflag='',pcor=T,tcor=T,fcor=T,fcorset=-0.00074,ltmpco=0.5,sflco=70,ico=0.0,eco=1.0,png=F,daily=T,nlines=3,c13cor=T,skipupdate=T,logfile=T,calcalco=1.0,ltltco=2.0,hdp2009fix=F){
# loc is three letter location code
# unit is one letter AIRCOA unit name
# startchr is YYMMDD character string of first day of output to process
# endchr is YYMMDD character string of last day of output to process
# inflag is an optional flag attached to the input file names by aircoa_raw_day.r
# outflag is output flag to be attached to the output of this program
# pcor selects whether to apply the pressure smoother
# tcor selects whether to apply the box and Li820 temperature smoother
# fcor selects whether to apply a flushing correction
# ltmpco is the maximum allowed deviation from the median Li820 temperature in C
# sflco is the minimum allowed sample flow in sccm
# ico is the fraction of the day to be excluded at the start of the first day
# eco is the fraction of the day to be included at the start of the last day
# png selects png or bitmap processing for the output graphs (png processing is not available in batch mode, with bitmap processing output is still png files but of lower quality)
# daily selects whether to make calibrated timeseries and profile plots for every day in period
# nlines specifies how many inlet lines exist (excluding LT) *currently must be 3 or 5*

print(paste(system('date',T),paste('CONCDAYS starting',loc,startchr,endchr)))

if(logfile){
# start output and error logging:
log<-file(paste(loc,"_",startchr,"_",endchr,outflag,".log",sep=''),open="wt")
sink(log)
sink(log,type="message")
print(paste('aircoa_conc_days starting',loc,startchr,endchr))
print(system('date',T))
}

# R can only plot in current time zone
Sys.setenv(TZ="GMT")

# convert character dates to numbers:
start<-as.numeric(startchr)
end<-as.numeric(endchr)

# extract numeric year (-2000) from date:
year<-trunc(start/10000)

# define jog time period for gap checking and for inlet delay calculations: 
skipsec<-50 # seconds
navesec<-100 # seconds
njogsec<-skipsec+navesec

# handle case of no inflag (normal): 
if(inflag==''){inflag<-'.'}

# insert '_' before any specified outflag:
if(outflag!=''){outflag<-paste('_',outflag,sep='')}

# set up to convert from day of year to day since Dec. 31, 2004
firstyear<-5 # starting year for dating measurements using days
yrdays<-0 # total days in previous years that need to be added to doy
if(year>firstyear){
	for(i in c(firstyear:(year-1))){
		if(i/4-trunc(i/4)==0){ # leap year
			yrdays<-yrdays+366
		}else{
			yrdays<-yrdays+365
		}
	}
}


# READ IN AND CONCATENATE ALL FILES TO BE PROCESSED

# set up data list:
data<-list(doy=NULL,co2i=NULL,prs=NULL,ltmp=NULL,btmp=NULL,rh=NULL,sfl=NULL,fl1=NULL,fl2=NULL,fl3=NULL,fl4=NULL,fl5=NULL,fl6=NULL,
nlin=NULL,ngas=NULL,up=NULL,sdco2=NULL,sdprs=NULL,sdsfl=NULL,hs2=NULL,hs1=NULL,ls1=NULL,ls2=NULL,shpbn=NULL,lchk=NULL,pcorr=NULL,
co2p=NULL,co2t=NULL,co2f=NULL,lin0=NULL,lin1=NULL,cur0=NULL,cur1=NULL,cur2=NULL,co2c=NULL,co2n=NULL,co2final=NULL)

## all averages and standard deviations are nominally over 100 seconds every 2.5 minutes
## input values:
# doy is UTC day of year plus day fraction (noon on Jan. 1 = 1.5)
# co2i is the averaged Li820 CO2 value in approximate ppm
# prs is the averaged Li820 cell pressure in kPa
# ltmp is the averaged Li820 cell temperature in C 
# btmp is the averaged AIRCOA box temperature in C 
# rh is the averaged downstream (post-drying) RH in % 
# sfl is the averaged sample flow
# fl* is the averaged flow on lines 1 through 6
# nlin is the averaged sample line selected (1-6)
# ngas is the averaged analysis gas selected (1 = sample, 2 = LS2, 3 = LS1, 4 = HS1, 5 = HS2, 6 =  Guest Cal)
# up indicates an element of a 4-point calibration that is run in the order LS2, LS1, HS1, HS2 
# sdco2 is the standard deviation of the Li820 CO2 values in approximate ppm
# sdprs is the standard deviation of the Li820 cell pressure in kPa
# sdsfl is the standard deviation of the sample flow in sccm
# hs2 is the assigned value for the HS2 calibration cylinder
# hs1 is the assigned value for the HS1 calibration cylinder
# ls1 is the assigned value for the LS1 calibration cylinder
# ls2 is the assigned value for the LS2 calibration cylinder
# shpbn is the bin value to use in calculating jog shapes separately by what is being run
# lchk is the averaged indicator of whether the leak check valve was open or closed (0 or 1)
## derived values:
# pcorr is the coefficient multiplied by the pressure deviation for the pressure smoother
# co2p is the CO2 data after the pressure smoother step (whether applied or not)
# co2t is the CO2 data after the temperature smoother step (whether applied or not)
# co2f is the CO2 data after the flushing correction step (whether applied or not)
# lin0 is the intercept of the 4-hourly 4-point linear fits interpolated to all data times
# lin1 is the slope of the 4-hourly 4-point linear fits interpolated to all data times
# cur0 is the intercept of the 4-hourly 4-point 2nd-order fits interpolated to all data times
# cur1 is the linear coefficient of the 4-hourly 4-point 2nd-order fits interpolated to all data times
# cur2 is the 2nd-order coefficient of the 4-hourly 4-point 2nd-order fits interpolated to all data times
# co2c is the CO2 data after the interpolated 4-hourly 4-point calibrations have been applied
# co2n is the CO2 data from a 4-point calibration to entire record (for plotting only)
# co2final is the CO2 data after the the 30-min offset correction has been applied

# read in manual filter file
manfilt<-scan('../../aircoa_manfilt.txt',what=list(loc='',startdate='',starttime='',enddate='',endtime='',line=''),skip=1,flush=T)

files<-system(paste('ls ',substr(startchr,1,4),'/',loc,'/',loc,'*.out',sep=''),T)
print(files)

if(any(as.numeric(substr(files,14,19))>=start&as.numeric(substr(files,14,19))<=end)){ # do not run if no output files

if(as.numeric(substr(files[1],14,19))>=start){start<-as.numeric(substr(files[1],14,19))}
if(as.numeric(substr(files[length(files)],14,19))<=end){end<-as.numeric(substr(files[length(files)],14,19))}

for(i in c(1:length(files))){
	if (as.numeric(substr(files[i],14,19))>=start&as.numeric(substr(files[i],14,19))<=end&substr(files[i],20,20)==inflag){
        print(paste('Reading data from file ',files[i]))
	input<-scan(files[i],what=list(sec=0,co2=0,prs=0,ltmp=0,btmp=0,rh=0,sfl=0,fl1=0,fl2=0,fl3=0,fl4=0,fl5=0,fl6=0,
	nlin=0,ngas=0,lchk=0,pump=0,sdco2=0,sdprs=0,sdltmp=0,sdbtmp=0,sdrh=0,sdsfl=0,sdfl1=0,sdfl2=0,sdfl3=0,sdfl4=0,sdfl5=0,sdfl6=0,
	slco2=0,slprs=0,slltmp=0,slbtmp=0,slrh=0,slsfl=0,slfl1=0,slfl2=0,slfl3=0,slfl4=0,slfl5=0,slfl6=0,shpbn=0))


# FILTER OUT BAD DATA (NAs, manual, ltmp, sfl, btmp, ico, eco)

input<-data.frame(input)

input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
print(paste('Removed',length(attributes(input)$na.action),'record(s) with NA in them from',files[i],'at:'))
print(unique(attributes(input)$na.action))
}
attributes(input)$na.action<-NULL

# nlin is the averaged sample line selected (1-6)
# ngas is the averaged analysis gas selected (1 = sample, 2 = LS2, 3 = LS1, 4 = HS1, 5 = HS2, 6 =  Guest Cal)
manfilt$sdt=as.numeric(manfilt$startdate)+as.numeric(substr(manfilt$starttime,1,2))/24+as.numeric(substr(manfilt$starttime,3,4))/60/24
manfilt$edt=as.numeric(manfilt$enddate)+as.numeric(substr(manfilt$endtime,1,2))/24+as.numeric(substr(manfilt$endtime,3,4))/60/24
manfilt$nlin=rep(0,length(manfilt$loc))
manfilt$ngas=rep(0,length(manfilt$loc))
manfilt$nlin[substr(manfilt$line,1,1)=='L'&substr(manfilt$line,2,2)!='S'&substr(manfilt$line,2,2)!='T']=as.numeric(substr(manfilt$line[substr(manfilt$line,1,1)=='L'&substr(manfilt$line,2,2)!='S'&substr(manfilt$line,2,2)!='T'],2,2)) # e.g. 'L2'
manfilt$nlin[substr(manfilt$line,1,2)=='LT']=nlines+1 # LT
manfilt$ngas[substr(manfilt$line,1,2)=='LT']=1 # LT
manfilt$ngas[substr(manfilt$line,1,1)=='L'&substr(manfilt$line,2,2)!='S'&substr(manfilt$line,2,2)!='T']=1 # e.g. 'L2'
manfilt$ngas[substr(manfilt$line,1,2)=='HS']=as.numeric(substr(manfilt$line[substr(manfilt$line,1,2)=='HS'],3,3))+3 # e.g. 'HS2'
manfilt$ngas[substr(manfilt$line,1,2)=='LS']=4-as.numeric(substr(manfilt$line[substr(manfilt$line,1,2)=='LS'],3,3)) # e.g. 'LS1'
manfilt$ngas[substr(manfilt$line,1,2)=='GC']=6 # guest cal 
for(j in c(1:length(manfilt$loc))){
	if(manfilt$loc[j]==loc){
#		if(as.numeric(substr(files[i],14,19))>=manfilt$sdt[j]&as.numeric(substr(files[i],14,19))<=manfilt$edt[j]){
			if(manfilt$line[j]=='ALL'){
				input$co2[as.numeric(substr(files[i],14,19))+input$sec/86400>=manfilt$sdt[j]&as.numeric(substr(files[i],14,19))+input$sec/86400<=manfilt$edt[j]]<-NA
			} else {
				input$co2[as.numeric(substr(files[i],14,19))+input$sec/86400>=manfilt$sdt[j]&as.numeric(substr(files[i],14,19))+input$sec/86400<=manfilt$edt[j]&input$nlin==manfilt$nlin[j]&input$ngas==manfilt$ngas[j]]<-NA
			}
#		}
	}
}
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
        rem<-unique(attributes(input)$na.action)
        print(paste('Removed ',length(rem),' record(s) in manual filter periods from',files[i]))
        if(length(rem)>1){
                if((rem[2]-rem[1])!=1){
                        print(paste('at:',rem[1]))
                }else{
                        print(paste('from:',rem[1]))
                }
                if(length(rem)>2){
                        for(j in c(2:(length(rem)-1))){
                                if((rem[j]-rem[j-1])!=1&(rem[j+1]-rem[j])!=1){
                                        print(paste('at:',rem[j]))
                                }else if((rem[j]-rem[j-1])!=1){
                                        print(paste('from:',rem[j]))
                                }else if((rem[j+1]-rem[j])!=1){
                                        print(paste('to:',rem[j]))
                                }
                        }
                        if((rem[length(rem)]-rem[length(rem)-1])!=1){
                                print(paste('at:',rem[length(rem)]))
                        }else{
                                print(paste('to:',rem[length(rem)]))
                        }
                } else {
                        if((rem[2]-rem[1])!=1){
                                print(paste('at:',rem[2]))
                        }else{
                                print(paste('to:',rem[2]))
                        }
                }
        } else {
                print(paste('at:',rem[1]))
        }
}
attributes(input)$na.action<-NULL

medltmp<-median(input$ltmp,na.rm=T)
input$co2[abs(input$ltmp-medltmp)>ltmpco]<-NA
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
print(paste('Removed ',length(attributes(input)$na.action),' record(s) with Li-820 temperature anomaly > ',ltmpco,'C from',files[i],'at:'))
print(unique(attributes(input)$na.action))
}
attributes(input)$na.action<-NULL

input$co2[input$sfl<sflco&input$shpbn!=11&input$shpbn!=12]<-NA
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
print(paste('Removed ',length(attributes(input)$na.action),' record(s) with sample flow < ',sflco,' from',files[i],'at:'))
print(unique(attributes(input)$na.action))
}
attributes(input)$na.action<-NULL

input$co2[input$btmp*100-40>100]<-NA
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
print(paste('Removed ',length(attributes(input)$na.action),' record(s) with bad box temperature data at:'))
print(unique(attributes(input)$na.action))
}
attributes(input)$na.action<-NULL

## to fix HDP 2009 problems, remove first 2 values after every cal-sample or sample-cal switch
print(as.numeric(substr(files[i],14,19)))
if(hdp2009fix&as.numeric(substr(files[i],14,19))>=90518){ # in May, do not remove data until the 19th
	temp=input$ngas==1 # T for sample or LT, false for LS2-HS2
	temp2=c(F,diff(temp)!=0) # T for jog immediately following a switch
	temp3=c(F,F,diff(temp)!=0); temp3=temp3[1:(length(temp3)-1)] # T for jog 2 jogs following a switch 
	hdp2009filt=temp2|temp3 # T for first 2 jogs after a cal-sample switch
	input$co2[hdp2009filt]=NA
	input<-na.omit(input)
	if(length(attributes(input)$na.action)>=1){
        	print(paste('Removed ',length(attributes(input)$na.action),' record(s) following cal-sample and sample-cal switches from HDP in 2009 at:'))
	        print(unique(attributes(input)$na.action))
	}
	attributes(input)$na.action<-NULL
}

calcalfilt=rep(F,length(input$co2))
for(j in 2:5){
# get rid of all sample data within the cal jump
###### could narrow it if other cals look OK. . . .
	calind=c(1:length(input$co2))[input$ngas==j]
	if(length(calind)>1){
		for(k in c(2:length(calind))){
			if(abs(input$co2[calind[k-1]]-input$co2[calind[k]])>calcalco){
				calcalfilt[calind[k-1]:calind[k]]=T
			}
		}
	}
#	calcalfilt=abs(c(0,diff(input$co2[input$ngas==j])))>calcalco|abs(c(diff(input$co2[input$ngas==j]),0))>calcalco
#	input$co2[input$ngas==j][calcalfilt]=NA
}
input$co2[calcalfilt]=NA
# get rid of individual LTs (but not intervening sample data) on either side of jump
ltltfilt=abs(c(0,diff(input$co2[input$ngas==1&input$nlin==nlines+1])))>ltltco|abs(c(diff(input$co2[input$ngas==1&input$nlin==nlines+1]),0))>ltltco
input$co2[input$ngas==1&input$nlin==nlines+1][ltltfilt]=NA
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
	print(paste('Removed ',length(attributes(input)$na.action),' record(s) with cal-cal jumps at:'))
	print(unique(attributes(input)$na.action))
}
attributes(input)$na.action<-NULL

if (as.numeric(substr(files[i],14,19))==start){
	input$co2[input$sec/86400<ico]<-NA
}
if (as.numeric(substr(files[i],14,19))==end){
	input$co2[input$sec/86400>eco]<-NA
}
input<-na.omit(input)
if(length(attributes(input)$na.action)>=1){
	rem<-unique(attributes(input)$na.action)
	print(paste('Removed ',length(rem),' record(s) > eco or < ico from',files[i]))
	if(length(rem)>1){
		if((rem[2]-rem[1])!=1){
			print(paste('at:',rem[1]))
		}else{
			print(paste('from:',rem[1]))
		}
		if(length(rem)>2){
			for(j in c(2:(length(rem)-1))){
				if((rem[j]-rem[j-1])!=1&(rem[j+1]-rem[j])!=1){
					print(paste('at:',rem[j]))
				}else if((rem[j]-rem[j-1])!=1){
					print(paste('from:',rem[j]))
				}else if((rem[j+1]-rem[j])!=1){
					print(paste('to:',rem[j]))
				}
			}
			if((rem[length(rem)]-rem[length(rem)-1])!=1){
				print(paste('at:',rem[length(rem)]))
			}else{
				print(paste('to:',rem[length(rem)]))
			}
		} else {
			if((rem[2]-rem[1])!=1){
				print(paste('at:',rem[2]))
			}else{
				print(paste('to:',rem[2]))
			}
		}
	} else {
		print(paste('at:',rem[1]))
	}
}
attributes(input)$na.action<-NULL


# CALCULATE DAY OF YEAR AND SELECT VARIABLES

dpm<-c(31,28,31,30,31,30,31,31,30,31,30,31)
if((as.numeric(substr(files[i],14,15))/4-trunc(as.numeric(substr(files[i],14,15))/4))==0){dpm[2]<-29}

if(as.numeric(substr(files[i],16,17))>1){
	doy<-sum(dpm[1:(as.numeric(substr(files[i],16,17))-1)])+as.numeric(substr(files[i],18,19))
} else { # January
	doy<-as.numeric(substr(files[i],18,19))
}

data$doy<-c(data$doy,input$sec/86400+doy)
data$co2i<-c(data$co2i,input$co2)
data$prs<-c(data$prs,input$prs)
data$ltmp<-c(data$ltmp,input$ltmp)
data$btmp<-c(data$btmp,input$btmp)
data$rh<-c(data$rh,input$rh)
data$sfl<-c(data$sfl,input$sfl)
data$fl1<-c(data$fl1,input$fl1)
data$fl2<-c(data$fl2,input$fl2)
data$fl3<-c(data$fl3,input$fl3)
data$fl4<-c(data$fl4,input$fl4)
data$fl5<-c(data$fl5,input$fl5)
data$fl6<-c(data$fl6,input$fl6)
data$nlin<-c(data$nlin,input$nlin)
data$ngas<-c(data$ngas,input$ngas)
data$sdco2<-c(data$sdco2,input$sdco2)
data$sdprs<-c(data$sdprs,input$sdprs)
data$sdsfl<-c(data$sdsfl,input$sdsfl)
data$shpbn<-c(data$shpbn,input$shpbn)
data$lchk<-c(data$lchk,input$lchk)

	}
}

print(paste('Read in a total of ',length(data$doy),' records.'))

if(length(data$doy)>0){ # do not run if no raw_day output

# READ IN AND APPLY CLOCK DRIFT CORRECTIONS

print('Applying clock drift corrections')

timeoff<-scan('../../aircoa_timeoff.txt',what=list(loc='',unit='',date='',time='',off=0),skip=1,flush=T)

# convert date and time into Day of Year and Day Fraction
timeoff$doy<-rep(NA,length(timeoff$loc))
# dpm defined above
for(j in c(1:length(timeoff$loc))){
	if(as.numeric(substr(timeoff$date[j],3,4))>1){ # if after January
		timeoff$doy[j]<-sum(dpm[1:(as.numeric(substr(timeoff$date[j],3,4))-1)])+as.numeric(substr(timeoff$date[j],5,6))+((as.numeric(substr(timeoff$time[j],1,2))*60
			+as.numeric(substr(timeoff$time[j],4,5)))*60+as.numeric(substr(timeoff$time[j],7,8)))/86400
	} else {
		timeoff$doy[j]<-as.numeric(substr(timeoff$date[j],5,6))+((as.numeric(substr(timeoff$time[j],1,2))*60+as.numeric(substr(timeoff$time[j],4,5)))*60+as.numeric(substr(timeoff$time[j],7,8)))/86400
	}
	# convert from day of year to day since Dec. 31, 2004
	if(as.numeric(substr(timeoff$date[j],1,2))>firstyear){
		for(k in c(firstyear:(as.numeric(substr(timeoff$date[j],1,2))-1))){
			if(k/4-trunc(k/4)==0){ # leap year
				timeoff$doy[j]<-timeoff$doy[j]+366
			}else{
				timeoff$doy[j]<-timeoff$doy[j]+365
			}
		}
	}
}

# convert from day of year to day since Dec. 31, 2004
# yrdays calculated above
tempdoy<-data$doy+yrdays # for AIRCOA data, do this again below

if(length(timeoff$doy[timeoff$loc==loc])>1){
	timecorr<-approx(timeoff$doy[timeoff$loc==loc],timeoff$off[timeoff$loc==loc],tempdoy,rule=2)$y/86400
}else if(length(timeoff$doy[timeoff$loc==loc])==1){
	timecorr<-rep(timeoff$off[timeoff$loc==loc],length(tempdoy))/86400
} else {
	timecorr<-rep(0,length(tempdoy))/86400
}

olddoy<-data$doy
data$doy<-data$doy-timecorr

data$doy[trunc(olddoy)<trunc(data$doy)&trunc(olddoy)==trunc(max(olddoy))]<-trunc(max(olddoy))+0.9999 # time correction has pushed data past midnight on last day, put 9 seconds into last day
if(any(trunc(olddoy)<trunc(data$doy)&trunc(olddoy)==trunc(max(olddoy)))){
	print(paste('Shifted ',sum(trunc(olddoy)<trunc(data$doy)&trunc(olddoy)==trunc(max(olddoy))),' record(s) with time correction after midnight on last day'))
}
data$doy[trunc(olddoy)>trunc(data$doy)&trunc(olddoy)==trunc(min(olddoy))]<-trunc(min(olddoy))+0.0001 # time correction has pushed data before midnight on first day, put 9 seconds into first day
if(any(trunc(olddoy)>trunc(data$doy)&trunc(olddoy)==trunc(min(olddoy)))){
	print(paste('Shifted ',sum(trunc(olddoy)>trunc(data$doy)&trunc(olddoy)==trunc(min(olddoy))),' record(s) with time correction before midnight on first day'))
}

# Generate DateTime values
datetime=as.POSIXlt(strptime(paste(year,'-01-01 00:00:00',sep=''),format='%y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*(data$doy-1))
datetimeyrdays=rep(365,length(data$doy))
datetimeyrdays[(datetime$year+1900)/4-trunc((datetime$year+1900)/4)==0]=366
data$yrfrac=round(datetime$year+1900+as.numeric(difftime(datetime,strptime(paste(datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/datetimeyrdays,7)


# READ IN CALIBRATION DATA
print("Reading calibration data")

# cylrec is the record of all cylinders for all locations
# cylrec2 is the record of all installed cylinders for this location expanded to inlude lines for cylinder removal
# caldat are the laboratory measurements of all cylinders
# caldat2 are the laboratory measurements of an individual cylinder
# calvals2 are the assigned (interpolated) values for the installed cylinders at the same times as cylrec2
# calvals3 are the assigned (interpolated) values for the installed cylinders at the same times as cals (below)
# calvals4 are averages of the assigned (interpolated) values for the installed cylinders for plotting (below)

# read in all laboratory measurements
#caldat=data.frame(scan('../../aircoa_caldat.txt',what=list(id='',date='',run='',pos='',gmdco2=0,siggmd=0,nco2=0),skip=1),stringsAsFactors=F)
### work with new columns?
if(F){ # replacing w/ aircoa_ave_cals.r
caldat=data.frame(scan('../../aircoa_caldat.txt',what=list(id='',date='',run='',pos='',gmdco2=0,siggmd=0,slpgmd=0,nco2=0),skip=1),stringsAsFactors=F)
caldat=caldat[!is.element(caldat$run,letters[14:26]),] # excludes experimental runs (n-z)
caldat=caldat[caldat$pos!='WT',] # excludes WT determinations
} else {
source('../../aircoa_ave_cals.r')
#write('id date gmdco2 siggmd avesig nrun ntot','aircoa_caldat_ave.txt')
caldat=read.table('aircoa_caldat_ave.txt',header=T,colClasses=c('character','character',NA,NA,NA,NA,NA))
}
## also want to be able to exclude bad runs
# apply 13C correction to any Scott-Marrin cylinders (see AO2 III p. 60) (different from AO2 code correction, which is to sample data)
# instead of adjusting sample data down, will adjust cal gases (and any SM LTs) down
###### need a list of all non-Scott-Marrin cylinders ever used (once calfac_cylinfo.txt implemented, read from there)
ambient13cid=c('JJ8510','JJ8856','JJ18942','JJ23832','JJ23428','JJ30259')  ### from Duane, only ambient for first fill, JJ8856 was refilled in 0901, others??
ambient13cid=c(ambient13cid,'CC200663','CC221523','CC185231','CC170769') ### from Dave B.
ambient13cid=c(ambient13cid,'CA07554','CA07556','CA07560','CA07671') ### Duane cyls, only ambient for first fill
ambient13cid=c(ambient13cid,'CC71620','CC82631') ### refilled in 0901
###### this list may not account for all refills by SM
ambient13cend=c('999999','090101','999999','999999','999999','999999')  ### from Duane, only ambient for first fill, JJ8856 was refilled in 0901, others??
ambient13cend=c(ambient13cend,'999999','999999','999999','999999') ### from Dave B.
ambient13cend=c(ambient13cend,'999999','999999','999999','999999') ### Duane cyls, only ambient for first fill
ambient13cend=c(ambient13cend,'090101','090101') ### refilled in 0901
if(c13cor){
#### LTs will get treated as unknown (uncorrected) - for sites/times with Scott-Marrin LTs, need to adjust in aircoa_comp.r and all_lt.r
        caldat$gmdco2[!is.element(caldat$id,ambient13cid)]=(1-0.028*0.011*1/3)*caldat$gmdco2[!is.element(caldat$id,ambient13cid)] # these cyls have always had scott marrin fill gas (light 13C)
### only print cyls used at this site/month?
        print(caldat$id[!is.element(caldat$id,ambient13cid)])
        print(caldat$date[!is.element(caldat$id,ambient13cid)])
	for(i in c(1:length(ambient13cid))){
	        caldat$gmdco2[caldat$id==ambient13cid[i]&as.numeric(caldat$date)>as.numeric(ambient13cend[i])]=(1-0.028*0.011*1/3)*caldat$gmdco2[caldat$id==ambient13cid[i]&as.numeric(caldat$date)>as.numeric(ambient13cend[i])] # was once an ambient 13C gas but since refilled by Scott-Marrin (now has light 13C)
### only print cyls used at this site/month?
                print(caldat$id[caldat$id==ambient13cid[i]&as.numeric(caldat$date)>as.numeric(ambient13cend[i])])
                print(caldat$date[caldat$id==ambient13cid[i]&as.numeric(caldat$date)>as.numeric(ambient13cend[i])])
	}
	print(paste('Cal gas data corrected for 13C in cal gases by ',round(-0.028*0.011*1/3*10000,2),'E-4 * [CO2] (',round(-0.028*0.011*1/3*380,2),' @ 380 ppm)',sep=''))
}

# read in all cylinder fill dates
cylfill=data.frame(scan('../../aircoa_cylfill.txt',what=list(date=0,id='')),stringsAsFactors=F)

# read in record of all calibration cylinders used for this site
cylrec<-data.frame(scan('../../aircoa_cylrec.txt',what=list(loc='',unit='',date='',time='',HS2='',HS1='',LS1='',LS2='',LT=''),skip=1),stringsAsFactors=F)

cylrec2=cylrec[cylrec$loc==loc,]
if(length(cylrec2$loc)<1){print('Location not found in aircoa_cylrec.txt')}
# calculate year.fraction
cylrec2datetime=strptime(paste(cylrec2$date,cylrec2$time),format='%y%m%d %H%M',tz="GMT")
cylrec2yrdays=rep(365,length(cylrec2$loc))
cylrec2yrdays[(cylrec2datetime$year+1900)/4-trunc((cylrec2datetime$year+1900)/4)==0]=366
cylrec2$yrfrac=round(cylrec2datetime$year+1900+as.numeric(difftime(cylrec2datetime,strptime(paste(cylrec2datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/cylrec2yrdays,7)
# insert rows for cylinders immediately before they are removed (and a row at end) ## does not look like code is adding a row at end when more than 1 row exists, just using rule=2 for approx below
cylrec2=data.frame(cylrec2,stringsAsFactors=F)
if(length(cylrec2$yrfrac)>1){
temp=cylrec2[1:(length(cylrec2$yrfrac)-1),] # all but last row
temp$yrfrac=cylrec2$yrfrac[2:length(cylrec2$yrfrac)]-1/60/24/365 # 1 minute earlier than next row
##temp$yrfrac=cylrec2$yrfrac[2:length(cylrec2$yrfrac)] # yrfrac of next row
temp$date=cylrec2$date[2:length(cylrec2$yrfrac)] # date of next row
temp$time=cylrec2$time[2:length(cylrec2$yrfrac)] # time of next row
cylrec2=rbind(cylrec2,temp)
cylrec2=cylrec2[order(cylrec2$yrfrac),]
## date/time will be 1 min ahead of yearfrac
} else {
cylrec2=rbind(cylrec2,cylrec2)
cylrec2$yrfrac[2]=max(data$yrfrac,na.rm=T)+1/60/24/365 # 1 minute later
}

# look up and interpolate calibrations for all cylinders used for this site to their time of installation and removal
##### only really need to do for cylinders online in this month??
calvals2<-NULL
for(j in c(1:length(cylrec2$loc))){
	calvals2$yrfrac<-c(calvals2$yrfrac,cylrec2$yrfrac[j])
	priorfill=max(c(cylfill$date[cylfill$id==cylrec2$HS2[j]&cylfill$date<as.numeric(cylrec2$date[j])],101),na.rm=T) # returns Jan. 1 2000 if no other fill recorded
	postfill=min(c(cylfill$date[cylfill$id==cylrec2$HS2[j]&cylfill$date>as.numeric(cylrec2$date[j])],991231),na.rm=T) # returns Dec. 31 2099 if no other fill recorded
	caldat2=caldat[caldat$id==cylrec2$HS2[j]&as.numeric(caldat$date)>priorfill&as.numeric(caldat$date)<postfill,]
	if(length(caldat2$id)>0){
		caldat2yrdays=rep(365,length(caldat2$id))
		caldat2datetime=strptime(caldat2$date,format='%y%m%d',tz="GMT")
		caldat2yrdays[(caldat2datetime$year+1900)/4-trunc((caldat2datetime$year+1900)/4)==0]=366
		caldat2$yrfrac=round(caldat2datetime$year+1900+as.numeric(difftime(caldat2datetime,strptime(paste(caldat2datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/caldat2yrdays,7) 
		# approx'ing between individual cals - do monthly lumping as in calfac_make_secondaries.r?
		if(length(caldat2$yrfrac)>1){
			calvals2$HS2<-c(calvals2$HS2,approx(caldat2$yrfrac,caldat2$gmdco2,cylrec2$yrfrac[j],rule=2)$y) 
		} else {
			calvals2$HS2=c(calvals2$HS2,caldat2$gmdco2)
		}
	} else {
		print(paste('Could not find cyinder',cylrec2$HS2[j],'in aircoa_caldat.txt'))	
		calvals2$HS2=c(calvals2$HS2,NA)
	}
#print(calvals2$HS2)
	priorfill=max(c(cylfill$date[cylfill$id==cylrec2$HS1[j]&cylfill$date<as.numeric(cylrec2$date[j])],101),na.rm=T) # returns Jan. 1 2000 if no other fill recorded
	postfill=min(c(cylfill$date[cylfill$id==cylrec2$HS1[j]&cylfill$date>as.numeric(cylrec2$date[j])],991231),na.rm=T) # returns Dec. 31 2099 if no other fill recorded
	caldat2=caldat[caldat$id==cylrec2$HS1[j]&as.numeric(caldat$date)>priorfill&as.numeric(caldat$date)<postfill,]
	if(length(caldat2$id)>0){
                caldat2yrdays=rep(365,length(caldat2$id))
                caldat2datetime=strptime(caldat2$date,format='%y%m%d',tz="GMT")
                caldat2yrdays[(caldat2datetime$year+1900)/4-trunc((caldat2datetime$year+1900)/4)==0]=366
                caldat2$yrfrac=round(caldat2datetime$year+1900+as.numeric(difftime(caldat2datetime,strptime(paste(caldat2datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/caldat2yrdays,7)
                # approx'ing between individual cals - do monthly lumping as in calfac_make_secondaries.r?
		if(length(caldat2$yrfrac)>1){
	                calvals2$HS1<-c(calvals2$HS1,approx(caldat2$yrfrac,caldat2$gmdco2,cylrec2$yrfrac[j],rule=2)$y)
		} else {
			calvals2$HS1=c(calvals2$HS1,caldat2$gmdco2)
		}
	} else {
		print(paste('Could not find cyinder',cylrec2$HS1[j],'in aircoa_caldat.txt'))	
		calvals2$HS1=c(calvals2$HS1,NA)
	}
#print(calvals2$HS1)
	priorfill=max(c(cylfill$date[cylfill$id==cylrec2$LS1[j]&cylfill$date<as.numeric(cylrec2$date[j])],101),na.rm=T) # returns Jan. 1 2000 if no other fill recorded
	postfill=min(c(cylfill$date[cylfill$id==cylrec2$LS1[j]&cylfill$date>as.numeric(cylrec2$date[j])],991231),na.rm=T) # returns Dec. 31 2099 if no other fill recorded
	caldat2=caldat[caldat$id==cylrec2$LS1[j]&as.numeric(caldat$date)>priorfill&as.numeric(caldat$date)<postfill,]
	if(length(caldat2$id)>0){
                caldat2yrdays=rep(365,length(caldat2$id))
                caldat2datetime=strptime(caldat2$date,format='%y%m%d',tz="GMT")
                caldat2yrdays[(caldat2datetime$year+1900)/4-trunc((caldat2datetime$year+1900)/4)==0]=366
                caldat2$yrfrac=round(caldat2datetime$year+1900+as.numeric(difftime(caldat2datetime,strptime(paste(caldat2datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/caldat2yrdays,7)
                # approx'ing between individual cals - do monthly lumping as in calfac_make_secondaries.r?
		if(length(caldat2$yrfrac)>1){
                	calvals2$LS1<-c(calvals2$LS1,approx(caldat2$yrfrac,caldat2$gmdco2,cylrec2$yrfrac[j],rule=2)$y)
		} else {
			calvals2$LS1=c(calvals2$LS1,caldat2$gmdco2)
		}
	} else {
		print(paste('Could not find cyinder',cylrec2$LS1[j],'in aircoa_caldat.txt'))	
		calvals2$LS1=c(calvals2$LS1,NA)
	}
#print(calvals2$LS1)
	priorfill=max(c(cylfill$date[cylfill$id==cylrec2$LS2[j]&cylfill$date<as.numeric(cylrec2$date[j])],101),na.rm=T) # returns Jan. 1 2000 if no other fill recorded
	postfill=min(c(cylfill$date[cylfill$id==cylrec2$LS2[j]&cylfill$date>as.numeric(cylrec2$date[j])],991231),na.rm=T) # returns Dec. 31 2099 if no other fill recorded
	caldat2=caldat[caldat$id==cylrec2$LS2[j]&as.numeric(caldat$date)>priorfill&as.numeric(caldat$date)<postfill,]
	if(length(caldat2$id)>0){
                caldat2yrdays=rep(365,length(caldat2$id))
                caldat2datetime=strptime(caldat2$date,format='%y%m%d',tz="GMT")
                caldat2yrdays[(caldat2datetime$year+1900)/4-trunc((caldat2datetime$year+1900)/4)==0]=366
                caldat2$yrfrac=round(caldat2datetime$year+1900+as.numeric(difftime(caldat2datetime,strptime(paste(caldat2datetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/caldat2yrdays,7)
                # approx'ing between individual cals - do monthly lumping as in calfac_make_secondaries.r?
		if(length(caldat2$yrfrac)>1){
                	calvals2$LS2<-c(calvals2$LS2,approx(caldat2$yrfrac,caldat2$gmdco2,cylrec2$yrfrac[j],rule=2)$y)
		} else {
			calvals2$LS2=c(calvals2$LS2,caldat2$gmdco2)
		}
	} else {
		print(paste('Could not find cyinder',cylrec2$LS2[j],'in aircoa_caldat.txt'))	
		calvals2$LS2=c(calvals2$LS2,NA)
	}
#print(calvals2$LS2)
}

# approximate cal values to all data
data$hs2<-approx(cylrec2$yrfrac,calvals2$HS2,data$yrfrac,rule=2)$y
data$hs1<-approx(cylrec2$yrfrac,calvals2$HS1,data$yrfrac,rule=2)$y
data$ls1<-approx(cylrec2$yrfrac,calvals2$LS1,data$yrfrac,rule=2)$y
data$ls2<-approx(cylrec2$yrfrac,calvals2$LS2,data$yrfrac,rule=2)$y

# PLOT AVE DIAGNOSTICS 

# bitmap command necessary to run in batch mode (png gets better resolution)
if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_diagnostics.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_diagnostics.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(4,3))
par(oma=c(2.5,0,2,0))
par(mar=c(2,3,2,1)+.1)
par(mgp=c(2,1,0))

plot(datetime[data$ngas!=0],data$co2i[data$ngas!=0],main='Li-820 CO2',ylab='Licor ppm',type='n')
points(datetime[data$ngas==1&data$lchk==0],data$co2i[data$ngas==1&data$lchk==0],col=2)
points(datetime[data$ngas>1],data$co2i[data$ngas>1],col=1)
plot(datetime[data$ngas!=0&data$lchk==0],data$prs[data$ngas!=0&data$lchk==0],main='Li-820 Pressure',ylab='kPa',type='n')
points(datetime[data$ngas==1&data$lchk==0],data$prs[data$ngas==1&data$lchk==0],col=3)
points(datetime[data$ngas>1],data$prs[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$ltmp[data$ngas!=0],main='Li-820 Temperature',ylab='degrees C',type='n')
points(datetime[data$ngas==1&data$lchk==0],data$ltmp[data$ngas==1&data$lchk==0],col=6)
points(datetime[data$ngas>1],data$ltmp[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$btmp[data$ngas!=0]*100-40,main='Ambient Box Temperature',ylab='degrees C',type='n')
points(datetime[data$ngas==1&data$lchk==0],data$btmp[data$ngas==1&data$lchk==0]*100-40,col=6)
points(datetime[data$ngas>1],data$btmp[data$ngas>1]*100-40,col=1)
plot(datetime[data$ngas!=0],data$rh[data$ngas!=0]*100,main='Gas Stream RH',ylab='percent',type='n')
points(datetime[data$ngas==1&data$lchk==0],data$rh[data$ngas==1&data$lchk==0]*100,col='orange')
points(datetime[data$ngas>1],data$rh[data$ngas>1]*100,col=1)
plot(datetime[data$ngas!=0],data$sfl[data$ngas!=0],main='Sample Flow',ylab='ml/min',type='n',ylim=c(-25,200))
points(datetime[data$ngas==1&data$lchk==0],data$sfl[data$ngas==1&data$lchk==0],col=4)
points(datetime[data$ngas>1],data$sfl[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$fl1[data$ngas!=0],main='Line 1 Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl1[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl1[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$fl2[data$ngas!=0],main='Line 2 Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl2[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl2[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$fl3[data$ngas!=0],main='Line 3 Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl3[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl3[data$ngas>1],col=1)
if(nlines==3){
plot(datetime[data$ngas!=0],data$fl4[data$ngas!=0],main='Line 4 (LT) Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl4[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl4[data$ngas>1],col=1)
}else if(nlines==5){
plot(datetime[data$ngas!=0],data$fl4[data$ngas!=0],main='Line 4 Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl4[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl4[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$fl5[data$ngas!=0],main='Line 5 Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl5[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl5[data$ngas>1],col=1)
plot(datetime[data$ngas!=0],data$fl6[data$ngas!=0],main='Line 6 (LT) Flow',ylab='ml/min',type='n',ylim=c(-100,1000))
points(datetime[data$ngas==1&data$lchk==0],data$fl6[data$ngas==1&data$lchk==0],col=5)
points(datetime[data$ngas>1],data$fl6[data$ngas>1],col=1)
}else{
print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Average Diagnostics'),3,0.5,outer=T)
mtext('Date (UTC)',1,1,T)

dev.off()

# CALCULATE AND APPLY CONCENTRATION AND TIME DEPENDENT PRESSURE SMOOTHER
### time dependency necessary?

print('Concentration and Time Dependent Pressure Smoother')

# bitmap command necessary to run in batch mode (png gets better resolution)
if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_pressure.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_pressure.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfcol=c(2,2))
par(mar=c(3,3,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(0,0,2,0))

if(any(data$lchk==1)){  ### temporary solution, need to change when new p check implemented

pvect<-rep(NA,2)
prevprs<-c(NA,data$prs[1:(length(data$prs)-1)])
prevco2<-c(NA,data$co2i[1:(length(data$co2i)-1)])
prevngas<-c(NA,data$ngas[1:(length(data$ngas)-1)])
prevdoy<-c(NA,data$doy[1:(length(data$ngas)-1)])
gapcheck<-data$doy-prevdoy

## needed?
#x<-data$prs[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]-prevprs[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]
#y<-data$co2i[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]-prevco2[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]
#z<-data$doy[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)] # with a 10% margin exclude any spanning across missing data

# all hs2 ambient p checks, low P - previous flowing P
x<-data$prs[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)]-prevprs[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)]
y<-data$co2i[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)]-prevco2[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)]
z<-data$doy[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)] # with a 10% margin exclude any spanning across missing data
dt=datetime[data$shpbn==12&prevngas==5&gapcheck<(njogsec/86400*1.1)]
slp<-y/x

if(length(x)>1){ # this can happen if HS2 missing for month
hs2exist=T
plot(x,y,main='CO2 vs. Pressure HS 2',ylab='delta Licor ppm',xlab='delta kPa',type='n',xlim=c(min(c(0,min(x,na.rm=T))),max(c(0,max(x,na.rm=T)))),ylim=c(min(c(0,min(y,na.rm=T))),max(c(0,max(y,na.rm=T)))))
points(x,y,col='Blue')
points(0,0,col='Black')
abline(lsfit(x,y,intercept=F))
t<-ls.print(lsfit(x,y,intercept=F),print.it=F)
mtext(paste('P. Dep. = ',round(t$coef.table[[1]][1,1],3)),3,-1.5)
mtext(paste('CO2 (0) = ',round(mean(data$co2i[data$ngas==5],na.rm=T),2)),1,-3)
mtext(paste('Prs. (0) = ',round(mean(data$prs[data$ngas==5],na.rm=T),2)),1,-1.5)
x<-z
y<-slp
plot(dt,y,main='Slope vs. Time HS 2',ylab='ppm/kPa',xlab='Date (UTC)',type='n',ylim=c(-2,2),xlim=range(as.POSIXct(datetime),na.rm=T))
points(dt,y,col='Blue')
pvect[1]<-t$coef.table[[1]][1,1]
if(pcor){
mod<-cbind(rep(1,length(x[!is.na(y)])),x[!is.na(y)],x[!is.na(y)]^2)
fiths2<-lm.fit(mod,y[!is.na(y)])
}
} else {
hs2exist=F
plot(c(1:3),c(1:3),main='CO2 vs. Pressure HS 2',ylab='delta Licor ppm',xlab='delta kPa',axes=F,type='n')
plot(c(1:3),c(1:3),main='Slope vs. Time HS 2',ylab='ppm/kPa',xlab='Date (UTC)',axes=F,type='n',xlim=range(as.POSIXct(datetime),na.rm=T))
if(pcor){
fiths2<-NULL
fiths2$coef<-NA
}
}

# all ls2 ambient p checks, low P - previous flowing P
x<-data$prs[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]-prevprs[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]
y<-data$co2i[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]-prevco2[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)]
z<-data$doy[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)] # with a 10% margin exclude any spanning across missing data
dt=datetime[data$shpbn==12&prevngas==2&gapcheck<(njogsec/86400*1.1)] 
slp<-y/x

if(length(x)>1){ # this can happen if LS2 missing for month
ls2exist=T
plot(x,y,main='CO2 vs. Pressure LS 2',ylab='delta Licor ppm',xlab='delta kPa',type='n',xlim=c(min(c(0,min(x,na.rm=T))),max(c(0,max(x,na.rm=T)))),ylim=c(min(c(0,min(y,na.rm=T))),max(c(0,max(y,na.rm=T)))))
points(x,y,col='Purple')
points(0,0,col='Black')
abline(lsfit(x,y,intercept=F))
t<-ls.print(lsfit(x,y,intercept=F),print.it=F)
mtext(paste('P. Dep. = ',round(t$coef.table[[1]][1,1],3)),3,-1.5)
mtext(paste('CO2 (0) = ',round(mean(data$co2i[data$ngas==2],na.rm=T),2)),1,-3)
mtext(paste('Prs. (0) = ',round(mean(data$prs[data$ngas==2],na.rm=T),2)),1,-1.5)
x<-z
y<-slp
plot(dt,y,main='Slope vs. Time LS 2',ylab='ppm/kPa',xlab='Date (UTC)',type='n',ylim=c(-2,2),xlim=range(as.POSIXct(datetime),na.rm=T))
points(dt,y,col='Purple')
pvect[2]<-t$coef.table[[1]][1,1]
if(pcor){
mod<-cbind(rep(1,length(x[!is.na(y)])),x[!is.na(y)],x[!is.na(y)]^2)
fitls2<-lm.fit(mod,y[!is.na(y)])
}
} else {
ls2exist=F
plot(c(1:3),c(1:3),main='CO2 vs. Pressure LS 2',ylab='delta Licor ppm',xlab='delta kPa',axes=F,type='n')
plot(c(1:3),c(1:3),main='Slope vs. Time LS 2',ylab='ppm/kPa',xlab='Date (UTC)',axes=F,type='n',xlim=range(as.POSIXct(datetime),na.rm=T))
if(pcor){
fitls2<-NULL
fitls2$coef<-NA
}
}

if(pcor){
if(sum(!is.na(pvect))>1&hs2exist&ls2exist){
	concdep<-lsfit(c(mean(data$co2i[data$ngas==5],na.rm=T),mean(data$co2i[data$ngas==2],na.rm=T)),pvect)$coef[2]
} else {
	concdep<-0
}
avconc<-mean(c(mean(data$co2i[data$ngas==5],na.rm=T),mean(data$co2i[data$ngas==2],na.rm=T)),na.rm=T)
#replace NaNs with 0s
for(i in c(1:3)){
	if(is.na(fiths2$coef[i])){fiths2$coef[i]<-0}
	if(is.na(fitls2$coef[i])){fitls2$coef[i]<-0}
}
avfit0<-mean(c(fiths2$coef[1],fitls2$coef[1]),na.rm=T)
avfit1<-mean(c(fiths2$coef[2],fitls2$coef[2]),na.rm=T)
avfit2<-mean(c(fiths2$coef[3],fitls2$coef[3]),na.rm=T)

data$pcorr<-avfit0+avfit1*data$doy+avfit2*data$doy^2+concdep*(data$co2i-avconc)

if(concdep<0){
print(paste('pcorr = ',round(avfit0,3),' + ',round(avfit1,3),' * doy + ',round(avfit2,3),' * doy^2 - ',round(concdep,6)*-1,' * (co2i-avconc) ppm/kPa'))
} else {
print(paste('pcorr = ',round(avfit0,3),' + ',round(avfit1,3),' * doy + ',round(avfit2,3),' * doy^2 + ',round(concdep,6),' * (co2i-avconc) ppm/kPa'))
}

# put lines on all plots (assumes xlim and ylim same for all)
if(!is.na(pvect[2])){
lines(datetime[data$ngas==2],data$pcorr[data$ngas==2])
}
par(mfg=c(2,1))
if(!is.na(pvect[1])){
lines(datetime[data$ngas==5],data$pcorr[data$ngas==5])
}

} else {

mtext('NOT APPLIED',3,-1,outer=T)
data$pcorr<-rep(0,length(data$doy))

}

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Concentration and Time Dependent Pressure Smoother (lines in lower panels)'),3,0.5,outer=T)

} else {
pcor<-F
print('no lchk data for pcor')
data$pcorr<-rep(0,length(data$doy))
}

dev.off()

caldump<-NULL

if(pcor){
avprs<-mean(data$prs,na.rm=T)
data$co2p<-data$co2i-data$pcorr*(data$prs-avprs)
caldump$prsconcdep<-concdep
caldump$prsavfit0<-avfit0
caldump$prsavfit1<-avfit1
caldump$prsavfit2<-avfit2
caldump$prsavconc<-avconc
caldump$prsavprs<-avprs
} else {
print('pcorr not applied')
data$co2p<-data$co2i
caldump$prsconcdep<-0
caldump$prsavfit0<-0
caldump$prsavfit1<-0
caldump$prsavfit2<-0
caldump$prsavconc<-0
caldump$prsavprs<-0
}

### not working
#lowpchecks<-data[data$shpbn==12]
#highpchecks<-data[data$shpbn==11]
#print(lowpchecks)
#print(highpchecks)
### plot leakchecks versus time

# now remove ambient and high P leak checks
data$ngas[data$shpbn==12]<-0
data$ngas[data$shpbn==11]<-0


# CALCULATE AND APPLY TEMPERATURE SMOOTHER

print('Temperature Smoother')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_temperature1.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_temperature1.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(3,3))
par(mar=c(3,3,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(0,0,2,0))

zgas<-data$ngas[data$ngas>1]
zdoy<-data$doy[data$ngas>1]
zdt=datetime[data$ngas>1]
zref<-rep(NA,length(zdoy))
zref[zgas==5]<-data$hs2[data$ngas>1][zgas==5]
zref[zgas==4]<-data$hs1[data$ngas>1][zgas==4]
zref[zgas==3]<-data$ls1[data$ngas>1][zgas==3]
zref[zgas==2]<-data$ls2[data$ngas>1][zgas==2]
zoff<-data$co2p[data$ngas>1]-zref
if(length(zgas)>0){
plot(zdt,zoff,type='n',main='Zero Offsets After Pressure Cal.',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],zoff[zgas==5],col='Blue')
points(zdt[zgas==4],zoff[zgas==4],col='Green')
points(zdt[zgas==3],zoff[zgas==3],col='Red')
points(zdt[zgas==2],zoff[zgas==2],col='Purple')
for(i in c(5,4,3,2)){
days<-unique(trunc(zdoy[zgas==i])) 
for(j in days){ 
daymean<-mean(zoff[zgas==i&zdoy>=j&zdoy<(j+1)],na.rm=T) 
zoff[zgas==i&zdoy>=j&zdoy<=(j+1)]<-zoff[zgas==i&zdoy>=j&zdoy<=(j+1)]-daymean 
segments(j,daymean,j+1,daymean) 
} 
}
plot(zdt,zoff,type='n',main='Detrended Zero Offsets',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],zoff[zgas==5],col='Blue')
points(zdt[zgas==4],zoff[zgas==4],col='Green')
points(zdt[zgas==3],zoff[zgas==3],col='Red')
points(zdt[zgas==2],zoff[zgas==2],col='Purple')

toff<-data$btmp[data$ngas>1]*100-40
plot(zdt,toff,type='n',main='Box Temperature',xlab='Date (UTC)',ylab='Degrees C')
points(zdt,toff,col='Brown')
days<-unique(trunc(zdoy)) 
for(j in days){ 
daymean<-mean(toff[zdoy>=j&zdoy<(j+1)],na.rm=T) 
toff[zdoy>=j&zdoy<=(j+1)]<-toff[zdoy>=j&zdoy<=(j+1)]-daymean 
segments(j,daymean,j+1,daymean) 
} 
plot(zdt,toff,type='n',main='Detrended Box Temperature',xlab='Date (UTC)',ylab='Degrees C')
points(zdt,toff,col='Brown')

txdat<-NULL 
tydat<-NULL 
if(length(toff[zgas==5])>1){
plot(toff[zgas==5],zoff[zgas==5],type='n',main='HS2 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==5],zoff[zgas==5],col='Blue')
abline(lsfit(toff[zgas==5],zoff[zgas==5]))
t<-ls.print(lsfit(toff[zgas==5],zoff[zgas==5]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
txdat<-c(txdat,toff[zgas==5]) 
tydat<-c(tydat,zoff[zgas==5]) 
} else if(length(toff[zgas==5])==1){
plot(toff[zgas==5],zoff[zgas==5],type='n',main='HS2 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==5],zoff[zgas==5],col='Blue')
txdat<-c(txdat,toff[zgas==5]) 
tydat<-c(tydat,zoff[zgas==5]) 
} else {
plot(c(1:3),c(1:3),main='HS2 vs. Box Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==4])>1){
plot(toff[zgas==4],zoff[zgas==4],type='n',main='HS1 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==4],zoff[zgas==4],col='Green')
abline(lsfit(toff[zgas==4],zoff[zgas==4]))
t<-ls.print(lsfit(toff[zgas==4],zoff[zgas==4]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
txdat<-c(txdat,toff[zgas==4]) 
tydat<-c(tydat,zoff[zgas==4]) 
} else if(length(toff[zgas==4])==1){
plot(toff[zgas==4],zoff[zgas==4],type='n',main='HS1 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==4],zoff[zgas==4],col='Green')
txdat<-c(txdat,toff[zgas==4]) 
tydat<-c(tydat,zoff[zgas==4]) 
} else {
plot(c(1:3),c(1:3),main='HS1 vs. Box Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==3])>1){
plot(toff[zgas==3],zoff[zgas==3],type='n',main='LS1 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==3],zoff[zgas==3],col='Red')
abline(lsfit(toff[zgas==3],zoff[zgas==3]))
t<-ls.print(lsfit(toff[zgas==3],zoff[zgas==3]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
txdat<-c(txdat,toff[zgas==3]) 
tydat<-c(tydat,zoff[zgas==3]) 
} else if(length(toff[zgas==3])==1){
plot(toff[zgas==3],zoff[zgas==3],type='n',main='LS1 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==3],zoff[zgas==3],col='Red')
txdat<-c(txdat,toff[zgas==3]) 
tydat<-c(tydat,zoff[zgas==3]) 
} else {
plot(c(1:3),c(1:3),main='LS1 vs. Box Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==2])>1){
plot(toff[zgas==2],zoff[zgas==2],type='n',main='LS2 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==2],zoff[zgas==2],col='Purple')
abline(lsfit(toff[zgas==2],zoff[zgas==2]))
t<-ls.print(lsfit(toff[zgas==2],zoff[zgas==2]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
txdat<-c(txdat,toff[zgas==2]) 
tydat<-c(tydat,zoff[zgas==2]) 
} else if(length(toff[zgas==2])==1){
plot(toff[zgas==2],zoff[zgas==2],type='n',main='LS2 vs. Box Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==2],zoff[zgas==2],col='Purple')
txdat<-c(txdat,toff[zgas==2]) 
tydat<-c(tydat,zoff[zgas==2]) 
} else {
plot(c(1:3),c(1:3),main='LS2 vs. Box Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Temperature Smoother - Box'),3,0.5,outer=T)

boxtoff<-toff

dev.off()

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_temperature2.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_temperature2.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(3,3))
par(mar=c(3,3,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(0,0,2,0))

toff<-data$ltmp[data$ngas>1]
plot(zdt,toff,type='n',main='Cell Temperature',xlab='Date (UTC)',ylab='Degrees C')
points(zdt,toff,col='Brown')

ctxdat<-NULL 
ctydat<-NULL 
if(length(toff[zgas==5])>1){
plot(toff[zgas==5],zoff[zgas==5],type='n',main='HS2 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==5],zoff[zgas==5],col='Blue')
abline(lsfit(toff[zgas==5],zoff[zgas==5]))
t<-ls.print(lsfit(toff[zgas==5],zoff[zgas==5]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
ctxdat<-c(ctxdat,toff[zgas==5]) 
ctydat<-c(ctydat,zoff[zgas==5]) 
} else if(length(toff[zgas==5])==1){
plot(toff[zgas==5],zoff[zgas==5],type='n',main='HS2 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==5],zoff[zgas==5],col='Blue')
ctxdat<-c(ctxdat,toff[zgas==5]) 
ctydat<-c(ctydat,zoff[zgas==5]) 
} else {
plot(c(1:3),c(1:3),main='HS2 vs. Cell Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==4])>1){
plot(toff[zgas==4],zoff[zgas==4],type='n',main='HS1 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==4],zoff[zgas==4],col='Green')
abline(lsfit(toff[zgas==4],zoff[zgas==4]))
t<-ls.print(lsfit(toff[zgas==4],zoff[zgas==4]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
ctxdat<-c(ctxdat,toff[zgas==4]) 
ctydat<-c(ctydat,zoff[zgas==4]) 
} else if(length(toff[zgas==4])==1){
plot(toff[zgas==4],zoff[zgas==4],type='n',main='HS1 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==4],zoff[zgas==4],col='Green')
ctxdat<-c(ctxdat,toff[zgas==4]) 
ctydat<-c(ctydat,zoff[zgas==4]) 
} else {
plot(c(1:3),c(1:3),main='HS1 vs. Cell Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==3])>1){
plot(toff[zgas==3],zoff[zgas==3],type='n',main='LS1 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==3],zoff[zgas==3],col='Red')
abline(lsfit(toff[zgas==3],zoff[zgas==3]))
t<-ls.print(lsfit(toff[zgas==3],zoff[zgas==3]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
ctxdat<-c(ctxdat,toff[zgas==3]) 
ctydat<-c(ctydat,zoff[zgas==3]) 
} else if(length(toff[zgas==3])==1){
plot(toff[zgas==3],zoff[zgas==3],type='n',main='LS1 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==3],zoff[zgas==3],col='Red')
ctxdat<-c(ctxdat,toff[zgas==3]) 
ctydat<-c(ctydat,zoff[zgas==3]) 
} else {
plot(c(1:3),c(1:3),main='LS1 vs. Cell Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}
if(length(toff[zgas==2])>1){
plot(toff[zgas==2],zoff[zgas==2],type='n',main='LS2 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==2],zoff[zgas==2],col='Purple')
abline(lsfit(toff[zgas==2],zoff[zgas==2]))
t<-ls.print(lsfit(toff[zgas==2],zoff[zgas==2]),print.it=F)
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5)
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3)
ctxdat<-c(ctxdat,toff[zgas==2]) 
ctydat<-c(ctydat,zoff[zgas==2]) 
} else if(length(toff[zgas==2])==1){
plot(toff[zgas==2],zoff[zgas==2],type='n',main='LS2 vs. Cell Temperature',xlab='Degrees C',ylab='delta CO2 (ppm)')
points(toff[zgas==2],zoff[zgas==2],col='Purple')
ctxdat<-c(ctxdat,toff[zgas==2]) 
ctydat<-c(ctydat,zoff[zgas==2]) 
} else {
plot(c(1:3),c(1:3),main='LS2 vs. Cell Temperature',ylab='Degrees C',xlab='delta CO2 (ppm)',type='n',xaxt='n',yaxt='n')
}


if(length(ctxdat)!=length(txdat)){
print('Error in box and cell temperature lengths')
print(ctxdat)
print(txdat)
}

plot(txdat,tydat,type='n',main='All vs. Box Temperature - Individual Fit',xlab='Degrees C',ylab='delta CO2 (ppm)') 
points(boxtoff[zgas==5],zoff[zgas==5],col='Blue') 
points(boxtoff[zgas==4],zoff[zgas==4],col='Green') 
points(boxtoff[zgas==3],zoff[zgas==3],col='Red') 
points(boxtoff[zgas==2],zoff[zgas==2],col='Purple') 
abline(lsfit(txdat,tydat)) 
t<-ls.print(lsfit(txdat,tydat),print.it=F) 
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5) 
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3) 

plot(ctxdat,tydat,type='n',main='All vs. Cell Temperature - Individual Fit',xlab='Degrees C',ylab='delta CO2 (ppm)') 
points(toff[zgas==5],zoff[zgas==5],col='Blue') 
points(toff[zgas==4],zoff[zgas==4],col='Green') 
points(toff[zgas==3],zoff[zgas==3],col='Red') 
points(toff[zgas==2],zoff[zgas==2],col='Purple') 
abline(lsfit(ctxdat,tydat)) 
t<-ls.print(lsfit(ctxdat,tydat),print.it=F) 
mtext(paste('T. Dep. = ',round(t$coef.table[[1]][2,1],3)),3,-1.5) 
mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],3)),3,-3) 

mod<-cbind(rep(1,length(txdat)),txdat,ctxdat)
fit<-lm.fit(mod,tydat)

plot(txdat,tydat,type='n',main='All vs. Box Temperature - Combined Fit',xlab='Degrees C',ylab='delta CO2 (ppm)') 
points(boxtoff[zgas==5],zoff[zgas==5],col='Blue') 
points(boxtoff[zgas==4],zoff[zgas==4],col='Green') 
points(boxtoff[zgas==3],zoff[zgas==3],col='Red') 
points(boxtoff[zgas==2],zoff[zgas==2],col='Purple') 
segments(min(boxtoff),min(boxtoff)*fit$coef[2],max(boxtoff),max(boxtoff)*fit$coef[2])
mtext(paste('T. Dep. = ',round(fit$coef[2],3)),3,-1.5) 

plot(ctxdat,tydat,type='n',main='All vs. Cell Temperature - Combined Fit',xlab='Degrees C',ylab='delta CO2 (ppm)') 
points(toff[zgas==5],zoff[zgas==5],col='Blue') 
points(toff[zgas==4],zoff[zgas==4],col='Green') 
points(toff[zgas==3],zoff[zgas==3],col='Red') 
points(toff[zgas==2],zoff[zgas==2],col='Purple') 
segments(min(toff),min(toff)*fit$coef[3]+fit$coef[1],max(toff),max(toff)*fit$coef[3]+fit$coef[1]) # assumes all of intercept is in cell temp fit (detrended box temp centers on zero)
mtext(paste('T. Dep. = ',round(fit$coef[3],3)),3,-1.5) 

tcorr1<-fit$coef[2]
tcorr2<-fit$coef[3]
if(is.na(tcorr1)){tcorr1<-0} # happens if box temperature is absolutely constant (usually because sensor missing or pegged)
if(is.na(tcorr2)){tcorr2<-0} # happens if cell temperature is absolutely constant
print(paste('tcorr = ',round(tcorr1,3),' ppm/boxC + ',round(tcorr2,3),' ppm/cellC'))
mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Temperature Smoother - Cell and Fits (',round(tcorr1,3),' ppm/boxC + ',round(tcorr2,3),' ppm/cellC)'),3,0.5,outer=T)

if(!tcor) {
mtext('NOT APPLIED',3,-1,outer=T)
}

} else {
tcor<-F
print('no cal data for tcor')
}

dev.off()

if(tcor){
	avbtmp<-mean(data$btmp,na.rm=T)*100-40
	avctmp<-mean(data$ltmp,na.rm=T)
	data$co2t<-data$co2p-tcorr1*(data$btmp*100-40-avbtmp)-tcorr2*(data$ltmp-avctmp)
	caldump$tmptcorr1<-tcorr1
	caldump$tmptcorr2<-tcorr2
	caldump$tmpavbtmp<-avbtmp
	caldump$tmpavctmp<-avctmp
} else {
	print('tcorr not applied')
	data$co2t<-data$co2p
	caldump$tmptcorr1<-0
	caldump$tmptcorr2<-0
	caldump$tmpavbtmp<-0
	caldump$tmpavctmp<-0
}



# CALCULATE FLUSHING EFFECT (CORRECTION CAN BE APPLIED)
print('Flushing Effect')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_flushing.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_flushing.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(2,2))
par(mar=c(3,3,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(0,0,2,0))

if(length(zgas)>0){
zoff<-data$co2t[data$ngas>1]-zref
plot(zdt,zoff,type='n',main='Zero Offsets After Temperature Cal.',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],zoff[zgas==5],col='Blue')
points(zdt[zgas==4],zoff[zgas==4],col='Green')
points(zdt[zgas==3],zoff[zgas==3],col='Red')
points(zdt[zgas==2],zoff[zgas==2],col='Purple')
for(i in c(5,4,3,2)){
days<-unique(trunc(zdoy[zgas==i])) 
for(j in days){ 
daymean<-mean(zoff[zgas==i&zdoy>=j&zdoy<(j+1)],na.rm=T) 
zoff[zgas==i&zdoy>=j&zdoy<=(j+1)]<-zoff[zgas==i&zdoy>=j&zdoy<=(j+1)]-daymean 
segments(j,daymean,j+1,daymean) 
} 
}

# cal current-previous
##### need to calculate concentration jump before filtering out bad data at start of code
prevall<-c(NA,data$co2t[1:(length(data$co2t)-1)])
prevcal<-prevall[data$ngas>1]
curmprv<-data$co2t-prevall
cmpcal<-curmprv[data$ngas>1]
# adjust all offsets using estimate of zero-effect point
library('MASS')
if(sum(zgas==5)>2){zoff[zgas==5]=zoff[zgas==5]-rlm(zoff[zgas==5] ~ cmpcal[zgas==5])$coef[1]}
if(sum(zgas==4)>2){zoff[zgas==4]=zoff[zgas==4]-rlm(zoff[zgas==4] ~ cmpcal[zgas==4])$coef[1]}
if(sum(zgas==3)>2){zoff[zgas==3]=zoff[zgas==3]-rlm(zoff[zgas==3] ~ cmpcal[zgas==3])$coef[1]}
if(sum(zgas==2)>2){zoff[zgas==2]=zoff[zgas==2]-rlm(zoff[zgas==2] ~ cmpcal[zgas==2])$coef[1]}

#plot(zdt,zoff,type='n',main='Detrended Zero Offsets',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
plot(zdt,zoff,type='n',main='Detrended and Zero-adjusted Offsets',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],zoff[zgas==5],col='Blue')
points(zdt[zgas==4],zoff[zgas==4],col='Green')
points(zdt[zgas==3],zoff[zgas==3],col='Red')
points(zdt[zgas==2],zoff[zgas==2],col='Purple')

plot(zdt,cmpcal,type='n',main='Current - Previous',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],cmpcal[zgas==5],col='Blue')
points(zdt[zgas==4],cmpcal[zgas==4],col='Green')
points(zdt[zgas==3],cmpcal[zgas==3],col='Red')
points(zdt[zgas==2],cmpcal[zgas==2],col='Purple')

### use both air-cal and cal-cal transitions?
plot(c(1:3),c(1:3),xlim=c(-150,150),ylim=range(zoff,na.rm=T),type='n',main='Offsets vs. (Current - Previous)',xlab='delta CO2 (ppm)',ylab='delta CO2 (ppm)') 
fxdat<-NULL 
fydat<-NULL 
if(length(prevcal[zgas==5])>=2){
points(cmpcal[zgas==5],zoff[zgas==5],col='Blue') 
rlmfit=rlm(zoff[zgas==5] ~ cmpcal[zgas==5])
if(abs(summary(rlmfit)$coef[2,3])>3&quantile(cmpcal[zgas==5],0.1,na.rm=T)<0&quantile(cmpcal[zgas==5],0.9,na.rm=T)>0&rlmfit$converged){ # t-value > 3, spans 0, and fit converged
abline(rlmfit,col='Blue')
fxdat<-c(fxdat,cmpcal[zgas==5])
fydat<-c(fydat,zoff[zgas==5])
}
}
if(length(prevcal[zgas==4])>=2){
points(cmpcal[zgas==4],zoff[zgas==4],col='Green') 
rlmfit=rlm(zoff[zgas==4] ~ cmpcal[zgas==4])
if(abs(summary(rlmfit)$coef[2,3])>3&quantile(cmpcal[zgas==4],0.1,na.rm=T)<0&quantile(cmpcal[zgas==4],0.9,na.rm=T)>0&rlmfit$converged){ # t-value > 3, spans 0, and fit converged
abline(rlmfit,col='Green')
fxdat<-c(fxdat,cmpcal[zgas==4]) 
fydat<-c(fydat,zoff[zgas==4]) 
}
}
if(length(prevcal[zgas==3])>=2){
points(cmpcal[zgas==3],zoff[zgas==3],col='Red') 
rlmfit=rlm(zoff[zgas==3] ~ cmpcal[zgas==3])
if(abs(summary(rlmfit)$coef[2,3])>3&quantile(cmpcal[zgas==3],0.1,na.rm=T)<0&quantile(cmpcal[zgas==3],0.9,na.rm=T)>0&rlmfit$converged){ # t-value > 3, spans 0, and fit converged
abline(rlmfit,col='Red')
fxdat<-c(fxdat,cmpcal[zgas==3]) 
fydat<-c(fydat,zoff[zgas==3]) 
}
}
if(length(prevcal[zgas==2])>=2){
points(cmpcal[zgas==2],zoff[zgas==2],col='Purple') 
rlmfit=rlm(zoff[zgas==2] ~ cmpcal[zgas==2])
if(abs(summary(rlmfit)$coef[2,3])>3&quantile(cmpcal[zgas==2],0.1,na.rm=T)<0&quantile(cmpcal[zgas==2],0.9,na.rm=T)>0&rlmfit$converged){ # t-value > 3, spans 0, and fit converged
abline(rlmfit,col='Purple')
fxdat<-c(fxdat,cmpcal[zgas==2]) 
fydat<-c(fydat,zoff[zgas==2]) 
}
}

if(length(fxdat)>1){
        rlmfit=rlm(fydat ~ fxdat)
        if(rlmfit$converged){
                abline(rlmfit)
                mtext(paste('Prev. Dep. = ',round(summary(rlmfit)$coef[2,1],4)),3,-1.5)
                mtext(paste('S. Err. = ',round(summary(rlmfit)$coef[2,2],4)),3,-3)
                fcorr<-summary(rlmfit)$coef[2,1]
        } else {
                fcorr<-0
### set SE to 0 as well? why often not converging?
        }
#abline(lsfit(fxdat,fydat)) 
#t<-ls.print(lsfit(fxdat,fydat),print.it=F) 
#mtext(paste('Prev. Dep. = ',round(t$coef.table[[1]][2,1],4)),3,-1.5) 
#mtext(paste('S. Err. = ',round(t$coef.table[[1]][2,2],4)),3,-3) 
#fcorr<-t$coef.table[[1]][2,1]
} else {
fcorr<-0
}

print(paste('fit fcorr = ',round(fcorr,5),'ppm/ppm +/-',round(summary(rlmfit)$coef[2,2],5),'at',median(data$sfl,na.rm=T),'sccm'))
mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Fit Flushing Effect (',round(fcorr,4),'ppm/ppm )'),3,0.5,outer=T)

if(!fcor){
mtext('NO CORRECTION',3,-1,outer=T)
} else {
if(!is.na(fcorset)){
### fcorset must be NA to use fcor from fit
mtext(paste('fixed correction =',fcorset),3,-1,outer=T)
abline(0,fcorset)
}
}

}else{
print('no cal data for fcor')
if(is.na(fcorset)){ fcor<-F }
}

dev.off()

if(fcor){
if(!is.na(fcorset)){
	print(paste('applying specified fcor of',fcorset))
	data$co2f<-data$co2t-fcorset*(curmprv) 
	data$co2f[1]<-data$co2t[1]
	caldump$flsfcorr<-fcorset
} else {
	print('applying fcor from fit')
	data$co2f<-data$co2t-fcorset*(curmprv) 
	data$co2f[1]<-data$co2t[1]
	caldump$flsfcorr<-fcorset
}
}else{
data$co2f<-data$co2t
print('fcor not applied')
caldump$flsfcorr<-0
}


# CALCULATE TIME DEPENDENT 4-POINT CAL AND APPLY 

print('Time-dependent 4-Point Calibration')

## could replace doy with yrfrac here
data$cal4<-rep(F,length(data$doy))
data$up<-rep(F,length(data$doy))
cals<-list(doy=NULL,hs2=NULL,hs1=NULL,ls1=NULL,ls2=NULL,lin0=NULL,lin1=NULL,cur0=NULL,cur1=NULL,cur2=NULL,hs2f=NULL,hs1f=NULL,ls1f=NULL,ls2f=NULL,zoff=NULL,up=NULL)
calvals3<-NULL
## old code that only allowed 4-pt cals:
#for (i in c(1:(length(data$doy)-3))){
#	if(data$ngas[i]==5&data$ngas[i+1]==4&data$ngas[i+2]==3&data$ngas[i+3]==2){
#		data$cal4[i:(i+3)]<-T
#		data$up[i:(i+3)]<-F
#		cals$doy<-c(cals$doy,mean(data$doy[i:(i+3)]))
#		cals$hs2<-c(cals$hs2,data$co2f[i])
#		cals$hs1<-c(cals$hs1,data$co2f[i+1])
#		cals$ls1<-c(cals$ls1,data$co2f[i+2])
#		cals$ls2<-c(cals$ls2,data$co2f[i+3])
#		cals$up<-c(cals$up,F)
#		calvals3=rbind(calvals3,c(data$hs2[i],data$hs1[i+1],data$ls1[i+2],data$ls2[i+3]))
#	}
#	if(data$ngas[i]==2&data$ngas[i+1]==3&data$ngas[i+2]==4&data$ngas[i+3]==5){
#		data$cal4[i:(i+3)]<-T
#		data$up[i:(i+3)]<-T
#		cals$doy<-c(cals$doy,mean(data$doy[i:(i+3)]))
#		cals$hs2<-c(cals$hs2,data$co2f[i+3])
#		cals$hs1<-c(cals$hs1,data$co2f[i+2])
#		cals$ls1<-c(cals$ls1,data$co2f[i+1])
#		cals$ls2<-c(cals$ls2,data$co2f[i])
#		cals$up<-c(cals$up,T)
#		calvals3=rbind(calvals3,c(data$hs2[i+3],data$hs1[i+2],data$ls1[i+1],data$ls2[i]))
#	}
#}
# now grabbing any collection of cals of any number and in any order, as long as 3 different gases
lastrec=0
for(i in c(1:(length(data$doy)-2))){
        if(all(data$ngas[i:(i+2)]>1)&i>lastrec){ # start of next cal sequence
                calstart=i
                calend=length(data$doy)
                for(j in c(i:length(data$doy))){
                        if(data$doy[j]-data$doy[max(c(j-1,1))]>10/60/24|data$ngas[j]<2){ # gap of 10 min or air sample
                                calend=j-1
                                break # exit for(j loop
                        }
                }
                lastrec=calend
                if(length(unique(data$ngas[calstart:calend]))>2){ # 3 or 4 cal gases
                        data$cal4[calstart:calend]=T
                        cals$doy=c(cals$doy,mean(data$doy[calstart:calend]))
                        # clump repeated cals together:
                        cals$hs2=c(cals$hs2,mean(data$co2f[calstart:calend][data$ngas[calstart:calend]==5]))
                        cals$hs1=c(cals$hs1,mean(data$co2f[calstart:calend][data$ngas[calstart:calend]==4]))
                        cals$ls1=c(cals$ls1,mean(data$co2f[calstart:calend][data$ngas[calstart:calend]==3]))
                        cals$ls2=c(cals$ls2,mean(data$co2f[calstart:calend][data$ngas[calstart:calend]==2]))
                        if(var(data$doy[calstart:calend],data$co2f[calstart:calend],na.rm=T)>0){ # call it an UP
                                data$up[calstart:calend]=T
                                cals$up=c(cals$up,T)
                        } else { # call it a DOWN
                                data$up[calstart:calend][data$ngas[calstart:calend]<6]=F
                                cals$up=c(cals$up,F)
                        }
                        calvals3=rbind(calvals3,c(data$hs2[calstart],data$hs1[calstart],data$ls1[calstart],data$ls2[calstart])) # just use interpolated values at start of sequence
                }
        }
}

cals$lin0<-rep(NA,length(cals$doy))
cals$lin1<-rep(NA,length(cals$doy))
cals$cur0<-rep(NA,length(cals$doy))
cals$cur1<-rep(NA,length(cals$doy))
cals$cur2<-rep(NA,length(cals$doy))

if(length(cals$doy)>0){

# Generate DateTime values for plotting
caldatetime=strptime(paste(year,'-01-01 00:00:00',sep=''),format='%y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*(cals$doy-1)

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_fits_%d.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_fits_%d.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(5,8))
par(mar=c(2,2,1,1)+.1)
par(mgp=c(2,1,0))
par(oma=c(2.5,2.5,2,0))

# first calculate monthly-mean curvature parameters to allow 3-pt linear cals
meas<-c(mean(data$co2f[data$ngas==5],na.rm=T),mean(data$co2f[data$ngas==4],na.rm=T),mean(data$co2f[data$ngas==3],na.rm=T),mean(data$co2f[data$ngas==2],na.rm=T))
calvals4=c(mean(data$hs2[data$ngas==5],na.rm=T),mean(data$hs1[data$ngas==4],na.rm=T),mean(data$ls1[data$ngas==3],na.rm=T),mean(data$ls2[data$ngas==2],na.rm=T))
### may give weird results when cal gases are changed or curvature changes mid-month
mod<-cbind(rep(1,length(meas[!is.na(meas)])),meas[!is.na(meas)])
monfitlin<-lm.fit(mod,calvals4[!is.na(meas)])
monlin0=monfitlin$coef[1]
monlin1=monfitlin$coef[2]
mod<-cbind(rep(1,length(meas[!is.na(meas)])),meas[!is.na(meas)],meas[!is.na(meas)]^2)
monfitcur<-lm.fit(mod,monfitlin$resid)
if(sum(data$ngas==5)<2|sum(data$ngas==4)<2|sum(data$ngas==3)<2|sum(data$ngas==2)<2){ # less than 4 cal gases running
print('LESS THAN 4 CAL GASES AVAILABLE')
##### using linear fit to 3 gases - other option would be to fit a quadratic to 3. . . .
monfitcur$coef[1]=0
monfitcur$coef[2]=0
monfitcur$coef[3]=0
monfitcur$resids=monfitlin$resids
}
moncur0=monfitcur$coef[1]
moncur1=monfitcur$coef[2]
moncur2=monfitcur$coef[3]
#print('MONFITCUR')
#print(moncur0)
#print(moncur1)
#print(moncur2)

count4up=0
count4dn=0
for (i in c(1:length(cals$doy))){
#print(i)
#browser()
	meas<-c(cals$hs2[i],cals$hs1[i],cals$ls1[i],cals$ls2[i])
        if(sum(!is.na(meas))>2){
        mod<-cbind(rep(1,sum(!is.na(meas))),meas[!is.na(meas)])
        fitlin<-lm.fit(mod,calvals3[i,!is.na(meas)])
        cals$lin0[i]<-fitlin$coef[1]
        cals$lin1[i]<-fitlin$coef[2]
        if(sum(!is.na(meas))>3){
                mod<-cbind(rep(1,length(meas)),meas,meas^2)
                fitcur<-lm.fit(mod,fitlin$resid)
        } else {
                fitcur=monfitcur
                ### does curvature trend within a month?  if so, some form of shorter-term interpolation might work better
        }
#	mod<-cbind(rep(1,length(meas)),meas)
#	fitlin<-lm.fit(mod,calvals3[i,])
#	cals$lin0[i]<-fitlin$coef[1]
#	cals$lin1[i]<-fitlin$coef[2]
#	mod<-cbind(rep(1,length(meas)),meas,meas^2)
#	fitcur<-lm.fit(mod,fitlin$resid)
	cals$cur0[i]<-fitcur$coef[1]
	cals$cur1[i]<-fitcur$coef[2]
	cals$cur2[i]<-fitcur$coef[3]
	cals$zoff[i]<-mean(fitcur$resid)
#print(meas)
#print(fitlin$resid)
	plot(meas[!is.na(meas)],fitlin$resid,type='n',main=as.character(strftime(as.POSIXlt(caldatetime[i]),format='%d %H:%M',tz="GMT")),ylim=c(-.5,.5))
        points(meas[!is.na(meas)],fitlin$resid,col=c('Blue','Green','Red','Purple')[!is.na(meas)])
	#plot(meas,fitlin$resid,type='n',main=as.character(strftime(caldatetime[i],format='%d %H:%M',tz="GMT")),ylim=c(-.5,.5))
	#points(meas,fitlin$resid,col=c('Blue','Green','Red','Purple'))
#	lines(meas,fitcur$fit)
        xval=seq(min(meas[!is.na(meas)]),max(meas[!is.na(meas)]),1)
        yval=fitcur$coef[1]+fitcur$coef[2]*xval+fitcur$coef[3]*xval^2
        lines(xval,yval)
        if(sum(!is.na(meas))>3){
                if(cals$up[i]){
                        mtext('UP',3,-1,cex=.7)
                        count4up=count4up+1
                }else{
                        mtext('DOWN',3,-1,cex=.7)
                        count4dn=count4dn+1
                }
        } else {
                mtext('3-PT/AVECUR',3,-1,cex=.7)
        }
#	if(cals$up[i]){mtext('UP',3,-1,cex=.7)}else{mtext('DOWN',3,-1,cex=.7)}
	mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' 4-Hourly 2-nd Order Fits'),3,0.5,outer=T)
	mtext('Measured CO2 (ppm)',1,1,T)
	mtext('Deviation from Linear (ppm)',2,1,T)
	}
}

dev.off()

cals$hs2f<-cals$lin0+cals$lin1*cals$hs2+cals$cur0+cals$cur1*cals$hs2+cals$cur2*cals$hs2^2
cals$hs1f<-cals$lin0+cals$lin1*cals$hs1+cals$cur0+cals$cur1*cals$hs1+cals$cur2*cals$hs1^2
cals$ls1f<-cals$lin0+cals$lin1*cals$ls1+cals$cur0+cals$cur1*cals$ls1+cals$cur2*cals$ls1^2
cals$ls2f<-cals$lin0+cals$lin1*cals$ls2+cals$cur0+cals$cur1*cals$ls2+cals$cur2*cals$ls2^2

if(length(cals$doy)>1){
data$lin0<-approx(cals$doy,cals$lin0,data$doy,rule=2)$y
data$lin1<-approx(cals$doy,cals$lin1,data$doy,rule=2)$y
data$cur0<-approx(cals$doy,cals$cur0,data$doy,rule=2)$y
data$cur1<-approx(cals$doy,cals$cur1,data$doy,rule=2)$y
data$cur2<-approx(cals$doy,cals$cur2,data$doy,rule=2)$y
} else {
data$lin0<-rep(cals$lin0,length(data$doy))
data$lin1<-rep(cals$lin1,length(data$doy))
data$cur0<-rep(cals$cur0,length(data$doy))
data$cur1<-rep(cals$cur1,length(data$doy))
data$cur2<-rep(cals$cur2,length(data$doy))
}

data$co2c<-data$lin0+data$lin1*data$co2f+data$cur0+data$cur1*data$co2f+data$cur2*data$co2f^2

} else {
print('no cal data')
data$lin0<-rep(0,length(data$doy))
data$lin1<-rep(1,length(data$doy))
data$cur0<-rep(0,length(data$doy))
data$cur1<-rep(0,length(data$doy))
data$cur2<-rep(0,length(data$doy))
data$co2c<-data$co2f
}


# PLOT FIT PARAMETERS

print('Plotting Fit Parameters')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_parameters.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_parameters.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(5,1))
par(mar=c(0,4,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(4,0,2,0))

plot(datetime,data$lin0,main='Linear Offset',type='n',xlab='',ylab='ppm',ylim=mean(data$lin0)+c(-7.5,7.5),xaxt='n')
lines(datetime,data$lin0,col='Brown')
axis(1,labels=F)
plot(datetime,data$lin1,main='Linear Slope',type='n',xlab='',ylab='ppm/ppm',ylim=mean(data$lin1)+c(-.025,.025),xaxt='n')
lines(datetime,data$lin1,col='Orange')
axis(1,labels=F)
plot(datetime,data$cur0,main='2-nd Order Offset',type='n',xlab='',ylab='ppm',ylim=mean(data$cur0)+c(-15,15),xaxt='n')
lines(datetime,data$cur0,col='Light Blue')
axis(1,labels=F)
plot(datetime,data$cur1,main='2-nd Order Slope',type='n',xlab='',ylab='ppm/ppm',ylim=c(-.1,.1),xaxt='n')
lines(datetime,data$cur1,col='Light Green')
axis(1,labels=F)
plot(datetime,data$cur2,main='2-nd Order Curvature',type='n',xlab='',ylab='ppm/ppm^2',ylim=mean(data$cur2)+c(-.0001,.0001))
lines(datetime,data$cur2,col='Pink')

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' 4-Hourly Fit Parameters'),3,0.5,outer=T)
mtext('Date (UTC)',1,2.5,T)

dev.off()

# PLOT AVERAGE 4-POINT CALS

print('Plotting Average Fits')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_averages.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_averages.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(2,3))
par(oma=c(0,0,2,0))

if(any(data$cal4)){

## already did above (for all gases, not just ones in cal4) - necessary here?
meas<-c(mean(data$co2f[data$ngas==5&data$cal4],na.rm=T),mean(data$co2f[data$ngas==4&data$cal4],na.rm=T),mean(data$co2f[data$ngas==3&data$cal4],na.rm=T),mean(data$co2f[data$ngas==2&data$cal4],na.rm=T))
calvals4=c(mean(data$hs2[data$ngas==5&data$cal4],na.rm=T),mean(data$hs1[data$ngas==4&data$cal4],na.rm=T),mean(data$ls1[data$ngas==3&data$cal4],na.rm=T),mean(data$ls2[data$ngas==2&data$cal4],na.rm=T))
## may give weird results when cal gases are changed mid-month

mod<-cbind(rep(1,length(meas[!is.na(meas)])),meas[!is.na(meas)])
fitlin<-lm.fit(mod,calvals4[!is.na(meas)])
plot(meas,calvals4,type='n',main='Linear Fit to Average Values',xlab='Measured CO2 (ppm)',ylab='True CO2 (ppm)')
points(meas,calvals4,col=c('Blue','Green','Red','Purple'))
lines(meas[!is.na(meas)],fitlin$fit)
mod<-cbind(rep(1,length(meas[!is.na(meas)])),meas[!is.na(meas)],meas[!is.na(meas)]^2)
fitcur<-lm.fit(mod,fitlin$resid)
if(sum(data$ngas==5)<2|sum(data$ngas==4)<2|sum(data$ngas==3)<2|sum(data$ngas==2)<2){ # less than 4 cal gases running
print('LESS THAN 4 CAL GASES AVAILABLE')
##### using linear fit to 3 gases - other option would be to fit a quadratic to 3. . . .
fitcur$coef[1]=0
fitcur$coef[2]=0
fitcur$coef[3]=0
fitcur$fit=fitlin$fit
}
avefitcur<-fitcur
plot(meas[!is.na(meas)],fitcur$resid,type='n',main='2nd-order Fit to Residuals',xlab='Measured CO2 (ppm)',ylab='Delta CO2 (ppm)',ylim=c(-.5,.5))
points(meas[!is.na(meas)],fitlin$resid,col=c('Blue','Green','Red','Purple'))
xval=seq(min(meas[!is.na(meas)]),max(meas[!is.na(meas)]),1)
yval=fitcur$coef[1]+fitcur$coef[2]*xval+fitcur$coef[3]*xval^2
lines(xval,yval)
#lines(meas[!is.na(meas)],fitcur$fit)

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Average Fits'),3,0.5,outer=T)

data$co2n<-fitlin$coef[1]+fitlin$coef[2]*data$co2f+fitcur$coef[1]+fitcur$coef[2]*data$co2f+fitcur$coef[3]*data$co2f^2

zoff<-data$co2n[data$ngas>1]-zref
plot(zdt,zoff,type='n',main='Zero Offsets After Average Cal.',xlab='Date (UTC)',ylab='Delta CO2 (ppm)')
points(zdt[zgas==5],zoff[zgas==5],col='Blue')
points(zdt[zgas==4],zoff[zgas==4],col='Green')
points(zdt[zgas==3],zoff[zgas==3],col='Red')
points(zdt[zgas==2],zoff[zgas==2],col='Purple')
zsmth<-lowess(zdoy[!is.na(zoff)],zoff[!is.na(zoff)],f=10/length(zoff))
# f set to equal 10 points or 4 hours ?? (does not seem to scale)
lines(datetime,approx(zsmth$x,zsmth$y,data$doy,rule=2)$y)

if(length(cals$doy)>0){

trend<-approx(zsmth$x,zsmth$y,cals$doy,rule=2)$y

calsup<-c(mean(cals$hs2[cals$up]-trend[cals$up]),mean(cals$hs1[cals$up]-trend[cals$up]),mean(cals$ls1[cals$up]-trend[cals$up]),mean(cals$ls2[cals$up]-trend[cals$up]))
calsdn<-c(mean(cals$hs2[!cals$up]-trend[!cals$up]),mean(cals$hs1[!cals$up]-trend[!cals$up]),mean(cals$ls1[!cals$up]-trend[!cals$up]),mean(cals$ls2[!cals$up]-trend[!cals$up]))
plot(meas,calsup-calsdn,main='UP - DOWN Average Values',col=c('Blue','Green','Red','Purple'),ylim=c(-.5,.5),ylab='delta CO2',xlab='Measured CO2 (ppm)')

reshs2<-calvals3[,1]-(cals$lin0+cals$lin1*cals$hs2)
reshs1<-calvals3[,2]-(cals$lin0+cals$lin1*cals$hs1)
resls1<-calvals3[,3]-(cals$lin0+cals$lin1*cals$ls1)
resls2<-calvals3[,4]-(cals$lin0+cals$lin1*cals$ls2)
curhs2<-cals$cur0+cals$cur1*cals$hs2+cals$cur2*cals$hs2^2
curhs1<-cals$cur0+cals$cur1*cals$hs1+cals$cur2*cals$hs1^2
curls1<-cals$cur0+cals$cur1*cals$ls1+cals$cur2*cals$ls1^2
curls2<-cals$cur0+cals$cur1*cals$ls2+cals$cur2*cals$ls2^2
resup<-c(mean(reshs2[cals$up]),mean(reshs1[cals$up]),mean(resls1[cals$up]),mean(resls2[cals$up]))
resdn<-c(mean(reshs2[!cals$up]),mean(reshs1[!cals$up]),mean(resls1[!cals$up]),mean(resls2[!cals$up]))
curup<-c(mean(curhs2[cals$up]),mean(curhs1[cals$up]),mean(curls1[cals$up]),mean(curls2[cals$up]))
curdn<-c(mean(curhs2[!cals$up]),mean(curhs1[!cals$up]),mean(curls1[!cals$up]),mean(curls2[!cals$up]))
#avenup = mean(c(length(curhs2[cals$up]),length(curhs1[cals$up]),length(curls1[cals$up]),length(curls2[cals$up])))
#mtext(paste('avenup = ',avenup),3,-1.5)
#if(length(cals$hs2[cals$up])>0&length(cals$hs1[cals$up])>0&length(cals$ls1[cals$up])>0&length(cals$ls2[cals$up])>0){
#aveseup = mean(c(sqrt(var(cals$hs2[cals$up]-trend[cals$up])),sqrt(var(cals$hs1[cals$up]-trend[cals$up])),sqrt(var(cals$ls1[cals$up]-trend[cals$up])),sqrt(var(cals$ls2[cals$up]-trend[cals$up]))))/sqrt(avenup)
#mtext(paste('avestderrup = ',round(aveseup,2)),3,-2.5)
#}
mtext(paste('Nup = ',sum(cals$up),' (',count4up,' four-pt)',sep=''),3,-1.5)
if(sum(!is.na(cals$hs2[cals$up]))>0&sum(!is.na(cals$hs1[cals$up]))>0&sum(!is.na(cals$ls1[cals$up]))>0&sum(!is.na(cals$ls2[cals$up]))>0){
aveseup = mean(c(sqrt(var(cals$hs2[cals$up]-trend[cals$up],na.rm=T)),sqrt(var(cals$hs1[cals$up]-trend[cals$up],na.rm=T)),sqrt(var(cals$ls1[cals$up]-trend[cals$up],na.rm=T)),sqrt(var(cals$ls2[cals$up]-trend[cals$up],na.rm=T))))/sqrt(sum(cals$up))
mtext(paste('avestderrup = ',round(aveseup,2)),3,-2.5)
}
mtext(paste('Ndn = ',sum(!cals$up),' (',count4dn,' four-pt)',sep=''),1,-2.5)
if(sum(!is.na(cals$hs2[!cals$up]))>0&sum(!is.na(cals$hs1[!cals$up]))>0&sum(!is.na(cals$ls1[!cals$up]))>0&sum(!is.na(cals$ls2[!cals$up]))>0){
avesedn = mean(c(sqrt(var(cals$hs2[!cals$up]-trend[!cals$up],na.rm=T)),sqrt(var(cals$hs1[!cals$up]-trend[!cals$up],na.rm=T)),sqrt(var(cals$ls1[!cals$up]-trend[!cals$up],na.rm=T)),sqrt(var(cals$ls2[!cals$up]-trend[!cals$up],na.rm=T))))/sqrt(sum(!cals$up))
mtext(paste('avestderrdn = ',round(avesedn,2)),1,-1.5)
}

print(paste('measured:',meas))
print(paste('assigned:',calvals4))
print(resup)

plot(meas,resup,type='n',main='2nd-order Fit to Residuals [UP]',xlab='Measured CO2 (ppm)',ylab='Delta CO2 (ppm)',ylim=c(-.5,.5))
points(meas,resup,col=c('Blue','Green','Red','Purple'))
lines(xval,yval)
yvalup=mean(cals$cur0[cals$up],na.rm=T)+mean(cals$cur1[cals$up],na.rm=T)*xval+mean(cals$cur2[cals$up],na.rm=T)*xval^2
lines(xval,yvalup,col='Cyan')
### can get rid of curup, etc (? from LDR code)
#lines(meas,curup,col='Cyan')
#lines(meas,avefitcur$fit)

print(resdn)
plot(meas,resdn,type='n',main='2nd-order Fit to Residuals [DOWN]',xlab='Measured CO2 (ppm)',ylab='Delta CO2 (ppm)',ylim=c(-.5,.5))
points(meas,resdn,col=c('Blue','Green','Red','Purple'))
lines(xval,yval)
yvaldn=mean(cals$cur0[!cals$up],na.rm=T)+mean(cals$cur1[!cals$up],na.rm=T)*xval+mean(cals$cur2[!cals$up],na.rm=T)*xval^2
lines(xval,yvaldn,col='Orange')

#lines(meas,curdn,col='Orange')
#lines(meas,avefitcur$fit)

}

} else {
data$co2n<-data$co2f
}

dev.off()


# CALCULATE HIGH-RATE (30 min. weighted by neighbors) ZERO OFFSET AND APPLY

print('High-Rate Zero Offsets')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_offsets.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_offsets.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(3,1))
par(mar=c(0,3,3,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(4,0,2,0))

if(length(zgas)>0){

zoff<-data$co2c[data$ngas>1]-zref
z4pt<-data$cal4[data$ngas>1]
zoff4<-zoff[z4pt]
zoff1<-zoff[!z4pt]
plot(zdt,zoff,type='n',main='Zero Offsets After 4-Hr. Cal.',xlab='Date (UTC)',ylab='Delta CO2 (ppm)',ylim=c(-.3,.3),xaxt='n')
if(length(cals$doy)>0){
points(caldatetime,cals$zoff,pch=4)
}
points(zdt[zgas==5],zoff[zgas==5],col='Blue',type='b')
points(zdt[zgas==4],zoff[zgas==4],col='Green',type='b')
points(zdt[zgas==3],zoff[zgas==3],col='Red',type='b')
points(zdt[zgas==2],zoff[zgas==2],col='Purple',type='b')
axis(1,labels=F)
if(any(!is.na(zoff4))){
mtext(paste('Std. Dev. of 4-Pts. = ',round(sqrt(var(zoff4[!is.na(zoff4)])),2)),3,-1.2,cex=.8)
}
if(any(!is.na(zoff1))){
mtext(paste('Std. Dev. of 1-Pts. = ',round(sqrt(var(zoff1[!is.na(zoff1)])),2)),1,-1.2,cex=.8)
}

# normalize by gas
for(i in c(5,4,3,2)){
temp1<-zdoy[z4pt&zgas==i]
temp2<-zoff[z4pt&zgas==i]
if(length(temp1)>1){
zoff[zgas==i]<-zoff[zgas==i]-approx(temp1,temp2,zdoy,rule=2)$y[zgas==i]
}
}
plot(zdt,zoff,type='n',main='Normalized by Gas',xlab='Date (UTC)',ylab='Delta CO2 (ppm)',ylim=c(-.3,.3),xaxt='n')
points(zdt[zgas==5],zoff[zgas==5],col='Blue',type='p')
points(zdt[zgas==4],zoff[zgas==4],col='Green',type='p')
points(zdt[zgas==3],zoff[zgas==3],col='Red',type='p')
points(zdt[zgas==2],zoff[zgas==2],col='Purple',type='p')
points(zdt[z4pt],zoff[z4pt])
axis(1,labels=F)

prevoff<-c(NA,zoff[1:(length(zoff)-1)])
postoff<-c(zoff[2:length(zoff)],NA)
zsmth<-rep(NA,length(zoff))
# do 1-2-1 weighted running mean
for(i in c(1:length(zoff))){
	zsmth[i]<-mean(c(prevoff[i],zoff[i],zoff[i],postoff[i]),na.rm=T)
}
lines(zdt,zsmth,col='Dark Gray')

data$co2final<-data$co2c-approx(zdoy,zsmth,data$doy,rule=2)$y

zoff<-data$co2final[data$ngas>1]-zref
zoff4<-zoff[z4pt]
zoff1<-zoff[!z4pt]
plot(zdt,zoff,type='n',main='Final Zero Offsets',xlab='Date (UTC)',ylab='Delta CO2 (ppm)',ylim=c(-.3,.3))
points(zdt[zgas==5],zoff[zgas==5],col='Blue',type='b')
points(zdt[zgas==4],zoff[zgas==4],col='Green',type='b')
points(zdt[zgas==3],zoff[zgas==3],col='Red',type='b')
points(zdt[zgas==2],zoff[zgas==2],col='Purple',type='b')
if(any(!is.na(zoff4))){
mtext(paste('Std. Dev. of 4-Pts. = ',round(sqrt(var(zoff4[!is.na(zoff4)])),2)),3,-1.2,cex=.8)
}
if(any(!is.na(zoff1))){
mtext(paste('Std. Dev. of 1-Pts. = ',round(sqrt(var(zoff1[!is.na(zoff1)])),2)),1,-1.2,cex=.8)
}

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' 30-min. Zero Offsets'),3,0.5,outer=T)

} else {
data$co2final<-data$co2c
}

mtext('Date (UTC)',1,2.5,T)

dev.off()

# PLOT CAL VALUES

print('Plotting Calibration Values')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_calvals.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_calvals.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}
par(mfrow=c(4,4))
par(mar=c(2,2,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(2.5,2.5,2,0))

if(length(data$doy[data$ngas>1])>0){

if(any(data$ngas==5)){
plot(datetime[data$ngas==5],data$co2p[data$ngas==5],main='HS2 Pressure Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==5],data$co2p[data$ngas==5],col='Blue')
} else {
plot(c(1:3),c(1:3),main='HS2 Pressure Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==4)){
plot(datetime[data$ngas==4],data$co2p[data$ngas==4],main='HS1 Pressure Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==4],data$co2p[data$ngas==4],col='Green')
} else {
plot(c(1:3),c(1:3),main='HS1 Pressure Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==3)){
plot(datetime[data$ngas==3],data$co2p[data$ngas==3],main='LS1 Pressure Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==3],data$co2p[data$ngas==3],col='Red')
} else {
plot(c(1:3),c(1:3),main='LS1 Pressure Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==2)){
plot(datetime[data$ngas==2],data$co2p[data$ngas==2],main='LS2 Pressure Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==2],data$co2p[data$ngas==2],col='Purple')
} else {
plot(c(1:3),c(1:3),main='LS2 Pressure Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}

if(any(data$ngas==5)){
plot(datetime[data$ngas==5],data$co2f[data$ngas==5],main='HS2 Temperature Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==5],data$co2f[data$ngas==5],col='Blue')
} else {
plot(c(1:3),c(1:3),main='HS2 Temperature Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==4)){
plot(datetime[data$ngas==4],data$co2f[data$ngas==4],main='HS1 Temperature Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==4],data$co2f[data$ngas==4],col='Green')
} else {
plot(c(1:3),c(1:3),main='HS1 Temperature Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==3)){
plot(datetime[data$ngas==3],data$co2f[data$ngas==3],main='LS1 Temperature Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==3],data$co2f[data$ngas==3],col='Red')
} else {
plot(c(1:3),c(1:3),main='LS1 Temperature Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==2)){
plot(datetime[data$ngas==2],data$co2f[data$ngas==2],main='LS2 Temperature Smoothed',xlab='',ylab='',type='n')
points(datetime[data$ngas==2],data$co2f[data$ngas==2],col='Purple')
} else {
plot(c(1:3),c(1:3),main='LS2 Temperature Smoothed',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}

if(any(data$ngas==5)){
plot(datetime[data$ngas==5],data$co2c[data$ngas==5],ylim=c(-1,1)+median(data$co2c[data$ngas==5]),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n')
points(datetime[data$ngas==5],data$co2c[data$ngas==5],col='Blue')
} else {
plot(c(1:3),c(1:3),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==4)){
plot(datetime[data$ngas==4],data$co2c[data$ngas==4],ylim=c(-1,1)+median(data$co2c[data$ngas==4]),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n')
points(datetime[data$ngas==4],data$co2c[data$ngas==4],col='Green')
} else {
plot(c(1:3),c(1:3),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==3)){
plot(datetime[data$ngas==3],data$co2c[data$ngas==3],ylim=c(-1,1)+median(data$co2c[data$ngas==3]),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n')
points(datetime[data$ngas==3],data$co2c[data$ngas==3],col='Red')
} else {
plot(c(1:3),c(1:3),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==2)){
plot(datetime[data$ngas==2],data$co2c[data$ngas==2],ylim=c(-1,1)+median(data$co2c[data$ngas==2]),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n')
points(datetime[data$ngas==2],data$co2c[data$ngas==2],col='Purple')
} else {
plot(c(1:3),c(1:3),main='4-Hr. Interpolated 4-Pt. Cal.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}

if(any(data$ngas==5)){
plot(datetime[data$ngas==5],data$co2final[data$ngas==5],ylim=c(-1,1)+median(data$co2final[data$ngas==5]),main='30-min. Zero Off.',xlab='',ylab='',type='n')
points(datetime[data$ngas==5],data$co2final[data$ngas==5],col='Blue')
} else {
plot(c(1:3),c(1:3),main='30-min. Zero Off.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==4)){
plot(datetime[data$ngas==4],data$co2final[data$ngas==4],ylim=c(-1,1)+median(data$co2final[data$ngas==4]),main='30-min. Zero Off.',xlab='',ylab='',type='n')
points(datetime[data$ngas==4],data$co2final[data$ngas==4],col='Green')
} else {
plot(c(1:3),c(1:3),main='30-min. Zero Off.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==3)){
plot(datetime[data$ngas==3],data$co2final[data$ngas==3],ylim=c(-1,1)+median(data$co2final[data$ngas==3]),main='30-min. Zero Off.',xlab='',ylab='',type='n')
points(datetime[data$ngas==3],data$co2final[data$ngas==3],col='Red')
} else {
plot(c(1:3),c(1:3),main='30-min. Zero Off.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}
if(any(data$ngas==2)){
plot(datetime[data$ngas==2],data$co2final[data$ngas==2],ylim=c(-1,1)+median(data$co2final[data$ngas==2]),main='30-min. Zero Off.',xlab='',ylab='',type='n')
points(datetime[data$ngas==2],data$co2final[data$ngas==2],col='Purple')
} else {
plot(c(1:3),c(1:3),main='30-min. Zero Off.',xlab='',ylab='',type='n',xaxt='n',yaxt='n')
}

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Calibration Values'),3,0.5,outer=T)
mtext('Date (UTC)',1,1,T)
mtext('Measured CO2 (ppm)',2,1,T)


}

dev.off()


# PLOT FIT PARAMETERS IN PPM

print('Plotting Fit Parameters in ppm')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_paraminppm.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_paraminppm.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(6,1))
par(mar=c(0,4,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(4,0,2,0))

plot(datetime,data$lin0,main='4-Hourly Linear Offset',type='n',xlab='',ylab='ppm',ylim=mean(data$lin0)+c(-5,5),xaxt='n')
abline(h=mean(data$lin0)+0.4,col='Gray')
abline(h=mean(data$lin0)-0.4,col='Gray')
lines(datetime,data$lin0,col='Red')
axis(1,labels=F)
plot(datetime,data$lin1*50,main='4-Hourly Linear Slope * 50 ppm',type='n',xlab='',ylab='ppm',ylim=mean(data$lin1*50)+c(-.5,.5),xaxt='n')
abline(h=mean(data$lin1*50)+.05,col='Gray')
abline(h=mean(data$lin1*50)-.05,col='Gray')
lines(datetime,data$lin1*50,col='Green')
axis(1,labels=F)
midlin<-data$lin0+data$lin1*380
plot(datetime,midlin,main='4-Hourly Linear Value at 380 ppm',type='n',xlab='',ylab='ppm',ylim=mean(midlin)+c(-5,5),xaxt='n')
abline(h=mean(midlin)+.05,col='Gray')
abline(h=mean(midlin)-.0511,col='Gray')
lines(datetime,midlin,col='Brown')
axis(1,labels=F)
midcur<-data$cur0+data$cur1*380+data$cur2*380**2
plot(datetime,midcur,main='4-Hourly Curvature (deviation from linear at 380 ppm)',type='n',xlab='',ylab='ppm',ylim=mean(midcur)+c(-.5,.5),xaxt='n')
abline(h=mean(midcur)+.05,col='Gray')
abline(h=mean(midcur)-.05,col='Gray')
lines(datetime,midcur,col='Blue')
axis(1,labels=F)
if(length(zdoy)>0){
plot(zdt,zsmth,main='30-Minute Zero Offset',type='n',xlab='',ylab='ppm',ylim=mean(zsmth,na.rm=T)+c(-.5,.5),xaxt='n')
abline(h=mean(zsmth,na.rm=T)+0.06,col='Gray')
abline(h=mean(zsmth,na.rm=T)-0.06,col='Gray')
lines(zdt,zsmth,col='Purple')
axis(1,labels=F)
}
plot(datetime,data$btmp*100-40,main='Box Temperature',type='n',xlab='',ylab='Degrees C')
lines(datetime,data$btmp*100-40,col='Brown')

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Fit Parameters in PPM'),3,0.5,outer=T)
mtext('Date (UTC)',1,2.5,T)

dev.off()


# READ IN INLET DATA

print('Reading Inlet Data')

inlets<-scan('../../aircoa_inlets.txt',what=list(loc='',unit='',date=0,time='',L1=0,L2=0,L3=0,L4=0,L5=0,hz1=0,hz2=0,hz3=0,hz4=0,hz5=0,upl=0),skip=1)
# line heights (1-5), horizontal distance to tower, and # of units per line
inlets$dayf<-as.numeric(substr(inlets$time,1,2))/24+as.numeric(substr(inlets$time,3,4))/60/24

data$height<-rep(NA,length(data$doy))
data$inlen<-rep(NA,length(data$doy))
data$tlag<-rep(NA,length(data$doy))
data$doyc<-rep(NA,length(data$doy))
data$trng<-rep(NA,length(data$doy))

dpm<-c(31,28,31,30,31,30,31,31,30,31,30,31)
# correct for leap year
if(year/4-trunc(year/4)==0) dpm[2]<-29 

# generate text year for filenames and figure labels
if(year<10) {yrt<-paste('0',as.character(year),sep='')} else { yrt<-as.character(year)} #  *** can not handle multiple year runs

# calculate total inlet lengths and spread across record
check<-NA
for(i in c(1:length(inlets$loc))){
	if(inlets$loc[i]==loc){
		check<-0
		iyear<-trunc(inlets$date[i]/10000)
		month<-trunc((inlets$date[i]-iyear*10000)/100)
		day<-inlets$date[i]-iyear*10000-month*100
		if(month>1){
        		doy<-sum(dpm[1:(month-1)])+day
		} else {
        		doy<-day
		}
		doy<-doy+inlets$dayf[i]
		if(year>=iyear){
			if(year>iyear){
				data$inlen[data$nlin==1&data$ngas==1]<-(inlets$L1[i]+inlets$hz1[i])/inlets$upl[i]
				data$inlen[data$nlin==2&data$ngas==1]<-(inlets$L2[i]+inlets$hz2[i])/inlets$upl[i]
				data$inlen[data$nlin==3&data$ngas==1]<-(inlets$L3[i]+inlets$hz3[i])/inlets$upl[i]
				data$height[data$nlin==1&data$ngas==1]<-inlets$L1[i]
				data$height[data$nlin==2&data$ngas==1]<-inlets$L2[i]
				data$height[data$nlin==3&data$ngas==1]<-inlets$L3[i]
				if(nlines==3){
					data$height[data$nlin==4&data$ngas==1]<-0
				}else if(nlines==5){
					data$inlen[data$nlin==4&data$ngas==1]<-(inlets$L4[i]+inlets$hz4[i])/inlets$upl[i]
					data$height[data$nlin==4&data$ngas==1]<-inlets$L4[i]
					data$inlen[data$nlin==5&data$ngas==1]<-(inlets$L5[i]+inlets$hz5[i])/inlets$upl[i]
					data$height[data$nlin==5&data$ngas==1]<-inlets$L5[i]
					data$height[data$nlin==6&data$ngas==1]<-0

				}else{
					print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
				}
			} else { # year = iyear
				data$inlen[data$doy>=doy&data$nlin==1&data$ngas==1]<-(inlets$L1[i]+inlets$hz1[i])/inlets$upl[i]
				data$inlen[data$doy>=doy&data$nlin==2&data$ngas==1]<-(inlets$L2[i]+inlets$hz2[i])/inlets$upl[i]
				data$inlen[data$doy>=doy&data$nlin==3&data$ngas==1]<-(inlets$L3[i]+inlets$hz3[i])/inlets$upl[i]
				data$height[data$doy>=doy&data$nlin==1&data$ngas==1]<-inlets$L1[i]
				data$height[data$doy>=doy&data$nlin==2&data$ngas==1]<-inlets$L2[i]
				data$height[data$doy>=doy&data$nlin==3&data$ngas==1]<-inlets$L3[i]
				if(nlines==3){
					data$height[data$doy>=doy&data$nlin==4&data$ngas==1]<-0
				}else if(nlines==5){
					data$inlen[data$doy>=doy&data$nlin==4&data$ngas==1]<-(inlets$L4[i]+inlets$hz4[i])/inlets$upl[i]
					data$height[data$doy>=doy&data$nlin==4&data$ngas==1]<-inlets$L4[i]
					data$inlen[data$doy>=doy&data$nlin==5&data$ngas==1]<-(inlets$L5[i]+inlets$hz5[i])/inlets$upl[i]
					data$height[data$doy>=doy&data$nlin==5&data$ngas==1]<-inlets$L5[i]
					data$height[data$doy>=doy&data$nlin==6&data$ngas==1]<-0
				}else{
					print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
				}
			}
		}
	}
}
if(is.na(check)){print('Location not found in aircoa_inlets.txt')}

# determine nlin for highest inlet (breaking ties by greater nlin)
if(any(!is.na(data$height))){
	maxlin<-max(data$nlin[data$height==max(data$height,na.rm=T)],na.rm=T)
	print(paste('maxlin',maxlin))
	# determine nlin for second highest inlet (breaking ties by greater nlin)
	if(any(!is.na(data$height[data$nlin!=maxlin]))){
	midlin<-max(data$nlin[data$nlin!=maxlin][data$height[data$nlin!=maxlin]==max(data$height[data$nlin!=maxlin],na.rm=T)],na.rm=T)
	print(paste('midlin',midlin))
	} else {
	print('Only 1 line')
	midlin=NA
	}
} else {
	maxlin<-max(data$nlin)
	print('No heights assigned')
	if(any(!is.na(data$height[data$nlin!=maxlin]))){
	midlin<-max(data$nlin[data$nlin!=maxlin])
	} else {
	print('Only 1 line')
	midlin=NA
	}
}
	
# CALCULATE LAG TIMES

##### redo to account for constant flow

nplen<-5 # m, length of nafion (2 x 96 in) plus internal
npvol<-3*nplen # cc, ~ 2 mm ID
livol<-.14*52 # cc, 14 cm of ~ 8 mm ID

print('Calculating Time-Dependent Time Lags')

datafl1<-data$fl1
datafl2<-data$fl2
datafl3<-data$fl3
datafl4<-data$fl4
datafl5<-data$fl5
for(i in c(1:length(data$doy))){
    if(data$ngas[i]==1&data$nlin[i]<=nlines&data$lchk[i]==0){
	invol<-13*(data$inlen[i]) # cc, ~ 4 mm ID
	cellv<-invol+npvol+livol/2 # cc
	slowf<-data$sfl[i]*101.3/data$prs[i] # accm, should div by inlet/naphion/cell P not just cell P
	if(i==1) { purge<-get(paste('datafl',data$nlin[i],sep=''))[i+1]*101.3/data$prs[i] } else { purge<-get(paste('datafl',data$nlin[i],sep=''))[i-1]*101.3/data$prs[i] } # should div by inlet/naphion/cell P not just cell P
	## hard coding EFS separate purge pump arrangement (assuming constant flows of 4.5 SLPM)
	if(loc=='EFS'&start<080912){
		if(data$nlin[i]==2) { 
			purge<-4500*101.3/data$prs[i] # should div by inlet/naphion/cell P not just cell P
		}
		if(data$nlin[i]==3) { 
			purge<-4500*101.3/data$prs[i] # should div by inlet/naphion/cell P not just cell P
		}
	}
	if(purge==0) purge<-NA

	#at start of meas. period
	v1<-slowf*skipsec/60 # volume pulled in during skip period
	if(v1>cellv){ # greater than total inlet volume
	t1<-cellv/slowf*60 # start time of integration
	} else {
	t1<-(cellv-v1)/purge*60+skipsec # start time of integration
	}
	#at end of meas. period
	v1<-slowf*njogsec/60 # volume pulled in during njog period
	if(v1>cellv){ # greater than total inlet volume
	t2<-cellv/slowf*60  # end time of integration
	} else {
	t2<-(cellv-v1)/purge*60+njogsec # end time of integration
	}

	data$tlag[i]<-mean(t1,t2)
	data$doyc[i]<-data$doy[i]-data$tlag[i]/86400
	data$trng[i]<-t1-t2+navesec
   } else {
	data$tlag[i]<-0
        data$doyc[i]<-data$doy[i]
        data$trng[i]<-navesec
   }
}

yrdoyc<-data$doyc # old day of year
data$doyc<-data$doyc+yrdays  ## yrdays here is days in whole years since 2004, elsewhere it is days in a year (365 or 366 depending on leap year)
if(year/4-trunc(year/4)==0){ # leap year
	yrfrac=2000+year+(yrdoyc-1)/366
}else{
	yrfrac=2000+year+(yrdoyc-1)/365
}

# WRITE DATA

print('Writing Data')

if(nlines==3){
	outmat<-rbind(round(yrfrac,7),round(data$co2i,3),round(data$prs,3),round(data$ltmp,3),round(data$btmp,3),
	round(data$rh,3),round(data$sfl,2),round(data$fl1,2),round(data$fl2,2),round(data$fl3,2),round(data$fl4,2),
	data$nlin,data$ngas,data$up,round(data$sdco2,3),round(data$sdprs,3),round(data$sdsfl,2),
	round(data$hs2,3),round(data$hs1,3),round(data$ls1,3),round(data$ls2,3),data$shpbn,
	round(data$pcorr,3),round(data$co2p,3),round(data$co2t,3),round(data$co2f,3),round(data$lin0,3),round(data$lin1,4),
	round(data$cur0,3),round(data$cur1,3),round(data$cur2,6),round(data$co2c,3),round(data$co2n,3),round(data$co2final,3))
	outmat2<-cbind(c("yrfrac","co2i","prs","ltmp","btmp","rh","sfl","fl1","fl2","fl3","fl4","nlin","ngas","up","sdco2","sdprs","sdsfl","hs2","hs1","ls1","ls2",
			"shpbn","pcorr","co2p","co2t","co2f","lin0","lin1","cur0","cur1","cur2","co2c","co2n","co2final"),outmat) # binding text somehow ensures 7 digits in yrfrac output
	write(outmat2,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".con",sep=''),ncol=34)
}else if(nlines==5){
	outmat<-rbind(round(yrfrac,7),round(data$co2i,3),round(data$prs,3),round(data$ltmp,3),round(data$btmp,3),
	round(data$rh,3),round(data$sfl,2),round(data$fl1,2),round(data$fl2,2),round(data$fl3,2),round(data$fl4,2),
	round(data$fl5,2),round(data$fl6,2),data$nlin,data$ngas,data$up,round(data$sdco2,3),round(data$sdprs,3),round(data$sdsfl,2),
	round(data$hs2,3),round(data$hs1,3),round(data$ls1,3),round(data$ls2,3),data$shpbn,
	round(data$pcorr,3),round(data$co2p,3),round(data$co2t,3),round(data$co2f,3),round(data$lin0,3),round(data$lin1,4),
	round(data$cur0,3),round(data$cur1,3),round(data$cur2,6),round(data$co2c,3),round(data$co2n,3),round(data$co2final,3))
	outmat2<-cbind(c("yrfrac","co2i","prs","ltmp","btmp","rh","sfl","fl1","fl2","fl3","fl4","fl5","fl6","nlin","ngas","up","sdco2","sdprs","sdsfl","hs2","hs1","ls1","ls2",
			"shpbn","pcorr","co2p","co2t","co2f","lin0","lin1","cur0","cur1","cur2","co2c","co2n","co2final"),outmat)
	write(outmat2,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".con",sep=''),ncol=36)
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?

}


# outputting LT results with line results
mesmat<-rbind(round(yrfrac[data$ngas==1],7),round(data$co2final[data$ngas==1],3),data$nlin[data$ngas==1],data$height[data$ngas==1],round(data$prs[data$ngas==1],3),
	round(data$btmp[data$ngas==1]*100-40,2),round(data$sdco2[data$ngas==1],3),round(data$trng[data$ngas==1],2))
rownames(mesmat)<-c('yrfrac','co2final','nlin','height','prs','btmp','sdco2','trng')


# INTERPOLATE DATA TO EVEN 15 MINUTES

print('Interpolating to 15 Minutes')

interp<-list(doyc=NULL,co2int1=NULL,co2int2=NULL,co2int3=NULL,co2int4=NULL,co2int5=NULL,prs=NULL,btmp=NULL)

interp$doyc<-floor(min(data$doyc,na.rm=T))+c(1:((ceiling(max(data$doyc,na.rm=T))-floor(min(data$doyc,na.rm=T)))*96))/96
if(any(data$nlin==1&data$ngas==1)){interp$co2int1<-approx(data$doyc[data$nlin==1&data$ngas==1],data$co2final[data$nlin==1&data$ngas==1],interp$doyc,rule=2)$y}else{interp$co2int1<-rep(NA,length(interp$doyc))}
if(any(data$nlin==2&data$ngas==1)){interp$co2int2<-approx(data$doyc[data$nlin==2&data$ngas==1],data$co2final[data$nlin==2&data$ngas==1],interp$doyc,rule=2)$y}else{interp$co2int2<-rep(NA,length(interp$doyc))}
if(any(data$nlin==3&data$ngas==1)){interp$co2int3<-approx(data$doyc[data$nlin==3&data$ngas==1],data$co2final[data$nlin==3&data$ngas==1],interp$doyc,rule=2)$y}else{interp$co2int3<-rep(NA,length(interp$doyc))}
if(nlines==5){
if(any(data$nlin==4&data$ngas==1)){interp$co2int4<-approx(data$doyc[data$nlin==4&data$ngas==1],data$co2final[data$nlin==4&data$ngas==1],interp$doyc,rule=2)$y}else{interp$co2int4<-rep(NA,length(interp$doyc))}
if(any(data$nlin==5&data$ngas==1)){interp$co2int5<-approx(data$doyc[data$nlin==5&data$ngas==1],data$co2final[data$nlin==5&data$ngas==1],interp$doyc,rule=2)$y}else{interp$co2int5<-rep(NA,length(interp$doyc))}
} # allow for 2 or 4?
interp$prs<-approx(data$doyc,data$prs,interp$doyc,rule=2)$y
interp$btmp<-approx(data$doyc,data$btmp,interp$doyc,rule=2)$y

# screen for gaps > 30 min.
for(i in c(1:length(interp$doyc))){
	if(any(data$nlin==1&data$ngas==1)){if(min(abs(interp$doyc[i]-data$doyc[data$nlin==1&data$ngas==1]),na.rm=T)>0.01042|
		all(interp$doyc[i]-data$doyc[data$nlin==1&data$ngas==1]>0,na.rm=T)|
		all(interp$doyc[i]-data$doyc[data$nlin==1&data$ngas==1]<0,na.rm=T)){ interp$co2int1[i]<-NA}}
	if(any(data$nlin==2&data$ngas==1)){if(min(abs(interp$doyc[i]-data$doyc[data$nlin==2&data$ngas==1]),na.rm=T)>0.01042|
		all(interp$doyc[i]-data$doyc[data$nlin==2&data$ngas==1]>0,na.rm=T)|
		all(interp$doyc[i]-data$doyc[data$nlin==2&data$ngas==1]<0,na.rm=T)){ interp$co2int2[i]<-NA}}
	if(any(data$nlin==3&data$ngas==1)){if(min(abs(interp$doyc[i]-data$doyc[data$nlin==3&data$ngas==1]),na.rm=T)>0.01042|
		all(interp$doyc[i]-data$doyc[data$nlin==3&data$ngas==1]>0,na.rm=T)|
		all(interp$doyc[i]-data$doyc[data$nlin==3&data$ngas==1]<0,na.rm=T)){ interp$co2int3[i]<-NA}}
	if(nlines==5){
		if(any(data$nlin==4&data$ngas==1)){if(min(abs(interp$doyc[i]-data$doyc[data$nlin==4&data$ngas==1]),na.rm=T)>0.01042|
			all(interp$doyc[i]-data$doyc[data$nlin==4&data$ngas==1]>0,na.rm=T)|
			all(interp$doyc[i]-data$doyc[data$nlin==4&data$ngas==1]<0,na.rm=T)){ interp$co2int4[i]<-NA}}
		if(any(data$nlin==5&data$ngas==1)){if(min(abs(interp$doyc[i]-data$doyc[data$nlin==5&data$ngas==1]),na.rm=T)>0.01042|
			all(interp$doyc[i]-data$doyc[data$nlin==5&data$ngas==1]>0,na.rm=T)|
			all(interp$doyc[i]-data$doyc[data$nlin==5&data$ngas==1]<0,na.rm=T)){ interp$co2int5[i]<-NA}}
	} # allow for 2 or 4?
	if(min(abs(interp$doyc[i]-data$doyc),na.rm=T)>0.01042) {
		interp$prs[i]<-NA
		interp$btmp[i]<-NA
	}
}

if(year/4-trunc(year/4)==0){ # leap year
	intyrfrac=2000+year+(interp$doyc-yrdays-1)/366
}else{
	intyrfrac=2000+year+(interp$doyc-yrdays-1)/365
}

if(nlines==3){
	intmat<-rbind(round(intyrfrac,7),round(interp$co2int1,3),round(interp$co2int2,3),round(interp$co2int3,3),round(interp$prs,3),round(interp$btmp*100-40,2))
}else if(nlines==5){
	intmat<-rbind(round(intyrfrac,7),round(interp$co2int1,3),round(interp$co2int2,3),round(interp$co2int3,3),round(interp$co2int4,3),round(interp$co2int5,3),round(interp$prs,3),round(interp$btmp*100-40,2))
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
if(nlines==3){
	rownames(intmat)<-c('yrfrac','co2int1','co2int2','co2int3','prs','btmp')
}else if(nlines==5){
	rownames(intmat)<-c('yrfrac','co2int1','co2int2','co2int3','co2int4','co2int5','prs','btmp')
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}


# CALCULATE HOURLY MEANS AND SIGMAS

# Generate DateTime values for plotting
datetimec=strptime(paste(year,'-01-01 00:00:00',sep=''),format='%y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*(yrdoyc-1)
#### could speed up below by using tapply

print('Averaging to 60 Minutes')

hourly<-list(doyc=NA,hour=NA,co2hr1=NA,co2hr2=NA,co2hr3=NA,co2hr4=NA,co2hr5=NA,co2sd1=NA,co2sd2=NA,co2sd3=NA,co2sd4=NA,co2sd5=NA,co2dif=NA)

## hourly$doyc and hrlyrfrac correspond to middle of hour (00:30), and reported hour is of start of hour
hourly$doyc<-floor(min(data$doyc,na.rm=T))+(c(1:((ceiling(max(data$doyc,na.rm=T))-floor(min(data$doyc,na.rm=T)))*24))-.5)/24 # series from 0030 on the first day to 2330 on the last day
hourly$hour<-round((hourly$doyc-trunc(hourly$doyc))*24-.5,0) # start of hours, from 0 to 23, corresponding to full series in hourly$doyc

hourly<-data.frame(hourly)

for(i in c(1:length(hourly$doyc))){
	sel1<-data$nlin==1&data$ngas==1&data$doyc>(hourly$doyc[i]-0.5/24)&data$doyc<(hourly$doyc[i]+0.5/24)
	sel2<-data$nlin==2&data$ngas==1&data$doyc>(hourly$doyc[i]-0.5/24)&data$doyc<(hourly$doyc[i]+0.5/24)
	sel3<-data$nlin==3&data$ngas==1&data$doyc>(hourly$doyc[i]-0.5/24)&data$doyc<(hourly$doyc[i]+0.5/24)
	if(nlines==5){
		sel4<-data$nlin==4&data$ngas==1&data$doyc>(hourly$doyc[i]-0.5/24)&data$doyc<(hourly$doyc[i]+0.5/24)
		sel5<-data$nlin==5&data$ngas==1&data$doyc>(hourly$doyc[i]-0.5/24)&data$doyc<(hourly$doyc[i]+0.5/24)
	} # allow 2 or 4?

	# means and sigmas
	if(any(sel1)){
		hourly$co2hr1[i]<-mean(data$co2final[sel1],na.rm=T)
		hourly$co2sd1[i]<-sd(data$co2final[sel1],na.rm=T)
	} else {
		hourly$co2hr1[i]<-NA
		hourly$co2sd1[i]<-NA
	}
	if(any(sel2)){
		hourly$co2hr2[i]<-mean(data$co2final[sel2],na.rm=T)
		hourly$co2sd2[i]<-sd(data$co2final[sel2],na.rm=T)
	} else {
		hourly$co2hr2[i]<-NA
		hourly$co2sd2[i]<-NA
	}
	if(any(sel3)){
		hourly$co2hr3[i]<-mean(data$co2final[sel3],na.rm=T)
		hourly$co2sd3[i]<-sd(data$co2final[sel3],na.rm=T)
	} else {
		hourly$co2hr3[i]<-NA
		hourly$co2sd3[i]<-NA
	}
	if(nlines==5){
		if(any(sel4)){
			hourly$co2hr4[i]<-mean(data$co2final[sel4],na.rm=T)
			hourly$co2sd4[i]<-sd(data$co2final[sel4],na.rm=T)
		} else {
			hourly$co2hr4[i]<-NA
			hourly$co2sd4[i]<-NA
		}
		if(any(sel5)){
			hourly$co2hr5[i]<-mean(data$co2final[sel5],na.rm=T)
			hourly$co2sd5[i]<-sd(data$co2final[sel5],na.rm=T)
		} else {
			hourly$co2hr5[i]<-NA
			hourly$co2sd5[i]<-NA
		}
	} # allow 2 or 4?
	# line differences
	if(!is.na(midlin)){
	hourly$co2dif[i]<-hourly[,paste('co2hr',maxlin,sep='')][i]-hourly[,paste('co2hr',midlin,sep='')][i]
	} else {
	hourly$co2dif[i]<-NA
	}
}

if(year/4-trunc(year/4)==0){ # leap year
	hrlyrfrac=2000+year+(hourly$doyc-yrdays-1)/366
}else{
	hrlyrfrac=2000+year+(hourly$doyc-yrdays-1)/365
}

if(nlines==3){
	hrlmat<-rbind(round(hrlyrfrac,7),hourly$hour,round(hourly$co2hr1,3),round(hourly$co2hr2,3),round(hourly$co2hr3,3),round(hourly$co2sd1,3),round(hourly$co2sd2,3),round(hourly$co2sd3,3),round(hourly$co2dif,3))
}else if(nlines==5){
	hrlmat<-rbind(round(hrlyrfrac,7),hourly$hour,round(hourly$co2hr1,3),round(hourly$co2hr2,3),round(hourly$co2hr3,3),round(hourly$co2hr4,3),round(hourly$co2hr5,3),round(hourly$co2sd1,3),
		round(hourly$co2sd2,3),round(hourly$co2sd3,3),round(hourly$co2sd4,3),round(hourly$co2sd5,3),round(hourly$co2dif,3))
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
if(nlines==3){
	rownames(hrlmat)<-c('yrfrac','hour','co2hr1','co2hr2','co2hr3','co2sd1','co2sd2','co2sd3','co2dif')
}else if(nlines==5){
	rownames(hrlmat)<-c('yrfrac','hour','co2hr1','co2hr2','co2hr3','co2hr4','co2hr5','co2sd1','co2sd2','co2sd3','co2sd4','co2sd5','co2dif')
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}


# CALCULATE MEAN DIURNAL CYCLES

print('Calculating Mean Diurnal Cycles at 15-min Resolution')

interp$dayf<-interp$doyc-trunc(interp$doyc)
diur<-list(dayf=NULL,co2diur1=NULL,co2diur2=NULL,co2diur3=NULL,co2diur4=NULL,co2diur5=NULL,prs=NULL,btmp=NULL)
diur$dayf<-sort(unique(round(interp$dayf,4)))
for(i in c(1:length(diur$dayf))){
	diur$co2diur1[i]<-mean(interp$co2int1[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
	diur$co2diur2[i]<-mean(interp$co2int2[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
	diur$co2diur3[i]<-mean(interp$co2int3[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
	if(nlines==5){
		diur$co2diur4[i]<-mean(interp$co2int4[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
		diur$co2diur5[i]<-mean(interp$co2int5[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
	} # allow 2 or 4?
	diur$prs[i]<-mean(interp$prs[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
	diur$btmp[i]<-mean(interp$btmp[round(interp$dayf,4)==diur$dayf[i]],na.rm=T)
}

print('Calculating Diurnal Cycles of CO2-sigma Quantiles at Hourly Resolution for Highest Inlet')
hours<-c(0:23) 
sigmaq<-matrix(NA,24,7) # only doing top line
for(i in c(1:24)){
	sigmaq[i,]<-quantile(hourly[,paste('co2sd',maxlin,sep='')][hourly$hour==i-1],probs=c(0,.1,.25,.5,.75,.9,1),na.rm=T)
}


# PLOT FULL TIMESERIES

print('Plotting Full Timeseries')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_fullts.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_fullts.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(2,1))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

plot(datetimec,data$co2final,type='n',ylab='CO2 (ppm)',xlab='Date (UTC)',main=paste(loc,' ','AIRCOA ',unit,'  ',startchr,' to ',endchr,'  Calibrated Timeseries'))
points(datetimec[data$ngas>1],data$co2final[data$ngas>1])
points(datetimec[data$nlin==1&data$ngas==1],data$co2final[data$nlin==1&data$ngas==1],col=2,cex=0.25)
points(datetimec[data$nlin==2&data$ngas==1],data$co2final[data$nlin==2&data$ngas==1],col=3,cex=0.25)
points(datetimec[data$nlin==3&data$ngas==1],data$co2final[data$nlin==3&data$ngas==1],col=4,cex=0.25)
if(nlines==5){
points(datetimec[data$nlin==4&data$ngas==1],data$co2final[data$nlin==4&data$ngas==1],col=5,cex=0.25)
points(datetimec[data$nlin==5&data$ngas==1],data$co2final[data$nlin==5&data$ngas==1],col=6,cex=0.25)
}
if(F){ # below needs to be updated to work with DateTime (or scrapped)
x<-data$doyc[data$nlin==1&data$ngas==1]
y<-data$co2final[data$nlin==1&data$ngas==1]
### could speed up by doing points instead of segments
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=2)
	}
}
}
x<-data$doyc[data$nlin==2&data$ngas==1]
y<-data$co2final[data$nlin==2&data$ngas==1]
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=3)
	}
}
}
x<-data$doyc[data$nlin==3&data$ngas==1]
y<-data$co2final[data$nlin==3&data$ngas==1]
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=4)
	}
}
}
if(nlines==5){
	x<-data$doyc[data$nlin==4&data$ngas==1]
	y<-data$co2final[data$nlin==4&data$ngas==1]
	if(length(x)>1){
	for(i in c(2:length(x))){
		if(x[i]-x[i-1]<0.02084){
			segments(x[i-1],y[i-1],x[i],y[i],col=5)
		}
	}
	}
	x<-data$doyc[data$nlin==5&data$ngas==1]
	y<-data$co2final[data$nlin==5&data$ngas==1]
	if(length(x)>1){
	for(i in c(2:length(x))){
		if(x[i]-x[i-1]<0.02084){
			segments(x[i-1],y[i-1],x[i],y[i],col=6)
		}
	}
	}
} # allow for 2 or 4?
} # if(F)

curhghts<-c('Line 1','Line 2','Line 3','Line 4','Line 5')
for(j in c(1:length(inlets$loc))){
	if(inlets$loc[j]==loc&(inlets$date[j]+inlets$dayf[j])<(as.numeric(start)+ico)){curhghts<-c(paste(inlets$L1[j],'m'),
		paste(inlets$L2[j],'m'),paste(inlets$L3[j],'m'),paste(inlets$L4[j],'m'),paste(inlets$L5[j],'m'))} # assumes inlet heights are not changing within period
}

if(length(data$doyc[data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0])>0){ # do not make second plot if no sample data
plot(datetimec[data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0],data$co2final[data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0],type='n',ylab='CO2 (ppm)',xlab='Date (UTC)')
points(datetimec[data$nlin==1&data$ngas==1],data$co2final[data$nlin==1&data$ngas==1],col=2,cex=0.25)
points(datetimec[data$nlin==2&data$ngas==1],data$co2final[data$nlin==2&data$ngas==1],col=3,cex=0.25)
points(datetimec[data$nlin==3&data$ngas==1],data$co2final[data$nlin==3&data$ngas==1],col=4,cex=0.25)
if(nlines==5){
points(datetimec[data$nlin==4&data$ngas==1],data$co2final[data$nlin==4&data$ngas==1],col=5,cex=0.25)
points(datetimec[data$nlin==5&data$ngas==1],data$co2final[data$nlin==5&data$ngas==1],col=6,cex=0.25)
}
if(F){ # below needs to be updated to work with DateTime (or scrapped)
x<-data$doyc[data$nlin==1&data$ngas==1]
y<-data$co2final[data$nlin==1&data$ngas==1]
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=2)
	}
}
}
x<-data$doyc[data$nlin==2&data$ngas==1]
y<-data$co2final[data$nlin==2&data$ngas==1]
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=3)
	}
}
}
x<-data$doyc[data$nlin==3&data$ngas==1]
y<-data$co2final[data$nlin==3&data$ngas==1]
if(length(x)>1){
for(i in c(2:length(x))){
	if(x[i]-x[i-1]<0.02084){
		segments(x[i-1],y[i-1],x[i],y[i],col=4)
	}
}
}
if(nlines==5){
	x<-data$doyc[data$nlin==4&data$ngas==1]
	y<-data$co2final[data$nlin==4&data$ngas==1]
	if(length(x)>1){
	for(i in c(2:length(x))){
		if(x[i]-x[i-1]<0.02084){
			segments(x[i-1],y[i-1],x[i],y[i],col=5)
		}
	}
	}
	x<-data$doyc[data$nlin==5&data$ngas==1]
	y<-data$co2final[data$nlin==5&data$ngas==1]
	if(length(x)>1){
	for(i in c(2:length(x))){
		if(x[i]-x[i-1]<0.02084){
			segments(x[i-1],y[i-1],x[i],y[i],col=6)
		}
	}
	}
} # allow for 2 or 4?
} # if(F)

sel<-rep(F,5)
if(any(data$nlin==1&data$ngas==1)){sel[1]<-T}
if(any(data$nlin==2&data$ngas==1)){sel[2]<-T}
if(any(data$nlin==3&data$ngas==1)){sel[3]<-T}
if(nlines==5){
	if(any(data$nlin==4&data$ngas==1)){sel[4]<-T}
	if(any(data$nlin==5&data$ngas==1)){sel[5]<-T}
} # allow for 2 or 4?
legend((par('usr')[2]-par('usr')[1])*.1+par('usr')[1],(par('usr')[4]-par('usr')[3])*.9+par('usr')[3],curhghts[sel],lty=c(1,1,1,1,1)[sel],col=c(2,3,4,5,6)[sel],ncol=2)

} else { # do not make second plot if no sample data
print('No sample data for second plot')
}

dev.off()

# PLOT DIURNAL CYCLE

print('Plotting Diurnal Cycles')

if(png){png(paste(loc,"_",startchr,"_",endchr,outflag,"_diur.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",startchr,"_",endchr,outflag,"_diur.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(1,2))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

if(any(!is.na(c(diur$co2diur1,diur$co2diur2,diur$co2diur3,diur$co2diur4,diur$co2diur5)))){ # do not plot if no sample data
if(nlines==3){
	plot(rep(diur$dayf,3)*24,c(diur$co2diur1,diur$co2diur2,diur$co2diur3),type='n',ylab='CO2 (ppm)',xlab='Hour of Day (UTC)',xlim=c(0,24),xaxp=c(0,24,6))
}else if(nlines==5){
	plot(rep(diur$dayf,5)*24,c(diur$co2diur1,diur$co2diur2,diur$co2diur3,diur$co2diur4,diur$co2diur5),type='n',ylab='CO2 (ppm)',xlab='Hour of Day (UTC)',xlim=c(0,24),xaxp=c(0,24,6))
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
lines(diur$dayf*24,diur$co2diur1,col=2)
lines(diur$dayf*24,diur$co2diur2,col=3)
lines(diur$dayf*24,diur$co2diur3,col=4)
if(nlines==5){
	lines(diur$dayf*24,diur$co2diur4,col=5)
	lines(diur$dayf*24,diur$co2diur5,col=6)
} # allow for 2 or 4?
sel<-rep(F,5)
if(any(!is.na(diur$co2diur1))){sel[1]<-T}
if(any(!is.na(diur$co2diur2))){sel[2]<-T}
if(any(!is.na(diur$co2diur3))){sel[3]<-T}
if(nlines==5){
	if(any(!is.na(diur$co2diur4))){sel[4]<-T}
	if(any(!is.na(diur$co2diur5))){sel[5]<-T}
} # allow for 2 or 4?
legend(mean(diur$dayf,na.rm=T)*24,max(c(diur$co2diur1,diur$co2diur2,diur$co2diur3,diur$co2diur4,diur$co2diur5),na.rm=T),curhghts[sel],lty=c(1,1,1,1,1)[sel],col=c(2,3,4,5,6)[sel],ncol=2)

boxsd=hourly[,paste('co2sd',maxlin,sep='')]
#print(boxsd)
#print(hourly$hour)
#print(unique(hourly$hour)+0.5)
plot(boxsd,boxsd,xlim=c(0,24),xaxp=c(0,24,6),type='n',ylab='Standard Deviation (ppm)',xlab='Hour of Day (UTC)')
boxplot(boxsd ~ hourly$hour,ylab='Standard Deviation (ppm)',xlab='Hour of Day (UTC)',border='Blue',at=(unique(hourly$hour)+0.5)[is.element(c(0:23),hourly$hour[!is.na(boxsd)])],add=T,axes=F)
abline(h=1.0,col='Gray')

mtext(paste(loc,' ','AIRCOA',unit,' ',startchr,'to',endchr,outflag,' Mean Diurnal Cycle and Hourly Standard Deviations'),3,0.5,outer=T)

}# do not plot if no sample data

dev.off()


# PLOT WEEKLY TIMESERIES

print('Plotting Weekly Timeseries')

days<-c(floor(min(yrdoyc,na.rm=T)):floor(max(yrdoyc,na.rm=T)))
days<-days[days!=0]
weeki<-days[(days-1)/7-trunc((days-1)/7)==0]
if(length(weeki)>0){
if(weeki[1]>days[1]) weeki<-c(weeki[1]-7,weeki)
weeke<-weeki+7

for(i in c(1:length(weeki))){

if(any(yrdoyc>=weeki[i]&yrdoyc<weeke[i])){

# generate cumulative days for filenames and figure labels
prvdays<-c(0,cumsum(dpm)[1:11])

# generate text month and day for filenames and figure labels
mon<-c(1:12)[min((weeki[i]-prvdays)[weeki[i]-prvdays>0],na.rm=T)==weeki[i]-prvdays]
if(mon<10) {mnt1<-paste('0',as.character(mon),sep='')} else { mnt1<-as.character(mon)}
day<-weeki[i]-prvdays[mon]
if(day<10) {dyt1<-paste('0',as.character(day),sep='')} else { dyt1<-as.character(day)}
mon<-c(1:12)[min((weeke[i]-1-prvdays)[weeke[i]-1-prvdays>0],na.rm=T)==weeke[i]-1-prvdays]
if(mon<10) {mnt2<-paste('0',as.character(mon),sep='')} else { mnt2<-as.character(mon)}
day<-weeke[i]-1-prvdays[mon]
if(day<10) {dyt2<-paste('0',as.character(day),sep='')} else { dyt2<-as.character(day)}

if(png){png(paste(loc,"_",yrt,mnt1,dyt1,"_",yrt,mnt2,dyt2,"_weekts.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",yrt,mnt1,dyt1,"_",yrt,mnt2,dyt2,"_weekts.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(1,1))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

xlim=c(strptime(paste(year,'-01-01 00:00:00',sep=''),format='%y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*(weeki[i]-1),strptime(paste(year,'-01-01 00:00:00',sep=''),format='%y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*(weeke[i]-1))
plot(datetimec[yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[yrdoyc>=weeki[i]&yrdoyc<weeke[i]],type='n',ylab='CO2 (ppm)',xlab='Date (UTC)',xlim=xlim,
	main=paste(loc,' ','AIRCOA ',unit,'  Week ',yrt,mnt1,dyt1,' to ',yrt,mnt2,dyt2,'  Calibrated Timeseries',sep=''))
points(datetimec[data$ngas>1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$ngas>1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]])
lines(datetimec[data$nlin==1&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$nlin==1&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],col=2)
lines(datetimec[data$nlin==2&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$nlin==2&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],col=3)
lines(datetimec[data$nlin==3&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$nlin==3&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],col=4)
if(nlines==5){
	lines(datetimec[data$nlin==4&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$nlin==4&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],col=5)
	lines(datetimec[data$nlin==5&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],data$co2final[data$nlin==5&data$ngas==1&yrdoyc>=weeki[i]&yrdoyc<weeke[i]],col=6)
} # allow for 2 or 4?
sel<-rep(F,5)
if(any(data$nlin==1&data$ngas==1)){sel[1]<-T}
if(any(data$nlin==2&data$ngas==1)){sel[2]<-T}
if(any(data$nlin==3&data$ngas==1)){sel[3]<-T}
if(nlines==5){
        if(any(data$nlin==4&data$ngas==1)){sel[4]<-T}
        if(any(data$nlin==5&data$ngas==1)){sel[5]<-T}
} # allow for 2 or 4?
legend((par('usr')[2]-par('usr')[1])*.1+par('usr')[1],(par('usr')[4]-par('usr')[3])*.9+par('usr')[3],curhghts[sel],lty=c(1,1,1,1,1)[sel],col=c(2,3,4,5,6)[sel],ncol=2)


dev.off()

}
}
}

# PLOT HIGH SIDE CYLINDER PRESSURES

print('Plotting High Side Cylinder Pressures')

### add cylinder concentration histories

hiside<-scan('../../aircoa_hiside.txt',what=list(loc='',unit='',date='',time='',hs2p=0,hs1p=0,ls1p=0,ls2p=0,ltp=0),skip=1)
sel<-hiside$loc==loc

cylrec<-scan('../../aircoa_cylrec.txt',what=list(loc='',unit='',date='',time='',HS2='',HS1='',LS1='',LS2='',LT=''),skip=1)
selcyl<-cylrec$loc==loc

if(length(hiside$date[sel])>0){

if(png){png(paste(loc,"_hiside.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_hiside.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(1,1))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

# calculate day since Dec. 31, 2004 (noon on Jan. 1, 2005 = 1.5)
dpy<-c(NA,NA,NA,NA,0,365,365,365,366,365,365,365,366,365,365,365,366,365,365,365,366) # good thru 2020

# calculate year fraction
cyldatetime=strptime(paste(cylrec$date,cylrec$time),format='%y%m%d %H%M',tz="GMT")
cylyrdays=rep(365,length(cyldatetime$year))
cylyrdays[(cyldatetime$year+1900)/4-trunc((cyldatetime$year+1900)/4)==0]=366
cylrec$yrfrac=round(cyldatetime$year+1900+as.numeric(difftime(cyldatetime,strptime(paste(cyldatetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/cylyrdays,7)
hsdatetime=strptime(paste(hiside$date,hiside$time),format='%y%m%d %H%M',tz="GMT")
hsyrdays=rep(365,length(hsdatetime$year))
hsyrdays[(hsdatetime$year+1900)/4-trunc((hsdatetime$year+1900)/4)==0]=366
hiside$yrfrac=round(hsdatetime$year+1900+as.numeric(difftime(hsdatetime,strptime(paste(hsdatetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/hsyrdays,7)

# find install dates for each cylinder
hs2inst<-NA
hs1inst<-NA
ls1inst<-NA
ls2inst<-NA
ltinst<-NA
hs2ref<-'nocyl'
hs1ref<-'nocyl'
ls1ref<-'nocyl'
ls2ref<-'nocyl'
ltref<-'nocyl'
hs2breaks=NULL
hs1breaks=NULL
ls1breaks=NULL
ls2breaks=NULL
ltbreaks=NULL
for(i in c(1:length(cylrec$date))[selcyl]){
	if(cylrec$HS2[i]!=hs2ref){
		hs2inst<-cylrec$yrfrac[i]
		hs2ref<-cylrec$HS2[i]
		hs2breaks=c(hs2breaks,cylrec$yrfrac[i]-1/60/24/365)
	}
	if(cylrec$HS1[i]!=hs1ref){
		hs1inst<-cylrec$yrfrac[i]
		hs1ref<-cylrec$HS1[i]
		hs1breaks=c(hs1breaks,cylrec$yrfrac[i]-1/60/24/365)
	}
	if(cylrec$LS1[i]!=ls1ref){
		ls1inst<-cylrec$yrfrac[i]
		ls1ref<-cylrec$LS1[i]
		ls1breaks=c(ls1breaks,cylrec$yrfrac[i]-1/60/24/365)
	}
	if(cylrec$LS2[i]!=ls2ref){
		ls2inst<-cylrec$yrfrac[i]
		ls2ref<-cylrec$LS2[i]
		ls2breaks=c(ls2breaks,cylrec$yrfrac[i]-1/60/24/365)
	}
	if(cylrec$LT[i]!=ltref){
		ltinst<-cylrec$yrfrac[i]
		ltref<-cylrec$LT[i]
		ltbreaks=c(ltbreaks,cylrec$yrfrac[i]-1/60/24/365)
	}
}

plot(hiside$yrfrac[sel],hiside$ls2p[sel],type='n',xlab='Year',ylab='psi',ylim=c(0,2500),main=paste(loc,'High Side Cylinder Pressures as of ',hiside$date[sel][length(hiside$hs2p[sel])]))
points(hiside$yrfrac[sel],hiside$hs2p[sel],col='Blue')
x=c(hiside$yrfrac[sel],hs2breaks)
y=c(hiside$hs2p[sel],rep(NA,length(hs2breaks)))
y=y[order(x)]
x=x[order(x)]
lines(x,y,col='Blue')
points(hiside$yrfrac[sel],hiside$hs1p[sel],col='Green')
x=c(hiside$yrfrac[sel],hs1breaks)
y=c(hiside$hs1p[sel],rep(NA,length(hs1breaks)))
y=y[order(x)]
x=x[order(x)]
lines(x,y,col='Green')
points(hiside$yrfrac[sel],hiside$ls1p[sel],col='Red')
x=c(hiside$yrfrac[sel],ls1breaks)
y=c(hiside$ls1p[sel],rep(NA,length(ls1breaks)))
y=y[order(x)]
x=x[order(x)]
lines(x,y,col='Red')
points(hiside$yrfrac[sel],hiside$ls2p[sel],col='Purple')
x=c(hiside$yrfrac[sel],ls2breaks)
y=c(hiside$ls2p[sel],rep(NA,length(ls2breaks)))
y=y[order(x)]
x=x[order(x)]
lines(x,y,col='Purple')
points(hiside$yrfrac[sel],hiside$ltp[sel],col='Orange')
x=c(hiside$yrfrac[sel],ltbreaks)
y=c(hiside$ltp[sel],rep(NA,length(ltbreaks)))
y=y[order(x)]
x=x[order(x)]
lines(x,y,col='Orange')

# calculate venting rate
if(length(hiside$yrfrac[sel&hiside$yrfrac>hs2inst&!is.na(hiside$hs2p)])>1){
hs2sl<-lsfit(hiside$yrfrac[sel&hiside$yrfrac>hs2inst&!is.na(hiside$hs2p)],hiside$hs2p[sel&hiside$yrfrac>hs2inst&!is.na(hiside$hs2p)])$coef[2]/12 # per month
if(round(hs2sl,3)==0){ hs2dl=NA } else { hs2dl=(hiside$hs2p[sel&!is.na(hiside$hs2p)][length(hiside$hs2p[sel&!is.na(hiside$hs2p)])]-300)/hs2sl*-30.4 } # days
} else {
hs2sl<-NA
hs2dl<-NA
}
if(length(hiside$yrfrac[sel&hiside$yrfrac>hs1inst&!is.na(hiside$hs1p)])>1){
hs1sl<-lsfit(hiside$yrfrac[sel&hiside$yrfrac>hs1inst&!is.na(hiside$hs1p)],hiside$hs1p[sel&hiside$yrfrac>hs1inst&!is.na(hiside$hs1p)])$coef[2]/12 # per month
if(round(hs1sl,3)==0){ hs1dl=NA } else { hs1dl=(hiside$hs1p[sel&!is.na(hiside$hs1p)][length(hiside$hs1p[sel&!is.na(hiside$hs1p)])]-300)/hs1sl*-30.4 } # days
} else {
hs1sl<-NA
hs1dl<-NA
}
if(length(hiside$yrfrac[sel&hiside$yrfrac>ls1inst&!is.na(hiside$ls1p)])>1){
ls1sl<-lsfit(hiside$yrfrac[sel&hiside$yrfrac>ls1inst&!is.na(hiside$ls1p)],hiside$ls1p[sel&hiside$yrfrac>ls1inst&!is.na(hiside$ls1p)])$coef[2]/12 # per month
if(round(ls1sl,3)==0){ ls1dl=NA } else { ls1dl=(hiside$ls1p[sel&!is.na(hiside$ls1p)][length(hiside$ls1p[sel&!is.na(hiside$ls1p)])]-300)/ls1sl*-30.4 } # days
} else {
ls1sl<-NA
ls1dl<-NA
}
if(length(hiside$yrfrac[sel&hiside$yrfrac>ls2inst&!is.na(hiside$ls2p)])>1){
ls2sl<-lsfit(hiside$yrfrac[sel&hiside$yrfrac>ls2inst&!is.na(hiside$ls2p)],hiside$ls2p[sel&hiside$yrfrac>ls2inst&!is.na(hiside$ls2p)])$coef[2]/12 # per month
if(round(ls2sl,3)==0){ ls2dl=NA } else { ls2dl=(hiside$ls2p[sel&!is.na(hiside$ls2p)][length(hiside$ls2p[sel&!is.na(hiside$ls2p)])]-300)/ls2sl*-30.4 } # days
} else {
ls2sl<-NA
ls2dl<-NA
}
if(length(hiside$yrfrac[sel&hiside$yrfrac>ltinst&!is.na(hiside$ltp)])>1){
ltsl<-lsfit(hiside$yrfrac[sel&hiside$yrfrac>ltinst&!is.na(hiside$ltp)],hiside$ltp[sel&hiside$yrfrac>ltinst&!is.na(hiside$ltp)])$coef[2]/12 # per month
if(round(ltsl,3)==0){ ltdl=NA } else { ltdl=(hiside$ltp[sel&!is.na(hiside$ltp)][length(hiside$ltp[sel&!is.na(hiside$ltp)])]-300)/ltsl*-30.4 } # days
} else {
ltsl<-NA
ltdl<-NA
}

legend(max(hiside$yrfrac[sel])-diff(range(hiside$yrfrac[sel]))*.4,2250,c(
paste('HS2   (',round(hs2sl,0),'psi/mon,',round(hs2dl,0),'days left )'),
paste('HS1   (',round(hs1sl,0),'psi/mon,',round(hs1dl,0),'days left )'),
paste('LS1   (',round(ls1sl,0),'psi/mon,',round(ls1dl,0),'days left )'),
paste('LS2   (',round(ls2sl,0),'psi/mon,',round(ls2dl,0),'days left )'),
paste('LT   (',round(ltsl,0),'psi/mon ,',round(ltdl,0),'days left)')),lty=c(1,1,1,1,1),pch=c(0,0,0,0,0),col=c('Blue','Green','Red','Purple','Orange'))

dev.off()

}

if(!skipupdate){
print('Updating complete timeseries')

# UPDATE COMPLETE TIMESERIES

longstr<-'050801'
longend<-'999999'

oldruns<-system(paste('ls */',loc,'/',loc,'*.con',sep=''),T)
if(length(oldruns)>0){
	uselist<-rep(F,length(oldruns))
	for(j in c(1:length(oldruns))){ 
		if(as.numeric(substr(oldruns[j],14,19))>=as.numeric(longstr)&as.numeric(substr(oldruns[j],21,26))<=as.numeric(longend)){
			# starts after longstr and ends before longend
			uselist[j]<-T
		}
		if(substr(oldruns[j],14,19)==startchr|substr(oldruns[j],21,26)==endchr){ # same start or end as this run
			uselist[j]<-F
		}
			
	}
	if(any(uselist)){
		lstrchr<-unique(substr(oldruns,14,19)[uselist][as.numeric(substr(oldruns,14,19))[uselist]==min(as.numeric(substr(oldruns,14,19))[uselist])])
		lendchr<-unique(substr(oldruns,21,26)[uselist][as.numeric(substr(oldruns,21,26))[uselist]==max(as.numeric(substr(oldruns,21,26))[uselist])])
		if(start<as.numeric(lstrchr)){lstrchr<-startchr}
		if(end>as.numeric(lendchr)){lendchr<-endchr}
		print(paste('Updating Complete Timeseries from:',lstrchr,'to',lendchr))
		print(system('date',T))
		compdata<-NULL
		for(j in c(1:length(oldruns))[uselist]){
			if(nlines==3){
				input<-data.frame(scan(oldruns[j],what=list(yrfrac=0,co2i=0,prs=0,ltmp=0,btmp=0,rh=0,sfl=0,fl1=0,
				fl2=0,fl3=0,fl4=0,nlin=0,ngas=0,up=0,sdco2=0,sdprs=0,sdsfl=0,hs2=0,hs1=0,ls1=0,ls2=0,shpbn=0,
				pcorr=0,co2p=0,co2t=0,co2f=0,lin0=0,lin1=0,cur0=0,cur1=0,cur2=0,co2c=0,co2n=0,co2final=0),skip=1))
			}else if(nlines==5){
				input<-data.frame(scan(oldruns[j],what=list(yrfrac=0,co2i=0,prs=0,ltmp=0,btmp=0,rh=0,sfl=0,fl1=0,
				fl2=0,fl3=0,fl4=0,fl5=0,fl6=0,nlin=0,ngas=0,up=0,sdco2=0,sdprs=0,sdsfl=0,hs2=0,hs1=0,ls1=0,ls2=0,shpbn=0,
				pcorr=0,co2p=0,co2t=0,co2f=0,lin0=0,lin1=0,cur0=0,cur1=0,cur2=0,co2c=0,co2n=0,co2final=0),skip=1))
			}else{
				print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
			}
			compdata<-rbind(compdata,cbind(input$yrfrac,input$nlin,input$ngas,input$co2final)) # presently only using 4 columns
		}

		mesdata<-NULL
		for(j in c(1:length(oldruns))[uselist]){
			input<-data.frame(scan(paste(substr(oldruns[j],1,27),'mes',sep=''),what=list(yrfrac=0,co2final=0,nlin=0,height=0,prs=0,btmp=0,sdco2=0,trng=0),skip=1))
			mesdata<-rbind(mesdata,input)
		}

		intdata<-NULL
		for(j in c(1:length(oldruns))[uselist]){
			if(nlines==3){
				input<-data.frame(scan(paste(substr(oldruns[j],1,27),'int',sep=''),what=list(yrfrac=0,co2int1=0,co2int2=0,co2int3=0,prs=0,btmp=0),skip=1))
			}else if(nlines==5){
				input<-data.frame(scan(paste(substr(oldruns[j],1,27),'int',sep=''),what=list(yrfrac=0,co2int1=0,co2int2=0,co2int3=0,co2int4=0,co2int5=0,prs=0,btmp=0),skip=1))
			}else{
				print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
			}
			intdata<-rbind(intdata,input)
		}

		hrldata<-NULL
		hrlrun<-system(paste('ls ',substr(oldruns[j],1,27),'hrl',sep=''),T) # replace with "if exist"
		if(length(hrlrun)>0){ # some of the early lab runs have not been rerun to produce hrl files
		for(j in c(1:length(oldruns))[uselist]){
			if(nlines==3){
				input<-data.frame(scan(paste(substr(oldruns[j],1,27),'hrl',sep=''),what=list(yrfrac=0,hour=0,co2hr1=0,co2hr2=0,co2hr3=0,co2sd1=0,co2sd2=0,co2sd3=0,co2dif=0),skip=1))
			}else if(nlines==5){
				input<-data.frame(scan(paste(substr(oldruns[j],1,27),'hrl',sep=''),what=list(yrfrac=0,hour=0,co2hr1=0,co2hr2=0,co2hr3=0,co2hr4=0,co2hr5=0,co2sd1=0,co2sd2=0,co2sd3=0,co2sd4=0,co2sd5=0,co2dif=0),skip=1))
			}else{
				print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
			}
			hrldata<-rbind(hrldata,input)
		}
		} else {
        		print(paste('No old hrl files for unit in period: ',substr(oldruns[j],10,27)))
		}

		## temporary fix for change to year.frac from day since dec. 31, 2004
		compdata[,1][compdata[,1]<366]=(compdata[,1][compdata[,1]<366]-1)/365+2005
		compdata[,1][compdata[,1]<731]=(compdata[,1][compdata[,1]<731]-365-1)/365+2006
		compdata[,1][compdata[,1]<1096]=(compdata[,1][compdata[,1]<1096]-365-365-1)/365+2007
		compdata[,1][compdata[,1]<1462]=(compdata[,1][compdata[,1]<1462]-365-365-365-1)/366+2008
		compdata[,1][compdata[,1]<2000]=(compdata[,1][compdata[,1]<2000]-365-365-365-366-1)/365+2009
		mesdata$yrfrac[mesdata$yrfrac<366]=(mesdata$yrfrac[mesdata$yrfrac<366]-1)/365+2005
		mesdata$yrfrac[mesdata$yrfrac<731]=(mesdata$yrfrac[mesdata$yrfrac<731]-365-1)/365+2006
		mesdata$yrfrac[mesdata$yrfrac<1096]=(mesdata$yrfrac[mesdata$yrfrac<1096]-365-365-1)/365+2007
		mesdata$yrfrac[mesdata$yrfrac<1462]=(mesdata$yrfrac[mesdata$yrfrac<1462]-365-365-365-1)/366+2008
		mesdata$yrfrac[mesdata$yrfrac<2000]=(mesdata$yrfrac[mesdata$yrfrac<2000]-365-365-365-366-1)/365+2009
		intdata$yrfrac[intdata$yrfrac<366]=(intdata$yrfrac[intdata$yrfrac<366]-1)/365+2005
		intdata$yrfrac[intdata$yrfrac<731]=(intdata$yrfrac[intdata$yrfrac<731]-365-1)/365+2006
		intdata$yrfrac[intdata$yrfrac<1096]=(intdata$yrfrac[intdata$yrfrac<1096]-365-365-1)/365+2007
		intdata$yrfrac[intdata$yrfrac<1462]=(intdata$yrfrac[intdata$yrfrac<1462]-365-365-365-1)/366+2008
		intdata$yrfrac[intdata$yrfrac<2000]=(intdata$yrfrac[intdata$yrfrac<2000]-365-365-365-366-1)/365+2009
		hrldata$yrfrac[hrldata$yrfrac<366]=(hrldata$yrfrac[hrldata$yrfrac<366]-1)/365+2005
		hrldata$yrfrac[hrldata$yrfrac<731]=(hrldata$yrfrac[hrldata$yrfrac<731]-365-1)/365+2006
		hrldata$yrfrac[hrldata$yrfrac<1096]=(hrldata$yrfrac[hrldata$yrfrac<1096]-365-365-1)/365+2007
		hrldata$yrfrac[hrldata$yrfrac<1462]=(hrldata$yrfrac[hrldata$yrfrac<1462]-365-365-365-1)/366+2008
		hrldata$yrfrac[hrldata$yrfrac<2000]=(hrldata$yrfrac[hrldata$yrfrac<2000]-365-365-365-366-1)/365+2009
        } else {
        	print('No old data files for unit in selected period')
		lstrchr<-startchr
		lendchr<-endchr
		compdata<-NULL
		mesdata<-NULL
		intdata<-NULL
		hrldata<-NULL
	}

} else {
        print('No old data files for unit')
	lstrchr<-startchr
	lendchr<-endchr
	compdata<-NULL
	mesdata<-NULL
	intdata<-NULL
	hrldata<-NULL
}

print('Comparison file')
print(system('date',T))

if(nlines==3){
	rownames(outmat)<-c('yrfrac','co2i','prs','ltmp','btmp','rh','sfl','fl1','fl2','fl3','fl4','nlin','ngas','up','sdco2',
	'sdprs','sdsfl','hs2','hs1','ls1','ls2','shpbn','pcorr','co2p','co2t','co2f','lin0','lin1','cur0','cur1',
	'cur2','co2c','co2n','co2final')
}else if(nlines==5){
	rownames(outmat)<-c('yrfrac','co2i','prs','ltmp','btmp','rh','sfl','fl1','fl2','fl3','fl4','fl5','fl6','nlin','ngas','up','sdco2',
	'sdprs','sdsfl','hs2','hs1','ls1','ls2','shpbn','pcorr','co2p','co2t','co2f','lin0','lin1','cur0','cur1',
	'cur2','co2c','co2n','co2final')
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}


# presently only using 4 columns (- 1 line + 3 lines)
outmat<-data.frame(t(outmat)) 
compdata<-rbind(compdata,cbind(outmat$yrfrac,outmat$nlin,outmat$ngas,outmat$co2final))
colnames(compdata)<-c('yrfrac','nlin','ngas','co2final')

compdata<-data.frame(compdata)
# reorder
compdata<-compdata[order(compdata$yrfrac),]
compdata$yrfrac=round(compdata$yrfrac,7)
compdata2=rbind(c("YEAR.FRAC","NLIN","NGAS","CO2FINAL"),compdata) # binding text somehow ensures 7 digits in yrfrac output
write(t(compdata2),paste(loc,"_",lstrchr,"_",lendchr,".lco",sep=''),ncol=4) # presently only using 4 columns

print('Measured file')
print(system('date',T))

mesdata<-rbind(mesdata,t(mesmat))
mesdata<-data.frame(mesdata)
# reorder
mesdata<-mesdata[order(mesdata$yrfrac),]

# generate text year, month, day, hour, minute, and second
ye=trunc(mesdata$yrfrac)
lmedpy=rep(365,length(mesdata$yrfrac))
lmedpy[ye/4-trunc(ye/4)==0]=366
lmedatetime=round.POSIXt(as.POSIXlt(strptime(paste(ye,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*lmedpy*(mesdata$yrfrac-ye)),units="secs")
mo=lmedatetime$mon+1
da=lmedatetime$mday
ho=lmedatetime$hour
mi=lmedatetime$min
se=lmedatetime$sec
### do this above for monthly, then just read in?

flags<-rep('PRE',nrow(mesdata)) # placeholder
	
write("",paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''))
fairuse<-scan('../../aircoa_fairuse.txt',sep="\n",what='')
for(j in c(1:(length(fairuse)-1))){
	write(fairuse[j],paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
}
write(paste('This file written: ',system('date',T)),paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
write(fairuse[length(fairuse)],paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
write(c("YEAR.FRAC","YEAR","MON","DAY","HOUR","MIN","SEC","HGT","CO2","SIGMA","FLAG"),paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),ncol=11,append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),append=T)
mesdata$co2final[is.na(mesdata$co2final)]<- -999.99
mesdata$sdco2[is.na(mesdata$sdco2)]<- -999.99
write(rbind(round(mesdata$yrfrac,7),ye,mo,da,ho,mi,se,mesdata$height,mesdata$co2final,mesdata$sdco2,flags),paste(loc,"_",lstrchr,"_",lendchr,".lme",sep=''),ncol=11,append=T)

print('Interpolated file')
print(system('date',T))

intdata<-rbind(intdata,t(intmat))
intdata<-data.frame(intdata)
# reorder
intdata<-intdata[order(intdata$yrfrac),]
# remove redundant filler lines and add lines to fill
intdata$co2int1[is.na(intdata$co2int1)]<- -999.99 # switch from NAs so na.omit below doesn't exclude
intdata$co2int2[is.na(intdata$co2int2)]<- -999.99
intdata$co2int3[is.na(intdata$co2int3)]<- -999.99
if(nlines==3){
	intdata$yrfrac[intdata$co2int1==-999.99&intdata$co2int2==-999.99&intdata$co2int3==-999.99]<-NA
}else if(nlines==5){
	intdata$co2int4[is.na(intdata$co2int4)]<- -999.99
	intdata$co2int5[is.na(intdata$co2int5)]<- -999.99
	intdata$yrfrac[intdata$co2int1==-999.99&intdata$co2int2==-999.99&intdata$co2int3==-999.99&intdata$co2int4==-999.99&intdata$co2int5==-999.99]<-NA
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
intdata<-na.omit(intdata)
intdata$yrfrac[duplicated(intdata$yrfrac)]<-NA # if same times from 2 files each have data, take the one from the first file
intdata<-na.omit(intdata)

# calculate datetime values
ye=trunc(intdata$yrfrac)
intdpy=rep(365,length(intdata$yrfrac))
intdpy[ye/4-trunc(ye/4)==0]=366
intdatetime=round.POSIXt(as.POSIXlt(strptime(paste(ye,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*intdpy*(intdata$yrfrac-ye)),units="mins")
# merge data onto complete 15-min intervals
dayfrac=as.numeric(as.POSIXct(intdatetime))/60/60/24
fulldayfrac=seq(trunc(dayfrac[1]),trunc(dayfrac[length(dayfrac)])+1,15/60/24) ## replace w/ seq.POSIXt?
fulldatetime=as.POSIXlt(fulldayfrac*60*60*24,origin='1970-01-01',tz="GMT")
fullyrdays=rep(365,length(fulldatetime$year))
fullyrdays[(fulldatetime$year+1900)/4-trunc((fulldatetime$year+1900)/4)==0]=366
fullyrfrac=data.frame(round(fulldatetime$year+1900+as.numeric(difftime(fulldatetime,strptime(paste(fulldatetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/fullyrdays,7))
colnames(fullyrfrac)='yrfrac'
intdata$yrfrac=round(intdata$yrfrac,7)
intdata=merge(fullyrfrac,intdata,by='yrfrac',all.x=T)
# generate text year, month, day, hour, and minute
ye=trunc(intdata$yrfrac)
intdpy=rep(365,length(intdata$yrfrac))
intdpy[ye/4-trunc(ye/4)==0]=366
intdatetime=round.POSIXt(as.POSIXlt(strptime(paste(ye,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*intdpy*(intdata$yrfrac-ye)),units="mins")
mo=intdatetime$mon+1
da=intdatetime$mday
ho=intdatetime$hour
mi=intdatetime$min
se=intdatetime$sec
intdata$co2int1[is.na(intdata$co2int1)]<- -999.99 # new NAs created in merge, replace here
intdata$co2int2[is.na(intdata$co2int2)]<- -999.99
intdata$co2int3[is.na(intdata$co2int3)]<- -999.99
if(nlines==5){
	intdata$co2int4[is.na(intdata$co2int4)]<- -999.99
	intdata$co2int5[is.na(intdata$co2int5)]<- -999.99
} # allow 2 or 4?

# spread heights across record # inlets read in above
inlets<-scan('../../aircoa_inlets.txt',what=list(loc='',unit='',date='',time='',L1=0,L2=0,L3=0,L4=0,L5=0,hz1=0,hz2=0,hz3=0,hz4=0,hz5=0,upl=0),skip=1) # read in again with date as text 
h1<-rep(-999.99,nrow(intdata))
h2<-rep(-999.99,nrow(intdata))
h3<-rep(-999.99,nrow(intdata))
h4<-rep(-999.99,nrow(intdata))
h5<-rep(-999.99,nrow(intdata))
for(i in c(1:length(inlets$loc))){
	if(inlets$loc[i]==loc){
		inletdatetime=strptime(paste(inlets$date,inlets$time),format='%y%m%d %H%M',tz="GMT")
		h1[difftime(intdatetime,inletdatetime[i],units='secs')>0]<-inlets$L1[i]
		h2[difftime(intdatetime,inletdatetime[i],units='secs')>0]<-inlets$L2[i]
		h3[difftime(intdatetime,inletdatetime[i],units='secs')>0]<-inlets$L3[i]
		if(nlines==5){
			h4[difftime(intdatetime,inletdatetime[i],units='secs')>0]<-inlets$L4[i]
			h5[difftime(intdatetime,inletdatetime[i],units='secs')>0]<-inlets$L5[i]
		} # allow 2 or 4?
	}
}
flags<-rep('PRE',nrow(intdata)) ### placeholder - need to read in dynamically
write("",paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''))
for(j in c(1:(length(fairuse)-1))){
	write(fairuse[j],paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
}
write(paste('This file written: ',system('date',T)),paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
write(fairuse[length(fairuse)],paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
if(nlines==3){
	write(c("YEAR.FRAC","YEAR","MON","DAY","HOUR","MIN","HGT1","HGT2","HGT3","CO21","CO22","CO23","FLAG"),paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),ncol=13,append=T)
	write("",paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
	write(rbind(intdata$yrfrac,ye,mo,da,ho,mi,h1,h2,h3,intdata$co2int1,intdata$co2int2,intdata$co2int3,flags),paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),ncol=13,append=T)
}else if(nlines==5){
	write(c("YEAR.FRAC","YEAR","MON","DAY","HOUR","MIN","HGT1","HGT2","HGT3","HGT4","HGT5","CO21","CO22","CO23","CO24","CO25","FLAG"),paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),ncol=17,append=T)
	write("",paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),append=T)
	write(rbind(intdata$yrfrac,ye,mo,da,ho,mi,h1,h2,h3,h4,h5,intdata$co2int1,intdata$co2int2,intdata$co2int3,intdata$co2int4,intdata$co2int5,flags),paste(loc,"_",lstrchr,"_",lendchr,".lin",sep=''),ncol=17,append=T)
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}

print('Hourly file')
print(system('date',T))

hrldata<-rbind(hrldata,t(hrlmat))
hrldata<-data.frame(hrldata)
# reorder
hrldata<-hrldata[order(hrldata$yrfrac),]
# remove redundant filler lines and add lines to fill
hrldata$co2hr1[is.na(hrldata$co2hr1)]<- -999.99 # switch from NAs so na.omit below doesn't exclude
hrldata$co2hr2[is.na(hrldata$co2hr2)]<- -999.99
hrldata$co2hr3[is.na(hrldata$co2hr3)]<- -999.99
hrldata$co2sd1[is.na(hrldata$co2sd1)]<- -999.99 
hrldata$co2sd2[is.na(hrldata$co2sd2)]<- -999.99
hrldata$co2sd3[is.na(hrldata$co2sd3)]<- -999.99
hrldata$co2dif[is.na(hrldata$co2dif)]<- -999.99
if(nlines==3){
	hrldata$yrfrac[hrldata$co2hr1==-999.99&hrldata$co2hr2==-999.99&hrldata$co2hr3==-999.99]<-NA ### *** skip this step to keep all hours in output?
}else if(nlines==5){
	hrldata$co2hr4[is.na(hrldata$co2hr4)]<- -999.99
	hrldata$co2hr5[is.na(hrldata$co2hr5)]<- -999.99
	hrldata$co2sd4[is.na(hrldata$co2sd4)]<- -999.99
	hrldata$co2sd5[is.na(hrldata$co2sd5)]<- -999.99
	hrldata$yrfrac[hrldata$co2hr1==-999.99&hrldata$co2hr2==-999.99&hrldata$co2hr3==-999.99&hrldata$co2hr4==-999.99&hrldata$co2hr5==-999.99]<-NA ### *** skip this step to keep all hours in output?
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
hrldata<-na.omit(hrldata)
hrldata$yrfrac[duplicated(hrldata$yrfrac)]<-NA # if same times from 2 files each have data, take the one from the first file
hrldata<-na.omit(hrldata)

# calculate datetime values
ye=trunc(hrldata$yrfrac)
hrldpy=rep(365,length(hrldata$yrfrac))
hrldpy[ye/4-trunc(ye/4)==0]=366
hrldatetime=round.POSIXt(as.POSIXlt(strptime(paste(ye,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*hrldpy*(hrldata$yrfrac-ye)),units="mins") # on half hours
# merge data onto complete hours
dayfrac=as.numeric(as.POSIXct(hrldatetime))/60/60/24
fulldayfrac=seq(trunc(dayfrac[1])+0.5/24,trunc(dayfrac[length(dayfrac)])+23.5/24,1/24) ## replace w/ seq.POSIXt? # on half hours
fulldatetime=as.POSIXlt(fulldayfrac*60*60*24,origin='1970-01-01',tz="GMT")
fullyrdays=rep(365,length(fulldatetime$year))
fullyrdays[(fulldatetime$year+1900)/4-trunc((fulldatetime$year+1900)/4)==0]=366
fullyrfrac=data.frame(round(fulldatetime$year+1900+as.numeric(difftime(fulldatetime,strptime(paste(fulldatetime$year+1900,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT"),units='days'))/fullyrdays,7))
colnames(fullyrfrac)='yrfrac'
hrldata$yrfrac=round(hrldata$yrfrac,7) # on half hours
hrldata=merge(fullyrfrac,hrldata,by='yrfrac',all.x=T)
# generate text year, month, day, hour, and minute
ye=trunc(hrldata$yrfrac)
hrldpy=rep(365,length(hrldata$yrfrac))
hrldpy[ye/4-trunc(ye/4)==0]=366
hrldatetime=round.POSIXt(as.POSIXlt(strptime(paste(ye,'-01-01 00:00:00',sep=''),format='%Y-%m-%d %H:%M:%S',tz="GMT")+60*60*24*hrldpy*(hrldata$yrfrac-ye)),units="mins") # on half hours
mo=hrldatetime$mon+1
da=hrldatetime$mday
ho=hrldatetime$hour
mi=hrldatetime$min
se=hrldatetime$sec

hrldata$co2hr1[is.na(hrldata$co2hr1)]<- -999.99 # new NAs created in merge, replace here
hrldata$co2hr2[is.na(hrldata$co2hr2)]<- -999.99
hrldata$co2hr3[is.na(hrldata$co2hr3)]<- -999.99
hrldata$co2sd1[is.na(hrldata$co2sd1)]<- -999.99 
hrldata$co2sd2[is.na(hrldata$co2sd2)]<- -999.99
hrldata$co2sd3[is.na(hrldata$co2sd3)]<- -999.99
hrldata$co2dif[is.na(hrldata$co2dif)]<- -999.99
if(nlines==5){
	hrldata$co2hr4[is.na(hrldata$co2hr4)]<- -999.99
	hrldata$co2hr5[is.na(hrldata$co2hr5)]<- -999.99
	hrldata$co2sd4[is.na(hrldata$co2sd4)]<- -999.99
	hrldata$co2sd5[is.na(hrldata$co2sd5)]<- -999.99
} # allow 2 or 4?

# spread heights across record 
inlets<-scan('../../aircoa_inlets.txt',what=list(loc='',unit='',date='',time='',L1=0,L2=0,L3=0,L4=0,L5=0,hz1=0,hz2=0,hz3=0,hz4=0,hz5=0,upl=0),skip=1) # read in again with date as text 
h1<-rep(-999.99,nrow(hrldata))
h2<-rep(-999.99,nrow(hrldata))
h3<-rep(-999.99,nrow(hrldata))
h4<-rep(-999.99,nrow(hrldata))
h5<-rep(-999.99,nrow(hrldata))
for(i in c(1:length(inlets$loc))){
	if(inlets$loc[i]==loc){
		inletdatetime=strptime(paste(inlets$date,inlets$time),format='%y%m%d %H%M',tz="GMT")
		h1[difftime(hrldatetime,inletdatetime[i],units='secs')>0]<-inlets$L1[i]
		h2[difftime(hrldatetime,inletdatetime[i],units='secs')>0]<-inlets$L2[i]
		h3[difftime(hrldatetime,inletdatetime[i],units='secs')>0]<-inlets$L3[i]
		if(nlines==5){
			h4[difftime(hrldatetime,inletdatetime[i],units='secs')>0]<-inlets$L4[i]
			h5[difftime(hrldatetime,inletdatetime[i],units='secs')>0]<-inlets$L5[i]
		} # allow 2 or 4?
	}
}
flags<-rep('PRE',nrow(hrldata)) ### placeholder - need to read in dynamically
write("",paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''))
for(j in c(1:(length(fairuse)-1))){
	write(fairuse[j],paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
}
write(paste('This file written: ',system('date',T)),paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
write(fairuse[length(fairuse)],paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
write("",paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
if(nlines==3){
	write(c("YEAR.FRAC","YEAR","MON","DAY","HOUR(START)","HGT1","HGT2","HGT3","CO21","CO22","CO23","SIGMA1","SIGMA2","SIGMA3","CO2DIF","FLAG"),
		paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),ncol=16,append=T)
	write("",paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
	write(rbind(hrldata$yrfrac,ye,mo,da,ho,h1,h2,h3,hrldata$co2hr1,hrldata$co2hr2,hrldata$co2hr3,hrldata$co2sd1,hrldata$co2sd2,hrldata$co2sd3,hrldata$co2dif,flags),
		paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),ncol=16,append=T)
}else if(nlines==5){
	write(c("YEAR.FRAC","YEAR","MON","DAY","HOUR(START)","HGT1","HGT2","HGT3","HGT4","HGT5","CO21","CO22","CO23","CO24","CO25","SIGMA1","SIGMA2","SIGMA3","SIGMA4","SIGMA5","CO2DIF","FLAG"),
		paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),ncol=22,append=T)
	write("",paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),append=T)
	write(rbind(hrldata$yrfrac,ye,mo,da,ho,h1,h2,h3,h4,h5,hrldata$co2hr1,hrldata$co2hr2,hrldata$co2hr3,hrldata$co2hr4,hrldata$co2hr5,hrldata$co2sd1,hrldata$co2sd2,hrldata$co2sd3,hrldata$co2sd4,
		hrldata$co2sd5,hrldata$co2dif,flags),paste(loc,"_",lstrchr,"_",lendchr,".lhr",sep=''),ncol=22,append=T)
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}
### *** minute calculations not needed above


# PLOT COMPLETE TIMESERIES

print('Plotting Complete Timeseries')
print(system('date',T))

if(png){png(paste(loc,"_",lstrchr,"_",lendchr,"_longts.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",lstrchr,"_",lendchr,"_longts.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(1,1))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

hghts<-NULL
hghts<-rbind(hghts,c('Date','Line 1','Line 2','Line 3','Line 4','Line 5'))
inlets<-scan('../../aircoa_inlets.txt',what=list(loc='',unit='',date='',time='',L1=0,L2=0,L3=0,L4=0,L5=0,hz1=0,hz2=0,hz3=0,hz4=0,hz5=0,upl=0),skip=1) # read in again with date as text 
	# line heights (1-5), horizontal distance to tower, and # of units per line
inlets$dayf<-as.numeric(substr(inlets$time,1,2))/24+as.numeric(substr(inlets$time,3,4))/60/24

chk<-0
for(j in c(1:length(inlets$loc))){
        if((inlets$loc[j]==loc&(as.numeric(inlets$date[j])+inlets$dayf[j])>(as.numeric(lstrchr)+ico)&(as.numeric(inlets$date[j])+inlets$dayf[j])<(as.numeric(lendchr)+eco))|(inlets$loc[j]==loc&chk==0)){
		hghts<-rbind(hghts,c(inlets$date[j],paste(inlets$L1[j],'m'),paste(inlets$L2[j],'m'),paste(inlets$L3[j],'m'),paste(inlets$L4[j],'m'),paste(inlets$L5[j],'m')))
		chk<-1
	} 
}

print(system('date',T))

plot(compdata$yrfrac[compdata$ngas==1&compdata$nlin!=(nlines+1)&compdata$nlin!=0],compdata$co2final[compdata$ngas==1&compdata$nlin!=(nlines+1)&compdata$nlin!=0],type='n',ylab='CO2 (ppm)',
		xlab='Year',main=paste(loc,' ','AIRCOA ',unit,'  ',lstrchr,' to ',lendchr,'  Calibrated Timeseries'))
x<-compdata$yrfrac[compdata$nlin==1&compdata$ngas==1]
y<-compdata$co2final[compdata$nlin==1&compdata$ngas==1]
if(length(x)>1){
points(x,y,col=2,pch=16,cex=0.15)
}
x<-compdata$yrfrac[compdata$nlin==2&compdata$ngas==1]
y<-compdata$co2final[compdata$nlin==2&compdata$ngas==1]
if(length(x)>1){
points(x,y,col=3,pch=16,cex=0.15)
}
x<-compdata$yrfrac[compdata$nlin==3&compdata$ngas==1]
y<-compdata$co2final[compdata$nlin==3&compdata$ngas==1]
if(length(x)>1){
points(x,y,col=4,pch=16,cex=0.15)
}
if(nlines==5){
	x<-compdata$yrfrac[compdata$nlin==4&compdata$ngas==1]
	y<-compdata$co2final[compdata$nlin==4&compdata$ngas==1]
	if(length(x)>1){
	points(x,y,col=5,pch=16,cex=0.15)
	}
	x<-compdata$yrfrac[compdata$nlin==5&compdata$ngas==1]
	y<-compdata$co2final[compdata$nlin==5&compdata$ngas==1]
	if(length(x)>1){
	points(x,y,col=6,pch=16,cex=0.15)
	}
} # allow for 2 or 4?

print(system('date',T))

sel<-rep(F,5)
if(any(data$nlin==1&data$ngas==1)){sel[1]<-T}
if(any(data$nlin==2&data$ngas==1)){sel[2]<-T}
if(any(data$nlin==3&data$ngas==1)){sel[3]<-T}
if(nlines==5){
	if(any(data$nlin==4&data$ngas==1)){sel[4]<-T}
	if(any(data$nlin==5&data$ngas==1)){sel[5]<-T}
} # allow for 2 or 4?

if(nrow(hghts)==1){ # names only
legend(mean(compdata$yrfrac,na.rm=T),max(compdata$co2final[compdata$nlin>0&compdata$nlin!=(nlines+1)&compdata$ngas==1],na.rm=T),hghts[1,2:6][sel],pch=c(16,16,16,16,16),pt.cex=c(.3,.3,.3,.3,.3)[sel],
		col=c(2,3,4,5,6)[sel],ncol=2)
} else if(nrow(hghts)==2){ # names plus one row of heights
	legend(compdata$yrfrac[1]+diff(range(compdata$yrfrac,na.rm=T))*(.1),max(compdata$co2final[compdata$nlin>0&compdata$nlin!=(nlines+1)&compdata$ngas==1],na.rm=T),hghts[2,2:6][sel],
		pch=c(16,16,16,16,16),pt.cex=c(.3,.3,.3,.3,.3)[sel],col=c(2,3,4,5,6)[sel],ncol=2)
} else {
	for(j in c(2:nrow(hghts))){
		changes<-c(hghts[2:nrow(hghts),1],lendchr)
		if(as.numeric(changes[1])<as.numeric(lstrchr)){changes[1]<-lstrchr}
		legend(compdata$yrfrac[1]+diff(range(compdata$yrfrac,na.rm=T))*(.1+(j-2)*.25),max(compdata$co2final[compdata$nlin>0&compdata$nlin!=(nlines+1)&compdata$ngas==1],na.rm=T),hghts[j,2:6][sel],
			pch=c(16,16,16,16,16),pt.cex=c(.3,.3,.3,.3,.3)[sel],col=c(2,3,4,5,6)[sel],ncol=2,title=paste(changes[j-1],'to',changes[j]))
	}
}

dev.off()

} else {
	print('Skipping update of complete timeseries')
} # if(!skipupdate)

# PLOT DAILY TIMESERIES AND DAILY PROFILES

if(daily){

### include LT or plot separately?

print('Plotting Daily Timeseries and Profiles')
print(system('date',T))

dpm<-c(31,28,31,30,31,30,31,31,30,31,30,31)
year<-trunc(start/10000) #  *** can not handle multiple year runs
# correct for leap year
if(year/4-trunc(year/4)==0) dpm[2]<-29  # *** can not handle multiple year runs including a Feb. leap year
prvdays<-c(0,cumsum(dpm)[1:11])

prvdays<-prvdays+yrdays # convert from day of year to day since Dec. 31, 2004

for(i in c(floor(min(data$doyc,na.rm=T)):floor(max(data$doyc,na.rm=T)))) {
# could do #for(day in unique(as.character(trunc.POSIXt(datetimec,'days')))){

if(any(data$doyc>=i&data$doyc<i+1)&i!=0&i>prvdays[1]) {

# generate text month and day for filenames and figure labels
mon<-c(1:12)[min((i-prvdays)[i-prvdays>0],na.rm=T)==i-prvdays]
if(mon<10) {mnt<-paste('0',as.character(mon),sep='')} else { mnt<-as.character(mon)}
day<-i-prvdays[mon]
if(day<10) {dyt<-paste('0',as.character(day),sep='')} else { dyt<-as.character(day)}
# could do #mnt=substring(day,6,7)
# could do #yrt=substring(day,3,4)
# could do #dyt=substring(day,9,10)

# Timeseries
if(png){png(paste(loc,"_",yrt,mnt,dyt,substr(outflag,2,2),"_timeseries.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",yrt,mnt,dyt,substr(outflag,2,2),"_timeseries.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(2,1))
par(oma=c(0,0,2,0))
par(mar=c(3,3,2,1)+.1)
par(mgp=c(2,1,0))

if(any(data$doyc>=i&data$doyc<i+1)){

plot(datetimec[data$doyc>=i&data$doyc<i+1],data$co2final[data$doyc>=i&data$doyc<i+1],type='n',ylab='CO2 (ppm)',xlab='Time (UTC)')
points(datetimec[data$ngas>1&data$doyc>=i&data$doyc<i+1],data$co2final[data$ngas>1&data$doyc>i&data$doyc<i+1])
lines(datetimec[data$nlin==1&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==1&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=2)
lines(datetimec[data$nlin==2&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==2&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=3)
lines(datetimec[data$nlin==3&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==3&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=4)
if(nlines==5){
	lines(datetimec[data$nlin==4&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==4&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=5)
	lines(datetimec[data$nlin==5&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==5&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=6)
} # allow 2 or 4?

mtext(paste(loc,' ','AIRCOA ',unit,'  Day ',yrt,mnt,dyt,'  Calibrated Timeseries',sep=''),3,0.5,T)

}

if(any(data$doyc>=i&data$doyc<i+1&data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0)){

plot(datetimec[data$doyc>=i&data$doyc<i+1&data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0],data$co2final[data$doyc>=i&data$doyc<i+1&data$ngas==1&data$nlin!=(nlines+1)&data$nlin!=0],type='n',ylab='CO2 (ppm)'
	,xlab='Time (UTC)')
lines(datetimec[data$nlin==1&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==1&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=2)
lines(datetimec[data$nlin==2&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==2&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=3)
lines(datetimec[data$nlin==3&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==3&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=4)
if(nlines==5){
	lines(datetimec[data$nlin==4&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==4&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=5)
	lines(datetimec[data$nlin==5&data$ngas==1&data$doyc>=i&data$doyc<i+1],data$co2final[data$nlin==5&data$ngas==1&data$doyc>=i&data$doyc<i+1],col=6)
} # allow 2 or 4?

sel<-rep(F,5)
if(any(data$nlin==1&data$ngas==1)){sel[1]<-T}
if(any(data$nlin==2&data$ngas==1)){sel[2]<-T}
if(any(data$nlin==3&data$ngas==1)){sel[3]<-T}
if(nlines==5){
	if(any(data$nlin==4&data$ngas==1)){sel[4]<-T}
	if(any(data$nlin==5&data$ngas==1)){sel[5]<-T}
} # allow 2 or 4?

legend(mean(datetimec[data$ngas>1&data$doyc>=i&data$doyc<i+1],na.rm=T),(par('usr')[4]-par('usr')[3])*.9+par('usr')[3],curhghts[sel],lty=c(1,1,1,1,1)[sel],col=c(2,3,4,5,6)[sel],ncol=2)

}

dev.off()



# Profiles
if(png){png(paste(loc,"_",yrt,mnt,dyt,substr(outflag,2,2),"_profiles.png",sep=''),width=950,height=550)}else{
bitmap(paste(loc,"_",yrt,mnt,dyt,substr(outflag,2,2),"_profiles.png",sep=''),width=9.36,height=5.4,res=100,type='png16',pointsize=10)}

par(mfrow=c(2,6))
par(mar=c(3,3,2,1)+.1) #defaults 5,4,4,2 (bot,lef,top,rig)
par(mgp=c(2,1,0)) #defaults 3,1,0
par(oma=c(0,0,2,0))

profile<-NULL
for(j in c(0:11)){
	sub<-interp$doyc>=(i+j*2/24)&interp$doyc<(i+(j+1)*2/24)
	profile<-rbind(profile,mean(data.frame(t(intmat)[sub,]))[2:(nlines+1)])
}
heights<-c(mean(data$height[data$doyc>=i&data$doyc<(i+1)&data$nlin==1&data$ngas==1]),mean(data$height[data$doyc>=i&data$doyc<(i+1)&data$nlin==2&data$ngas==1]),
mean(data$height[data$doyc>=i&data$doyc<(i+1)&data$nlin==3&data$ngas==1])) # give error for HDP with no line 1?
if(nlines==5){
	heights<-c(heights,mean(data$height[data$doyc>=i&data$doyc<(i+1)&data$nlin==4&data$ngas==1]),mean(data$height[data$doyc>=i&data$doyc<(i+1)&data$nlin==5&data$ngas==1]))
} # allow 2 or 4?
titles<-c('0000-0200 UTC','0200-0400 UTC','0400-0600 UTC','0600-0800 UTC','0800-1000 UTC','1000-1200 UTC','1200-1400 UTC','1400-1600 UTC','1600-1800 UTC','1800-2000 UTC','2000-2200 UTC','2200-2400 UTC')

for(j in c(1:12)){

if(any(!is.na(profile[j,]))) {
plot(profile[j,],heights,type='n',xlab='CO2 (ppm)',ylab='Height (m)',xlim=c(min(profile,na.rm=T),max(profile,na.rm=T)),main=titles[j])
lines(profile[j,][order(heights)],heights[order(heights)],type='b',col='Red')
mtext(paste(loc,' ','AIRCOA ',unit,'  Day ',yrt,mnt,dyt,'  Calibrated Profiles',sep=''),3,0.5,T)
} else {
if(j<12) {
if(j>=6) { 
rw<-2
cl<-j+1-6 
} else { 
rw<-1
cl<-j+1
}
par(mfg=c(rw,cl))
}
}
}

dev.off()

}

}

} # if(daily)

caldump<<-caldump
dump('caldump',paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".dmp",sep=''))

mesmat<-cbind(c("Year.frac","CO2(ppm)","InletLine","InletHeight(m)","CellPressure(kPa)","BoxTemperature(C)","StdDevCO2(ppm)","SampleRange(sec)"),mesmat) # binding text somehow ensures 7 digits in yrfrac output
write(mesmat,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".mes",sep=''),ncol=8)

if(nlines==3){
	intmat<-cbind(c("Year.frac","Line1CO2(ppm)","Line2CO2(ppm)","Line3CO2(ppm)","CellPressure(kPa)","BoxTemperature(C)"),intmat) # binding text somehow ensures 7 digits in yrfrac output
	write(intmat,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".int",sep=''),ncol=6)
}else if(nlines==5){
	intmat<-cbind(c("Year.frac","Line1CO2(ppm)","Line2CO2(ppm)","Line3CO2(ppm)","Line4CO2(ppm)","Line5CO2(ppm)","CellPressure(kPa)","BoxTemperature(C)"),intmat)
	write(intmat,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".int",sep=''),ncol=8)
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}

if(nlines==3){
	hrlmat<-cbind(c("Year.frac","Hour(start)","Line1CO2(ppm)","Line2CO2(ppm)","Line3CO2(ppm)","Line1Sigma(ppm)","Line2Sigma(ppm)","Line3Sigma(ppm)","CO2diff(ppm)"),hrlmat) # binding text somehow ensures 7 digits in yrfrac output
	write(hrlmat,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".hrl",sep=''),ncol=9)
}else if(nlines==5){
	hrlmat<-cbind(c("Year.frac","Hour(start)","Line1CO2(ppm)","Line2CO2(ppm)","Line3CO2(ppm)","Line4CO2(ppm)","Line5CO2(ppm)","Line1Sigma(ppm)","Line2Sigma(ppm)","Line3Sigma(ppm)","Line4Sigma(ppm)",
		"Line5Sigma(ppm)","CO2diff(ppm)"),hrlmat)
	write(hrlmat,paste(substr(startchr,1,4),'/',loc,'/',loc,"_",startchr,"_",endchr,outflag,".hrl",sep=''),ncol=13)
}else{
	print('INVALID NUMBER OF INLETS (NLINES)') ### allow 2 or 4?
}

} # do not run if no raw_day output

} # do not run if no raw_day output files
else {print('No raw_day output files found for period')}

print('aircoa_conc_days ending')
print(system('date',T))

if(logfile){
# end output and error logging:
sink(type="message")
sink()
close(log)
}

print(paste(system('date',T),paste('CONCDAYS ending',loc,startchr,endchr)))

system('chmod -f g+w *.*')
system(paste('chmod -f g+w',substr(startchr,1,4),'/',loc,'/*.*'))

}
