//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <sstream>
#include <unistd.h>
#include <time.h>
#include <assert.h>

#include <atdUtil/Socket.h>
#include <atdUtil/UnixSocketAddress.h>
#include <atdUtil/McSocket.h>
#include <atdUtil/Thread.h>

#include <vector>

using namespace atdUtil;
using namespace std;


static char* argvhost = "localhost";
static int ntests = 10;
static int ntestsOK = 0;

class Reader: public Thread
{
public:
    Reader(const string& name,Socket* s) : Thread(name),sock(s)
    {
    }

    ~Reader() {
    }

    int run() throw(atdUtil::Exception)
    {
	cerr << "Reader::run, local=" << sock->getLocalSocketAddress().toString() <<
		" remote=" << sock->getRemoteSocketAddress().toString() << endl;
	bool ok = true;
	char buf[512];
	const char* eob = buf + sizeof(buf);
	char* bp = buf;
	const char* eod;
	const char* cp;
	for (int i = 0; ;) {
	    int l;
	    try {
		l = sock->recv(bp,eob-bp,0);
	    }
	    catch(const EOFException& e) {
	        break;
	    }
	    // cerr << "recv l=" << l << " buf=" << buf << endl;
	    if (!l) break;
	    eod = bp + l;
	    int lout;
	    for (cp = bp; cp < eod; cp += lout,i++) {
		ostringstream ost;
		ost << "quack " << i;
		string out = ost.str();
		lout = out.length() + 1;
		if (out.compare(cp)) {
		    cerr << getName() << " test failed, l=" << l <<
			    " cp=" << cp << " strlen=" << strlen(cp) << endl;
		    cerr << "out=" << out << endl;
		    ok = false;
		}
	    }
	    for (bp = buf ; cp < eod; ) *bp++ = *cp++;
	}
	sock->close();
	delete sock;
	if (ok) ntestsOK++;
	else cerr << "reader " << getName() << " test failed" << endl;
	return RUN_OK;
    }
private:
    Socket* sock;
};

class Writer: public Thread
{
public:
    Writer(const string& name, const SocketAddress& addr):
    	Thread(name), addrptr(addr.clone()) {}

    ~Writer() { delete addrptr; }

    int run() throw(atdUtil::Exception)
    {
	Socket sock(*addrptr);
	cerr << "Writer::run, local=" << sock.getLocalSocketAddress().toString() <<
		" remote=" << sock.getRemoteSocketAddress().toString() << endl;
	struct timespec slp;
	slp.tv_sec = 0;
	slp.tv_nsec = 10000000;
	for (int i = 0; i < 100; i++) {
	    ostringstream ost;
	    ost << "quack " << i;
	    string out = ost.str();
	    int lout = out.length() + 1;
	    int l = sock.send(out.c_str(),lout,0);
	    if (l != lout) cerr << "send l=" << l << endl;
	    nanosleep(&slp,0);
	}
	sock.close();
	return RUN_OK;
    }
private:
    SocketAddress* addrptr;
};

class SocketServer: public Thread
{
public:
    SocketServer(const SocketAddress& addr) : Thread("server"),
    	ssock(addr,10)
    {
    }
    ~SocketServer() {
        ssock.close();
	for (unsigned int i = 0; i < readers.size(); i++) {
	    readers[i]->join();
	    delete readers[i];
	}
    }

    int run() throw(atdUtil::Exception)
    {
	for (int i = 0;; i++) {
	    Socket* sock = ssock.accept();
	    ostringstream ost;
	    ost << sock->getRemoteSocketAddress().toString() << "_" << i << "_reader";
	    Reader* reader = new Reader(ost.str(),sock);
	    reader->start();
	    readers.push_back(reader);
	}
	return RUN_OK;
    }
private:
    ServerSocket ssock;
    vector<Reader*> readers;
};

int stream_test()
{
    ntests = 10;
    ntestsOK = 0;

    SocketServer* server;
    Inet4SocketAddress addr(Inet4Address::getByName(argvhost),9000);
    try {
	server = new SocketServer(addr);
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
	return 0;
    }
    try {
	server->start();
	vector<Writer*> writers;

	for (int i = 0; i < ntests; i++) {
	    ostringstream ost;
	    ost << addr.toString() << "_" << i << "_writer";
	    Writer* writer = new Writer(ost.str(),addr);
	    writer->start();
	    writers.push_back(writer);
	}

	for (unsigned int i = 0; i < writers.size(); i++) {
	    try {
		writers[i]->join();
	    }
	    catch (const Exception& e) {
		cerr << e.what() << endl;
	    }
	    delete writers[i];
	}
	cerr << "writers done, cancelling server" << endl;

	server->cancel();
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    try {
	server->join();
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    delete server;
    if (ntestsOK != ntests)
    	cerr << (ntests-ntestsOK) << " stream tests failed" << endl;
    else
    	cerr << "stream_test: all tests OK" << endl;
    return ntestsOK == ntests;
}

int unix_test()
{
    ntests = 10;
    ntestsOK = 0;

    SocketServer* server;
    UnixSocketAddress addr("test");
    try {
	server = new SocketServer(addr);
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
	return 0;
    }
    try {
	server->start();
	vector<Writer*> writers;

	for (int i = 0; i < ntests; i++) {
	    ostringstream ost;
	    ost << addr.toString() << "_" << i << "_writer";
	    Writer* writer = new Writer(ost.str(),addr);
	    writer->start();
	    writers.push_back(writer);
	}

	for (unsigned int i = 0; i < writers.size(); i++) {
	    try {
		writers[i]->join();
	    }
	    catch (const Exception& e) {
		cerr << e.what() << endl;
	    }
	    delete writers[i];
	}
	cerr << "writers done, cancelling server" << endl;

	server->cancel();
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    try {
	server->join();
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    delete server;
    if (ntestsOK != ntests)
    	cerr << (ntests-ntestsOK) << " unix tests failed" << endl;
    else
    	cerr << "unix_test: all tests OK" << endl;
    return ntestsOK == ntests;
}
class DatagramWriter: public Thread
{
public:
    DatagramWriter(DatagramSocket s,Inet4SocketAddress addr) :
    	Thread(string("writer to ") + addr.toString()), sock(s),toaddr(addr)
    {
    }
    ~DatagramWriter() {
    }

    int run() throw(atdUtil::Exception)
    {
	struct timespec slp;
	slp.tv_sec = 0;
	slp.tv_nsec = 10000000;
	for (int i = 0; i < 100; i++) {
	    ostringstream ost;
	    ost << "quack " << i;
	    string out = ost.str();
	    DatagramPacket dgram((char*)out.c_str(),out.length()+1,toaddr);
	    sock.send(dgram);
	    nanosleep(&slp,0);
	}
	sock.sendto("done",5,0,toaddr);
        sock.close();
	return RUN_OK;
    }
public:
    DatagramSocket sock;
    Inet4SocketAddress toaddr;
};

class DatagramReader: public Thread
{
public:
    DatagramReader(Inet4SocketAddress addr) :
    	Thread(string("reader of ") + addr.toString()),sock(addr)
    {
    }
    ~DatagramReader() {
    }

    int run() throw(atdUtil::Exception)
    {
	char buf[512];
	DatagramPacket dgram(buf,sizeof(buf));

	bool ok = true;

	for (int i = 0; ; i++) {
	    sock.receive(dgram);
/*
	    if (i == 0) {
		cerr << "receive datagram length=" << dgram.getLength() <<
		    " data=" << dgram.getData() << endl;
	        cerr << "from=" << dgram.getRemoteSocketAddress().toString() << endl;
	    }
*/
	    if (!strcmp(dgram.getData(),"done")) break;

	    int lout;
	    ostringstream ost;
	    ost << "quack " << i;
	    string out = ost.str();
	    lout = out.length() + 1;

	    if (dgram.getLength() != lout || out.compare(dgram.getData())) {
		cerr << getName() << " test failed, l=" << dgram.getLength() <<
			" data=" << dgram.getData() << " strlen=" << strlen(dgram.getData()) << endl;
		cerr << "out=" << out << endl;
		ok = false;
	    }
	}
	sock.close();
	if (ok) ntestsOK++;
	else cerr << "datagram reader " << getName() << " test failed" << endl;
	return RUN_OK;
    }
private:
    DatagramSocket sock;
};

int datagram_test()
{
    ntests = 10;
    ntestsOK = 0;

    try {
	vector<DatagramWriter*> writers;
	vector<DatagramReader*> readers;

	Inet4Address addr = Inet4Address::getByName(argvhost);

	for (int i = 0; i < ntests; i++) {
	    int nport = 9000 + i;
	    Inet4SocketAddress saddr(addr,nport);

	    DatagramReader* reader = new DatagramReader(saddr);
	    reader->start();
	    readers.push_back(reader);

	    DatagramWriter* writer = new DatagramWriter(DatagramSocket(),saddr);
	    writer->start();
	    writers.push_back(writer);
	}

	for (unsigned int i = 0; i < writers.size(); i++) {
	    try {
		writers[i]->join();
		readers[i]->join();
	    }
	    catch (const Exception& e) {
		cerr << e.what() << endl;
	    }
	    delete writers[i];
	    delete readers[i];
	}
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    if (ntestsOK != ntests)
    	cerr << (ntests-ntestsOK) << " tests failed in datagram test" << endl;
    else
    	cerr << "datagram test: all tests OK" << endl;
    return ntestsOK == ntests;
}

int multicast_test()
{
    ntests = 1;
    ntestsOK = 0;

    try {

	Inet4Address mcast = Inet4Address::getByName("239.0.0.1");
	cerr << "isMultiCastAddress()=" << mcast.isMultiCastAddress() << endl;
	int nport = 9000;

	MulticastSocket readsock(nport);
	readsock.joinGroup(mcast);

	Inet4SocketAddress maddr(mcast,nport);
	MulticastSocket sock;

	// if firewall, must be mindfull of the sending interface
	sock.setInterface(Inet4Address::getByName("127.0.0.1"));
	sock.setTimeToLive(1);
	
	DatagramWriter writer(sock,maddr);
	writer.start();

	char buf[512];
	bool ok = true;
	Inet4SocketAddress from;
	cerr << "starting read" << endl;

	for (int i = 0; ; i++) {
	    int l = readsock.recvfrom(buf,sizeof(buf),0,from);
	    // cerr << "recvfrom l=" << l << " buf=" << buf << " from=" << from.toString() << endl;
	    if (!strcmp(buf,"done")) break;
	    int lout;
	    ostringstream ost;
	    ost << "quack " << i;
	    string out = ost.str();
	    lout = out.length() + 1;
	    if (l != lout || out.compare(buf)) {
		cerr << readsock.getLocalSocketAddress().toString() << " test failed, l=" << l <<
			" buf=" << buf << " strlen=" << strlen(buf) << endl;
		cerr << "out=" << out << endl;
		ok = false;
	    }
	}
	readsock.close();
	if (ok) ntestsOK++;
	try {
	    writer.join();
	}
	catch (const Exception& e) {
	    cerr << e.what() << endl;
	}
    }
    catch(atdUtil::Exception& e) {
        cerr << e.what() << endl;
    }
    if (ntestsOK != ntests)
    	cerr << (ntests-ntestsOK) << " tests failed in multicast test" << endl;
    else
    	cerr << "multicast test: all tests OK" << endl;
    return ntestsOK == ntests;
}

int mcsocket_test(bool doMulticast)
{
    string addrstring;
    if (doMulticast) addrstring = "239.0.0.10";
    else addrstring = "127.0.0.1";

    Inet4Address mcastAddr = Inet4Address::getByName(addrstring);
    int mport = 10000;
    Inet4SocketAddress mcastSockAddr(mcastAddr,mport);

    Inet4Address anyAddr = Inet4Address::getByName("0.0.0.0");
    Inet4SocketAddress mcastSockAddr2(anyAddr,mport);

    int pport = 99;

    McSocket server;
    if (doMulticast)
	server.setInet4McastSocketAddress(mcastSockAddr);
    else
	server.setInet4McastSocketAddress(mcastSockAddr2);
    server.setRequestNumber(pport);

    McSocket client;
    client.setInet4McastSocketAddress(mcastSockAddr);
    client.setRequestNumber(pport);
    if (doMulticast)
    	client.setInterface(Inet4Address::getByName("127.0.0.1"));

    class McThread: public Thread {
    public:
	  McThread(McSocket& clnt):
	  	Thread("McThread"),client(clnt) {}
	int run() throw(Exception)
	{
	    Socket* socket = client.connect();
	    cerr << "client connected, socket=" << hex << (void*)socket << dec << endl;
	    char buf[16];

	    size_t l = socket->recv(buf,sizeof(buf));
	    buf[l] = '\0';
	    cerr << "client read, l=" << l << endl;
	    if (strcmp(buf,"hello\n"))
	        throw Exception("McThread socket read not as expected");
	    try {
		l = socket->recv(buf,sizeof(buf));
	    }
	    catch(const EOFException& e) {
		cerr << "client EOF, closing socket" << endl;
		socket->close();
		delete socket;
		return RUN_OK;
	    }
	    cerr << "client no EOF, closing socket" << endl;
	    socket->close();
	    delete socket;
	    throw Exception("McThread socket read not as expected");
	}
    private:
	McSocket& client;
    } cthread(client);

    cthread.start();

    Socket* socket = server.accept();
    cerr << "McSocket server accepted" << endl;
    socket->send("hello\n",7);

    cerr << "server closing socket" << endl;
    socket->close();
    delete socket;

    cerr << "joining cthread" << endl;
    cthread.join();

    server.close();
    client.close();

    assert(McSocketListener::check() == 0);
    sleep(1);
    return 1;
}

int main(int argc, char** argv)
{
    if (argc > 1)
	argvhost = argv[1];
    else
    	argvhost = "localhost";
    if (!stream_test()) return 1;
    if (!unix_test()) return 1;
    if (!datagram_test()) return 1;
    if (!multicast_test()) return 1;
    if (!mcsocket_test(true)) return 1;
    if (!mcsocket_test(false)) return 1;
    return 0;
}

