//--------------------------------------------
// This is the motor control code using Girdon's array classes
// simple test program to drive the motor of the hotfilm probe
// I/O lines of the Prometheus are used to set motor direction and drive
// the motor the I/O lines are used to detect when the probe is at
// the maximum point for CCW and CW. 
// The zero position of the probe is determined by a define number of
// steps from either max point......fix this!!!!
//
// Modifications:
// 22Feb06,jwm	- added struct for motor io lines
//		- added pass param decoder 'get_options'
//              - modified the 'zero position' to use 'center' position
//              - modified the move routine to assume integer 'steps'
//                NOTE this is yet to be firmed up and converted to degrees.
//                For now the divider circuit works on divide counts of 1024
//                resulting in ~.44 degrees per pulse.  NOTE that this
//                was working with an input drive voltage of 3.2
//                resulting in ~19hz pulses.....don't know if this'd
//                work with 5v input, and then we'd need the 4096
//                divisor tap on the interface.
//		- Ultimately will modify the 'zero position' to use
//		the distinct interrupter line.
//
// I/O Port definitions
// CORRECTED PER ACTUAL INTERFACE BOARD
// PortA,C=input	PORT-Physical-Pin-Numbers
// 			motor 1		motor 2		motor 3		
// Clock		C4-21		A4-5		A3-4	
// Left (CCW)		C6-23		A6-7		A1-2
// Center		C0-17		A0-1		A7-8
// Right (CW)		C2-19		A2-3		A5-6
//
// Motor on/off		B7-16		B3-12		B1-10
// Direction		B6-15		B2-11		B0-9
// Logic on/off	 	B4-13

// arguments get pass to the program to define if the probe needs to be
// moved to the zero position or moved to a new position. If new postion
// then the direction flag is also provided.
// If no arguments are passed then the program aborts
// argv[1]: position steps (sign determines direction

#include <dsc/PromDigitalIO.h>

#include <iostream>

#include <unistd.h>

#include <time.h>


using namespace std;

// Global Variables
int motor, Step;
int scopeProbePin=17;	// DEBUGGING toggle for the scope = C1, pin 18
int logiconoff=12;	// Universal control line for logic +5 feed to all
struct LineDefs {	// NOTE the line#'s are 1 less than physical pin#'s
	int clock;
	int ccwlimit;
	int center;
	int cwlimit;
	int onoff;
	int direction;
} ioPins[4] = {
	{ 0, 0, 0, 0, 0, 0},	// only for convenient indexing of the
	{20,22,16,18,15,14},	// motors 1-3
	{ 4, 6, 0, 2,11,10},
	{ 3, 1, 7, 5, 9, 8}
};

//--------------------------------------------
class Stepper
{
public:
    Stepper(int ioport, int irq);

    ~Stepper();

    void init() throw(atdUtil::IOException);
    int StepProbe(int step) throw(atdUtil::IOException);
    int ZeroProbe() throw(atdUtil::IOException);
    int MoveProbe(int step) throw(atdUtil::IOException);
    void close() throw(atdUtil::IOException);
    void testE() throw(atdUtil::IOException)
    {
        throw atdUtil::IOException("device","testE","big honkin error");
    }

private:

	dsc::PromDigitalOut out;
	dsc::PromDigitalIn in;
	atdUtil::BitArray outMask;
	atdUtil::BitArray inMask;
	atdUtil::BitArray outMotor;
};
   

Stepper::Stepper(int ioport, int irq) :
	out(ioport,irq),
	in(ioport,irq),
	outMask(out.getNumberOfOutputs()),
	inMask(in.getNumberOfInputs()),
	outMotor(out.getNumberOfOutputs())
{
}

Stepper::~Stepper()
{
    cerr << "~Stepper called" << endl;
}

//--------------------------------------------
// setup class arguments

void Stepper::init() throw(atdUtil::IOException)
{
	cerr << "out.open" << endl;
	out.open();
	cerr << "out.open done" << endl;

	cerr << "in.open" << endl;
	in.open();

	cerr << "out.getNumberOfOutputs" << endl;
	outMask.setBits(8,16,0xff);	// PORT-B
	out.reserveOutputs(outMask);

	inMask.setBits(0,8,0xff);	// PORT-A
	inMask.setBits(16,24,0xff);	// PORT-C
	in.reserveInputs(inMask);
	cerr << "init done" << endl;
}

void Stepper::close() throw(atdUtil::IOException)
{
    out.close();
    in.close();
}

// StepProbe: function increment the motor to the desired location
// input: the number of steps to make
// output: 0 -> ok, -1 -> max switch turned on

int Stepper::StepProbe(int Step) throw(atdUtil::IOException)
{
	int ic=0,icw=0,iccw=0;
//	struct timespec sleep2msec = { 0, 0 };
	
	cerr << "stepprobe" << endl;
	atdUtil::BitArray inMotor(out.getNumberOfOutputs());

	// turn logic and then motor Vcc on
	outMotor.setBit(logiconoff,true);
	out.write(outMask,outMotor);
	outMotor.setBit(ioPins[motor].onoff,true);
	out.write(outMask,outMotor);

	// check to see if motor last stopped in an active high state
	// on the encoder, if so then wait till state goes low to start
	// counting moves
// we won't need this with the 4060 counter, always starts low
//	do {
//		inMotor = in.read(inMask);
//	} while(inMotor.getBit(ioPins[motor].clock));
		
	// Move the probe to one clock pulse before the final postion.
	// This is done so that we will stop immediately upon the rising
	// edge of the last pulse rather than waiting for the succeeding
	// 'low-level' as is done in this loop....perhaps this can be 
	// improved and condensed into 1 loop at some point....
	// While moving we need to check the limit switches to ensure
	// we do not burn up the motor.
	for (int i = Step; i > 1;  --i ) {

		do {
			inMotor = in.read(inMask);		// read the input lines
//			cout << "stepprobe" << endl <<inMotor.toString() << endl;

			//  check to make sure we have not maxed out movement range
			// if so stop motor and return -1
			icw= inMotor.getBit(ioPins[motor].cwlimit);
			iccw=inMotor.getBit(ioPins[motor].ccwlimit); 
			if ( icw || iccw ) {
				// turn motor off
				outMotor.setBit(ioPins[motor].onoff,false);
				out.write(outMask,outMotor);
				outMotor.setBit(logiconoff,false);
				out.write(outMask,outMotor);
				cout << "LimitHit: cw="<<icw<<" ccw="<<iccw << endl;
				return -1;
			}
		//nanosleep(&sleep2msec,0);
		// wait for a clock signal to go high
		}  while(!(inMotor.getBit(ioPins[motor].clock)));	

		// now read the input lines and wait for the clock signal to go low
		do {
			inMotor = in.read(inMask);
			//nanosleep(&sleep2msec,0);
		} while(inMotor.getBit(ioPins[motor].clock));

	}
	
	// now move the probe one more postion but stop as
	// soon as the clock signal goes high
		do {
			inMotor = in.read(inMask);		// read the input lines
//			cout << "Stepprobe (last step):" << endl << inMotor.toString() << endl;

			// check to make sure we have not maxed out movement range
			// if so stop motor and return -1
			if ((inMotor.getBit(ioPins[motor].ccwlimit)) ||
			    (inMotor.getBit(ioPins[motor].cwlimit))) {

				// turn motor off
				outMotor.setBit(ioPins[motor].onoff,false);
				out.write(outMask,outMotor);
				outMotor.setBit(logiconoff,false);
				out.write(outMask,outMotor);
				return -1;
			}
//			sleep(1);
		} while(!(inMotor.getBit(ioPins[motor].clock)));	// wait for a clock signal to go high


	// turn motor off
	outMotor.setBit(ioPins[motor].onoff,false);
	out.write(outMask,outMotor);
	outMotor.setBit(logiconoff,false);
	out.write(outMask,outMotor);
	return 0;
}



//--------------------------------------------
// ZeroProbe:
// Function moves probe to the starting position which is defined
// as parallel with the sonic boom.
// The probe is moved in a CW direction until either the 'Center' signal
// is received in which case we're done; or until the CWstop signal is received
// and then the motor direction is changed to CCW and the probe is moved back
// until the 'Center' signal is found.
// The number of clock signals is determined
// by the returned pulses between 'CWstop' and 'Center'

int Stepper::ZeroProbe() throw(atdUtil::IOException)
{	
	atdUtil::BitArray inMotor(in.getNumberOfInputs());
	int iccw=0,icw=0,ic=0;

	cout << "ZeroProbe:" << endl;

	// set motor direction CW
	outMotor.setBit(ioPins[motor].direction,true);
	out.write(outMask,outMotor);

	// turn motor on
	outMotor.setBit(logiconoff,true);
	out.write(outMask,outMotor);
	outMotor.setBit(ioPins[motor].onoff,true);
	out.write(outMask,outMotor);

	// read the input lines and test until either the center
	// position or the CWStop is detected
	do {
		inMotor = in.read(inMask);
		ic=  inMotor.getBit(ioPins[motor].center);
		icw=inMotor.getBit(ioPins[motor].cwlimit); 
	}  while( !ic && !icw );

	// turn motor off
	outMotor.setBit(ioPins[motor].onoff,false);
	out.write(outMask,outMotor);
	outMotor.setBit(logiconoff,false);
	out.write(outMask,outMotor);

	if ( ic==1 && !icw ) {	// if hit center THAT's IT!
		cout << "Zero-Position centertriggered" << endl;
		return 1;
	}
	cout << "     CW EndLimit:" << endl;


	// now move the probe other direction to the Zero position
	// turn motor on

	outMotor.setBit(ioPins[motor].direction,false);
	out.write(outMask,outMotor);
	sleep(1);

	cout << "Move Probe back to Zero Position" << endl;
	outMotor.setBit(logiconoff,true);
	out.write(outMask,outMotor);
	outMotor.setBit(ioPins[motor].onoff,true);
	out.write(outMask,outMotor);

	// Read the input lines and test until the CenterSwitch is detected
	do {
		inMotor = in.read(inMask);		// read the input lines
		ic=  inMotor.getBit(ioPins[motor].center);
		iccw=inMotor.getBit(ioPins[motor].ccwlimit); 
	}  while( !ic && !iccw );

	// turn off motor
	outMotor.setBit(ioPins[motor].onoff,false);
	out.write(outMask,outMotor);
	outMotor.setBit(logiconoff,false);
	out.write(outMask,outMotor);

	cout << "Zero-Position; centertrigger=" << ic << " ccwlimit=" << iccw << endl;

	return 1;
}




//--------------------------------------------
// MoveProbe:  function moves the probe based on input parameters
// Dir: "CW" or "CCW" (direction to move)
// Step; number of steps to move

int Stepper::MoveProbe(int Step) throw(atdUtil::IOException)
{

	int status;
	bool dir = true;

	// set the direction bit 
	if (Step < 0)
		dir = false;	// to CW

	// set motor direction
	outMotor.setBit(ioPins[motor].direction,dir);
	out.write(outMask,outMotor);

	// now move the probe forward to the new position position
	status = StepProbe(abs(Step));

	return status;
}


//--------------------------------------------
int main(int argc, char** argv)
{
    int get_options(int,char**);
    
    try {
	Stepper stepper(0x280,4);

	if (argc < 2) {
		cout << "No input parameters, program aborting!" << endl;
		return 0;
	}
	else
	{	motor=-1;Step=1000;	// set bogus
		if ( !get_options(argc,argv) )	return 0;
	}

	// initialize stuff
	stepper.init();

	//int Step = atoi(argv[1]);

	// where do we want the probe to move to?
	int status = 0;
	if(Step>=-360 && Step <=360) {
		// ok we need to move the probe
		if (Step == 0)
			status = stepper.ZeroProbe();		// move probe to starting position and quit
		else
			status = stepper.MoveProbe(Step);	// move probe to new position
	}
	stepper.close();
    }
    catch (const atdUtil::Exception& e) {
	cerr << e.what() << endl;
	return 1;
    }
    return 0;
}



//--------------------------------------------------------------------------
//	load pass parameters values and/or request them from the operator
//--------------------------------------------------------------------------
int get_options(int argc, char** argv)
{	int	i;
	char	*p;
	char	*strfind(char*,char*), *strfindb(char*,char*);/* lookup utilities */

	for (i=1; i<argc; ++i)		/* look for operator input... */
	{
		p = argv[i];

		if ( strfind(p,"-s") )
		{	++i;
			Step=atoi(argv[i]);
		}
		else if ( strfind(p,"-m") )
		{	++i;
			motor=atoi(argv[i]);
		}
		else if ( strfind(p,"-h") || strfind(p,"?") )
		{
	cout << "MotorStep Controller Usage: " << endl;
	cout << "\t-s x	CCW step count (-x=ClockWise rotation;  0=CenterPosition)" << endl;
	cout << "\t-m y	motor# 0=all,1-3 " << endl;
	return 0;
		}
	}
	cout << "Options: Step =" << Step << endl;
	cout << "         Motor=" << motor << endl;


	if ( motor<0 || motor>3 || Step<-360 || Step>360 ) {
		cout << "Unacceptable Results...Error Exit!" << endl;
		return 0;
	}
	return 2;
}



/*-------------------------------------------------------------------------
 * Function to FIND the specified string in the input line
 * Returns	NULL		if not found
 *	  	POINTER		to location in line immediately after where
 *				string is found.
 * Requires  both strings to be null terminated
 -------------------------------------------------------------------------*/
char* strfindb (char* line, char* str)
{	int i,ii;

	if( !line )	return (NULL);

	for ( i=0;  line[i] != (char)NULL ;  i++) {
		for ( ii=0;  line[i+ii] == str[ii];  )
			if (str[++ii]==(char)NULL) return(line+i+strlen(str));
	}
	return 0;
}



/*------------------------------------------------------------------------
 * Function to FIND the specified string in the input line
 * Returns	  NULL		if not found
 *		  POINTER	to location in line where string begins.
 * Requires  both strings to be null terminated
 -------------------------------------------------------------------------*/
char* strfind (char* line,char* str)
{	int i,ii;

	if( !line )	return (NULL);

	for ( i=0;  line[i] != (char)NULL ;  i++) {
		for ( ii=0;  line[i+ii] == str[ii];  )
			if (str[++ii]==(char)NULL) return(line+i);
	}
	return 0;
}




#ifdef OLD_METHOD
//--------------------------------------------
// ZeroProbe:  function moves probe to the starting position which is defined
// as parallel with the sonic boom
// The probe is sent in a CW direction until the CWstop signal is received
// then the motor direction is changed to CCW and the probe is moved backed
// a define number of clock signals. The number of clock signals is determined
// by the number of slots in the increment wheel,

int Stepper::ZeroProbe() throw(atdUtil::IOException)
{	
	int ZeroStep = 10;
	atdUtil::BitArray inMotor(in.getNumberOfInputs());

	cout << "ZeroProbe:" << endl;
	// set motor direction CW
	outMotor.setBit(ioPins[motor].direction,true);
	out.write(outMask,outMotor);

	// turn motor on
	outMotor.setBit(logiconoff,true);
	out.write(outMask,outMotor);
	outMotor.setBit(ioPins[motor].onoff,true);
	out.write(outMask,outMotor);

	// read the input lines and test until the CWStop is detected
	do {
		inMotor = in.read(inMask);
//		cout << "ZeroProbe:" <<  endl << inMotor.toString() << endl;
		sleep(1);
	} while(!(inMotor.getBit(ioPins[motor].cwlimit)));

	// turn motor off
	outMotor.setBit(logiconoff,false);
	out.write(outMask,outMotor);
	outMotor.setBit(ioPins[motor].onoff,false);
	out.write(outMask,outMotor);

	// set motor direction CCW
	outMotor.setBit(ioPins[motor].direction,false);
	out.write(outMask,outMotor);

	sleep(10);
	// now move the probe forward to the Zero position
	return StepProbe(ZeroStep);
}
#endif
