//
//              Copyright 2004 (C) by UCAR
//

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/select.h>
#include <signal.h>
#include <errno.h>

#include <atdTermio/SerialPort.h>
#include <atdTermio/SerialOptions.h>
#include <iostream>

/**
 * Simple program to read and write from stdin/stdout to a serial port.
 */

using namespace std;
using namespace atdUtil;
using namespace atdTermio;

struct termios stdin_termio;
bool stdin_changed = false;

bool outahere = false;

int setupStdin() throw(IOException) {
  struct termios term_io_new;
  /* get the termio characterstics for stdin */
  if (isatty(0)) {
    if (tcgetattr(0,&stdin_termio) < 0)
      throw IOException("stdin","tcgetattr",errno);
    /* copy and turn off local echo */
    memcpy( &term_io_new, &stdin_termio, sizeof stdin_termio);
    term_io_new.c_lflag &= ~ECHO & ~ICANON;

    term_io_new.c_iflag &= ~ICRNL & ~INLCR; 

    term_io_new.c_oflag &= ~OPOST;

    term_io_new.c_cc[VMIN] = 1;
    term_io_new.c_cc[VTIME] = 1;		// 1/10 second

    if (tcsetattr(0,TCSAFLUSH,&term_io_new) < 0)
      throw IOException("stdin","tcsetattr",errno);
    stdin_changed = true;
  }
  return 0;
}
void restoreStdin() throw(IOException) {
  if (isatty(0) && stdin_changed) {
    if (tcsetattr(0,TCSAFLUSH,&stdin_termio) < 0)
      throw IOException("stdin","tcsetattr",errno);
  }
}

/*
 * catch signals
 * 
 */
void signalCatcher(int s)
{
  const char *signame;
  switch (s) {
  case SIGHUP:
    signame = "hangup";
    break;
  case SIGINT:
    signame = "interrupt";
    break;
  case SIGTERM:
    signame = "terminate";
    break;
  default:
    signame = "";
    break;
  }
  fprintf(stderr,"received %s signal (#%d)\r\n",signame,s);
  outahere = true;
}

void setupSignals() {
    struct sigaction action;

    action.sa_handler = &signalCatcher;
    memset(&action.sa_mask,0,sizeof action.sa_mask);
    memset(&action.sa_flags,0,sizeof action.sa_flags);

    sigaction(SIGINT,&action,0);
    sigaction(SIGTERM,&action,0);
    sigaction(SIGHUP,&action,0);
}

void usage(const char* argv0) {
  cerr << "usage: " << argv0 << " tty_device options\n\
tty_device: /dev/ttySX for example\n\
options:" << SerialOptions::usage() << endl;
  exit(1);
}

int main(int argc, char**argv)
{
  if (argc == 1) usage(argv[0]);

  const char* ttyname = argv[1];
  SerialPort sp(ttyname);

  try {
    if (argc > 2) {
      const char* ttyoptions = argv[2];
      SerialOptions sopts;

      sopts.parse(ttyoptions);

      sp.setOptions(sopts);
#ifdef QUACK
      cerr << sopts.toString() << endl;
      sp.setBaudRate(sopts.getBaudRate());
      sp.setParity(sopts.getParity());
      sp.setDataBits(sopts.getDataBits());
      sp.setStopBits(sopts.getStopBits());
      sp.setLocal(sopts.getLocal());
      sp.setFlowControl(sopts.getFlowControl());
      sp.setRaw(sopts.getRaw());
      // turn CRNL->NL and NL->CRNL conversions back on
      if (!sopts.getRaw()) {
	sp.iflag() |= sopts.getNewlineIflag();
	sp.oflag() |= OPOST | sopts.getNewlineOflag();
      }
#endif

      sp.setBlocking(false);

      sp.open(O_RDWR | O_NOCTTY | O_NDELAY);

    }
    else {
      sp.setBaudRate(9600);
      sp.setParity(SerialPort::NONE);
      sp.setDataBits(8);
      sp.setStopBits(1);
      sp.setLocal(true);
      sp.setFlowControl(SerialPort::NOFLOWCONTROL);
      sp.setRaw(true);
      sp.setRawLength(1);
      sp.setBlocking(false);

      sp.open(O_RDWR | O_NOCTTY | O_NDELAY);

      // turn CRNL->NL and NL->CRNL conversions back on
      sp.iflag() |= ICRNL;
      sp.oflag() |= OPOST | ONLCR;
      sp.setTermioConfig();
    }


    setupStdin();
    setupSignals();

    fd_set rset;
    fd_set eset;
    FD_ZERO(&rset);
    FD_ZERO(&eset);

    FD_SET(sp.fd(),&rset);
    FD_SET(sp.fd(),&eset);

    FD_SET(0,&rset);
    FD_SET(0,&eset);

    char buf[512];
    int nread,nwrite;
    int nfd = sp.fd() + 1;	// highest numbered descriptor

    while(!outahere) {
      fd_set trset = rset;
      fd_set teset = eset;

      int nfdsel = ::select(nfd,&trset,0,&teset,0);
      if (nfdsel < 0)
        throw IOException(argv[0],"select",errno);

      if (!nfdsel) continue;

      if (FD_ISSET(0,&trset)) {	// characters on stdin
        if ((nread = read(0,buf,sizeof(buf))) < 0)
	  throw IOException("stdin","read",errno);
	/* check for zero-length read or ctrl-D=0x4 */
	outahere = (nread == 0) || (nread == 1 && buf[0] == '\x04');

	for (nwrite = 0; nwrite < nread; ) {
	  int l = sp.write(buf,nread);
	  if (l < 0)
	    throw IOException(sp.getName(),"write",errno);
	  nwrite += l;
	}
	if (!--nfdsel) continue;
      }
      if (FD_ISSET(sp.fd(),&trset)) {// characters on serial port
	nfdsel--;
	outahere = (nread = sp.read(buf,sizeof(buf))) == 0;

	for (nwrite = 0; nwrite < nread; ) {
	  int l = write(1,buf,nread);
	  if (l < 0)
	    throw IOException("stdout","write",errno);
	  nwrite += l;
	}
	if (!--nfdsel) continue;
      }
      if (FD_ISSET(0,&teset))  	// error on stdin
	  throw IOException("stdin","select","exception");
      if (FD_ISSET(sp.fd(),&teset)) 	// error on serial port
	  throw IOException(sp.getName(),"select","exception");
    }
    restoreStdin();
  }
  catch(IOException &ioe) {
    cerr << ioe.toString() << endl;
    try {
      restoreStdin();
    }
    catch(IOException &ioe2) {}
    return 1;
  }
  catch(ParseException &pe) {
    cerr << pe.toString() << endl;
    return 1;
  }
}

