//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <iomanip>
#include <map>
#include <string>
#include <unistd.h>
using namespace std;

#include <atdISFF/DaqMotor.h>

map<atdISFF::Stepper::StepperStatus, std::string> statusText;

typedef enum OurCommands {RESET, MOVE, ZERO};

int usage(const char *argv0) {
  cerr << "Usage: " << argv0 << " data_acq_hostname" << endl;
  exit(1);
}

//-------------------------------------------------------------------------
//
// Send a command. Return true if the command was accepted, false otherwise
// Try a number of times if the status is not idle, waiting a second between tries.
//
bool
send_command(atdISFF::DaqMotor& motor, 
	     int motorNumber,
	     OurCommands cmd,
	     double arg)
{
  atdISFF::Stepper::StepperStatus status = atdISFF::Stepper::STEPPERSTATUS_FAULT;  
  bool cmdStatus = false;
  double position;

  // try five times to send the command. This involves
  // first getting an idle status, then sending the command
  for (int i = 0; i < 5; i++) {
    if (motor.status(motorNumber, status, position)) {
      // First we must have an idle status
      if (status == atdISFF::Stepper::STEPPERSTATUS_IDLE) {
	switch (cmd) {
	case RESET: {
	  cmdStatus = motor.reset(motorNumber); 
	  break;
	}
	case ZERO: {
	  cmdStatus = motor.zero(motorNumber); 
	  break;
	}
	case MOVE: {
	  cmdStatus = motor.move(motorNumber, arg); 
	  break;
	}
	}
      } else {
	cerr << "Motor " << motorNumber << ", ";
	cerr << "Stepper is not idle; waiting for 1 second" << endl;
	sleep(1);
      }

      // now get out of the for-loop if the command was accepted
      if (cmdStatus)
	break;
    }  
  }
  // either all iterations failed, or one success
  return cmdStatus;

}

//-------------------------------------------------------------------------
//
// Send a command that has no argument. Return true if the command 
// was accepted, false otherwise.
//
void
send_command(atdISFF::DaqMotor& motor, 
	     int motorNumber, 
	     OurCommands cmd)
{
  if (!send_command(motor, motorNumber, cmd, 0.0)) {
    cerr << "Motor " << motorNumber << ", ";
    cerr << "command failed; resetting motor" << endl;
    sleep(1);
    motor.reset(motorNumber);
    sleep(1);
  }
}

//-------------------------------------------------------------------------
void
show_status(atdISFF::DaqMotor& motor, int motorNumber) {

  atdISFF::Stepper::StepperStatus status;
  double position;

  if (motor.status(motorNumber, status, position)) {
    cout << "Motor " << motorNumber << ", ";
    cout << "motor status is " << statusText[status] 
	 << ", position is " << position << endl;
  } else {
    cerr << "Motor " << motorNumber << ", ";
    cerr << "cannot get status from motor" << endl;
  }
}

//-------------------------------------------------------------------------
//
//
int main(int argc, char** argv) {

  if (argc != 2) usage(argv[0]);

  statusText[atdISFF::Stepper::STEPPERSTATUS_IDLE] = "idle";
  statusText[atdISFF::Stepper::STEPPERSTATUS_BUSY] = "busy";
  statusText[atdISFF::Stepper::STEPPERSTATUS_FAULT] = "fault";

  atdISFF::DaqMotor motor(argv[1]);

  for (int motorNumber = 3; motorNumber > 0; motorNumber--) {
    // The commands are repeated here since in the development
    // phase we don't even have a motor attached, and the
    // command will fail, and be followed by the reset
    // call in send_command()

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, 100.0);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, 100.0);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "zero motor" << endl;
    send_command(motor, motorNumber, ZERO);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "zero motor" << endl;
    send_command(motor, motorNumber, ZERO);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, 34.65);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, 34.65);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "zero motor" << endl;
    send_command(motor, motorNumber, ZERO);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "zero motor" << endl;
    send_command(motor, motorNumber, ZERO);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, -7.5);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "moving motor" << endl;
    send_command(motor, motorNumber, MOVE, -7.5);
    show_status(motor, motorNumber);

    cout << "Motor " << motorNumber << ", ";
    cout << "reset motor" << endl;
    send_command(motor, motorNumber, RESET);
    show_status(motor, motorNumber);
  
  }
  return 0;
}

