
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>

#include "emerald.h"


int printConfig(int fd,const char* devname) {
  emerald_config config;
  int i;

  if (ioctl(fd,EMERALD_IOCGPORTCONFIG,&config) < 0) {
    fprintf(stderr,"ioctl EMERALD_IOCGPORTCONFIG: %s: %s\n",devname,strerror(errno));
    return -1;
  }
  printf("RAM config:\n");
  for (i = 0; i < 8; i++) printf("ioport=0x%x,irq=%d\n",
    	config.ports[i].ioport,config.ports[i].irq);

  printf("EEPROM config:\n");
  if (ioctl(fd,EMERALD_IOCGEEPORTCONFIG,&config) < 0) {
    fprintf(stderr,"ioctl EMERALD_IOCGEEPORTCONFIG: %s: %s\n",devname,strerror(errno));
    return -1;
  }
  for (i = 0; i < 8; i++) printf("ioport=0x%x,irq=%d\n",
    	config.ports[i].ioport,config.ports[i].irq);
  return 0;

}
int setConfig(int fd,const char* devname,int write_eeprom,int port0Addr,const int* irqs) {
  int i;
  emerald_config config;

  /* printf("setting ports:\n"); */
  for (i = 0; i < 8; i++) {
    config.ports[i].ioport = port0Addr + i * 8;
    config.ports[i].irq = irqs[i];
    /*
    printf("ioport=0x%x,irq=%d\n",
      config.ports[i].ioport,config.ports[i].irq);
    */
  }

/* We shouldn't need to set the RAM config. The last step of
 * setting the EEPROM config now requests that the RAM settings
 * be read from EEPROM.
 */

  if (write_eeprom) {
    if (ioctl(fd,EMERALD_IOCSEEPORTCONFIG,&config) < 0) {
      fprintf(stderr,"ioctl EMERALD_IOCSEEPORTCONFIG: %s: %s\n",devname,strerror(errno));
      return -1;
    }
    if (ioctl(fd,EMERALD_IOCEECONFIGLOAD) < 0) {
      fprintf(stderr,"ioctl EMERALD_IOCEECONFIGLOAD: %s: %s\n",devname,strerror(errno));
      return -1;
    }
  }
  else {
    if (ioctl(fd,EMERALD_IOCSPORTCONFIG,&config) < 0) {
      fprintf(stderr,"ioctl EMERALD_IOCSPORTCONFIG: %s: %s\n",devname,strerror(errno));
      return -1;
    }
  }

  if (ioctl(fd,EMERALD_IOCPORTENABLE) < 0) {
    fprintf(stderr,"ioctl EMERALD_IOCPORTENABLE: %s: %s\n",devname,strerror(errno));
    return -1;
  }
  return 0;
}
int doEnable(int fd,const char* devname)
{
  if (ioctl(fd,EMERALD_IOCPORTENABLE) < 0) {
    fprintf(stderr,"ioctl EMERALD_IOCPORTENABLE: %s: %s\n",devname,strerror(errno));
    return -1;
  }
  printf("ports enabled\n");
  return 0;
}

int loadConfig(int fd,const char* devname)
{
  if (ioctl(fd,EMERALD_IOCEECONFIGLOAD) < 0) {
    fprintf(stderr,"ioctl EMERALD_IOCEECONFIGLOAD: %s: %s\n",devname,strerror(errno));
    return -1;
  }
  printf("config loaded from EEPROM to RAM\n");
  return 0;
}

void usage(const char* argv0)
{
  fprintf(stderr,"Usage: %s [-e] [-E] [-a port0addr] [-i irq ...] [-l] emerald_device\n\n",argv0);
  fprintf(stderr,"\
  -e:  enable ports only\n\
  -E: set ioport addresses and IRQs in EEPROM, else in volatile RAM\n\
  -a port0addr: port0addr is the ioport address of serial port 0\n\
    the 8 ports will be configured at port0addr, port0addr+0x8,\n\
    port0addr+0x10, ..., port0addr+0x38\n\
  -i irq: irq is the interrupt level to use for each port.\n\
    if less than 8 irqs are listed, the last will will be repeated\n\
  -l: load config from EEPROM into RAM\n\
\n\
For example, set ioport addresses to 0x100,0x108,0x110,0x118...,0x138\n\
and all irq=3. Values are set in EEPROM:\n\
  %s -E -a 0x100 -i 3 /var/tmp/emerald0\n",
  	argv0);
  exit(1);
}

int main(int argc, char** argv) {

  int port0Addr = -1;
  int irqs[8];
  int nirq = 0;
  int fd;
  const char* devname;
  int enable = 0;
  int write_eeprom = 0;
  int load_config = 0;
  int i;

  extern char *optarg;       /* set by getopt() */
  extern int optind;       /* "  "     "     */
  int opt_char;     /* option character */
  while ((opt_char = getopt(argc, argv, "a:eEi:l")) != -1) {
    switch(opt_char) {
    case 'a':
      if (sscanf(optarg,"0x%x",&port0Addr) != 1) usage(argv[0]);
      break;
    case 'e':
      enable = 1;
      break;
    case 'E':
      write_eeprom = 1;
      break;
    case 'i':
      if (sscanf(optarg,"%d",irqs + nirq++) != 1) usage(argv[0]);
      for ( ; optind < argc; optind++) {
	int l = strlen(argv[optind]);
        for (i = 0; i < l; i++) if (!isdigit(argv[optind][i])) break;
	if (i < l) break;
	if (sscanf(argv[optind],"%d",irqs + nirq++) != 1) usage(argv[0]);
      }
      break;
    case 'l':
      load_config = 1;
      break;
    }
  }

  for (i = nirq; i < 8; i++) irqs[i] = irqs[nirq-1];

  if (optind == argc) usage(argv[0]);
  devname = argv[optind++];

  fd = open(devname,port0Addr < 0 ? O_RDONLY : O_RDWR);
  if (fd < 0) {
    fprintf(stderr,"open %s: %s\n",devname,strerror(errno));
    return 1;
  }

  if (port0Addr >= 0) {
    if (printConfig(fd,devname) < 0) return 1;
    if (setConfig(fd,devname,write_eeprom,port0Addr,irqs) < 0) return 1;
    if (printConfig(fd,devname) < 0) return 1;
  }
  else if (enable && doEnable(fd,devname) < 0) return 1;
  else if (load_config && loadConfig(fd,devname) < 0) return 1;
  close(fd);
  return 0;
}
