
#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif
                                                                                
#include <linux/config.h>
#include <linux/module.h>
                                                                                
#include <linux/init.h>   /* module_init() */
#include <linux/kernel.h>   /* printk() */
#include <linux/slab.h>   /* kmalloc() */
#include <linux/fs.h>       /* everything... */
#include <linux/errno.h>    /* error codes */
#include <linux/types.h>    /* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>    /* O_ACCMODE */
#include <linux/ioport.h>
#include <asm/io.h>		/* outb, inb */
#include <asm/uaccess.h>	/* access_ok */
                                                                                
#include <asm/system.h>     /* cli(), *_flags */
                                                                                
#include "emerald.h"          /* local definitions */

static int emerald_major =   EMERALD_MAJOR;
static unsigned long emerald_ioports[EMERALD_MAX_NR_DEVS] = {0,0,0,0};
static int emerald_nr_devs = 0;
static emerald_device* emerald_devices = 0;
                                                                                
MODULE_PARM(emerald_major,"i");
MODULE_PARM(emerald_ioports,"1-4l");	/* io port virtual address */
MODULE_AUTHOR("Gordon Maclean");
MODULE_LICENSE("GPL");

#ifdef CONFIG_DEVFS_FS
devfs_handle_t emerald_devfs_dir = 0;
static char devname[4];
#endif

/* 
 * To detect if compiling for VIPER
 * check ifdef __LINUX_ARM_ARCH
 * or CONFIG_ARCH_VIPER
 */

/*
 * return 0 if config is not OK.
 * According to the Emerald-MM-8 User Manual, v2.4,
 * valid port addresses are 0x100-0x3f8.
 * Valid irqs are 2,3,4,5,6,7,10,11,12 and 15.
 * On the viper, ISA irqs 2 & 15 are not mapped.
 */
int emerald_check_config(emerald_config* config) {
  int i,j;
#ifdef CONFIG_ARCH_VIPER
  int valid_irqs[]={3,4,5,6,7,10,11,12};
  int nvalid = 8;
#else
  int valid_irqs[]={2,3,4,5,6,7,10,11,12,15};
  int nvalid = 10;
#endif

  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    PDEBUGG(KERN_INFO "emerald: ioport=%d\n",config->ports[i].ioport);
    if (config->ports[i].ioport > 0x3f8) return 0;
    if (config->ports[i].ioport < 0x100) return 0;
    for (j = 0; j < nvalid; j++) {
      PDEBUGG("emerald: checking irq=%d against %d\n",
      	config->ports[i].irq,valid_irqs[j]);
      if (valid_irqs[j] == config->ports[i].irq) break;
    }
    if (j == nvalid) return 0;
  }
  return 1;
}

void emerald_read_config(emerald_device* dev) {
  int i;
  /* read ioport values. irqs are not readable, except from eeprom */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(i,dev->ioport+EMERALD_APER);
    wmb();		/* flush */
    dev->config.ports[i].ioport = inb(dev->ioport+EMERALD_ARR) << 3;
    dev->config.ports[i].irq = 0;
  }
}

void emerald_write_config(emerald_device* dev,emerald_config* config) {
  int i;
  /* write ioport and irq values. */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(i,dev->ioport+EMERALD_APER);
    wmb();		/* flush */
    outb(config->ports[i].ioport >> 3,dev->ioport+EMERALD_AIDR);
    wmb();		/* flush */

    outb(i+EMERALD_NR_PORTS,dev->ioport+EMERALD_APER);
    wmb();		/* flush */
    outb(config->ports[i].irq,dev->ioport+EMERALD_AIDR);
    wmb();		/* flush */
  }
  dev->config = *config;
}

/*
 * Read the ioport and irq configuration for the 8 serial ports
 * from the Emerald EEPROM.
 */
void emerald_read_eeconfig(emerald_device* dev,emerald_config* config) {
  int i;
  unsigned char busy;

  /* another way to wait */
  /*
  set_current_state(TASK_INTERRUPTIBLE);
  schedule_timeout(1);
  */

  /* get ioport values from EEPROM addresses 0-7 */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(i,dev->ioport+EMERALD_ECAR);
    wmb();		/* flush */

    /* wait for busy bit in EMERALD_EBR to clear */
    do {
      unsigned long jwait = jiffies + 5;
      while (jiffies < jwait) schedule();
      busy = inb(dev->ioport+EMERALD_EBR);
    } while(busy & 0x80);

    config->ports[i].ioport = inb(dev->ioport+EMERALD_EDR) << 3;
  }

  /* get irq values from EEPROM addresses 8-15 */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(i+EMERALD_NR_PORTS,dev->ioport+EMERALD_ECAR);
    wmb();		/* flush */

    /* wait for busy bit in EMERALD_EBR to clear */
    do {
      unsigned long jwait = jiffies + 5;
      while (jiffies < jwait) schedule();
      busy = inb(dev->ioport+EMERALD_EBR);
    } while(busy & 0x80);

    config->ports[i].irq = inb(dev->ioport+EMERALD_EDR);
  }
}

/*
 * Write the ioport and irq configuration for the 8 serial ports
 * to the Emerald EEPROM.
 */
void emerald_write_eeconfig(emerald_device* dev,emerald_config* config) {
  int i;
  unsigned char busy;

  /* write ioport values to EEPROM addresses 0-7 */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(config->ports[i].ioport >> 3,dev->ioport+EMERALD_EDR);
    wmb();		/* flush */
    outb(i + 0x80,dev->ioport+EMERALD_ECAR);
    wmb();		/* flush */

    /* wait for busy bit in EMERALD_EBR to clear */
    do {
      unsigned long jwait = jiffies + 5;
      while (jiffies < jwait) schedule();
      busy = inb(dev->ioport+EMERALD_EBR);
    } while(busy & 0x80);
  }

  /* write irq values to EEPROM addresses 8-15 */
  for (i = 0; i < EMERALD_NR_PORTS; i++) {
    outb(config->ports[i].irq,dev->ioport+EMERALD_EDR);
    wmb();		/* flush */
    outb(i + EMERALD_NR_PORTS + 0x80,dev->ioport+EMERALD_ECAR);
    wmb();		/* flush */

    /* wait for busy bit in EMERALD_EBR to clear */
    do {
      unsigned long jwait = jiffies + 5;
      while (jiffies < jwait) schedule();
      busy = inb(dev->ioport+EMERALD_EBR);
    } while(busy & 0x80);

  }
}

/*
 * Load the the ioport and irq configuration from the Emerald
 * EEPROM into ram.
 */
void emerald_load_config_from_eeprom(emerald_device* dev) {

  unsigned char busy;

  outb(0x80,dev->ioport+EMERALD_CRR);	/* reload configuration from eeprom */
  wmb();		/* flush */
  /* wait for busy bit in EMERALD_EBR to clear */
  do {
    unsigned long jwait = jiffies + 5;
    while (jiffies < jwait) schedule();
    busy = inb(dev->ioport+EMERALD_EBR);
  } while(busy & 0x80);

}

void emerald_enable_ports(emerald_device* dev) {
  outb(0x80,dev->ioport+EMERALD_APER);	/* enable ports */
}


#ifdef EMERALD_DEBUG /* use proc only if debugging */
/*
 * The proc filesystem: function to read
 */
                                                                                
int emerald_read_procmem(char *buf, char **start, off_t offset,
                   int count, int *eof, void *data)
{
    int i, j, len = 0;
    int limit = count - 80; /* Don't print more than this */
    PDEBUGG("read_proc, count=%d\n",count);
                                                                                
    for (i = 0; i < emerald_nr_devs && len <= limit; i++) {
        struct emerald_device *d = emerald_devices + i;
	PDEBUGG("read_proc, i=%d, device=0x%lx\n",i,(unsigned long)d);
        if (down_interruptible(&d->sem))
                return -ERESTARTSYS;
	emerald_read_config(d);
        len += sprintf(buf+len,"\nDiamond Emerald-MM-8 %i: ioport %lx\n",
                       i, d->ioport);
	/* loop over serial ports */
        for (j = 0; len <= limit && j < EMERALD_NR_PORTS; j++) {
            len += sprintf(buf+len, "  port %d, ioport=%x,irq=%d\n",
	    	j,d->config.ports[j].ioport,d->config.ports[j].irq);
        }
        up(&d->sem);
    }
    *eof = 1;
    return len;
}

static void emerald_create_proc()
{
    PDEBUGG("within emerald_create_proc\n");
    create_proc_read_entry("emerald", 0 /* default mode */,
                           NULL /* parent dir */, emerald_read_procmem,
                           NULL /* client data */);
}
                                                                                
static void emerald_remove_proc()
{
    /* no problem if it was not registered */
    remove_proc_entry("emerald", NULL /* parent dir */);
}

#endif

/*
 * The cleanup function is used to handle initialization failures as well.
 * Thefore, it must be careful to work correctly even if some of the items
 * have not been initialized
 */
void emerald_cleanup_module(void)
{
    int i;
                                                                                
#ifndef CONFIG_DEVFS_FS
    /* cleanup_module is never called if registering failed */
    unregister_chrdev(emerald_major, "emerald");
#endif
                                                                                
#ifdef EMERALD_DEBUG /* use proc only if debugging */
    emerald_remove_proc();
#endif
    if (emerald_devices) {
        for (i=0; i<emerald_nr_devs; i++) {
	    if (emerald_devices[i].region) 
		release_region(emerald_devices[i].ioport,EMERALD_IO_REGION_SIZE);
            /* the following line is only used for devfs */
            if (emerald_devices[i].handle) 
	    	devfs_unregister(emerald_devices[i].handle);
        }
        kfree(emerald_devices);
    }
#ifdef CONFIG_DEVFS_FS
    /* once again, only for devfs */
    devfs_unregister(emerald_devfs_dir);
#endif /* CONFIG_DEVFS_FS */
}

int emerald_init_module(void)
{
    int result, i;
                                                                                
#ifdef CONFIG_DEVFS_FS
    /* If we have devfs, create /dev/emerald to put files in there */
    emerald_devfs_dir = devfs_mk_dir(NULL, "emerald", NULL);
    if (!emerald_devfs_dir) return -EBUSY; /* problem */
                                                                                
#else /* no devfs, do it the "classic" way  */
                                                                                
    /*
     * Register your major, and accept a dynamic number. This is the
     * first thing to do, in order to avoid releasing other module's
     * fops in emerald_cleanup_module()
     */
    result = register_chrdev(emerald_major, "emerald", &emerald_fops);
    if (result < 0) {
        printk(KERN_WARNING "emerald: can't get major %d\n",emerald_major);
        return result;
    }
    if (emerald_major == 0) emerald_major = result; /* dynamic */
    PDEBUGG("major=%d\n",emerald_major);

#endif /* CONFIG_DEVFS_FS */

    for (i=0; i < EMERALD_MAX_NR_DEVS; i++)
      if (emerald_ioports[i] == 0) break;
    emerald_nr_devs = i;
    PDEBUGG("nr_devs=%d\n",emerald_nr_devs);

    /*
     * allocate the devices 
     */
    emerald_devices = kmalloc(emerald_nr_devs * sizeof(emerald_device), GFP_KERNEL);
    if (!emerald_devices) {
        result = -ENOMEM;
        goto fail;
    }
    memset(emerald_devices, 0, emerald_nr_devs * sizeof(emerald_device));
    for (i=0; i < emerald_nr_devs; i++) {
        emerald_devices[i].ioport = emerald_ioports[i];
	if (!( emerald_devices[i].region =
	    request_region(emerald_devices[i].ioport,EMERALD_IO_REGION_SIZE,
			"emerald"))) {
	    result = -ENODEV;
	    goto fail;
	}
	sema_init(&emerald_devices[i].sem,1);
#ifdef CONFIG_DEVFS_FS
        sprintf(devname, "%i", i);
        emerald_devices[i].handle =
		devfs_register(emerald_devfs_dir, devname,
                       DEVFS_FL_AUTO_DEVNUM,
                       0, 0, S_IFCHR | S_IRUGO | S_IWUGO,
                       &emerald_fops,
		       emerald_devices+i);	// private data
#endif
    }
                                                                                
    /* ... */
                                                                                
#ifndef EMERALD_DEBUG
    EXPORT_NO_SYMBOLS; /* otherwise, leave global symbols visible */
#endif
                                                                                
#ifdef EMERALD_DEBUG /* only when debugging */
    PDEBUGG("create_proc\n");
    emerald_create_proc();
#endif
                                                                                
    return 0; /* succeed */
                                                                                
  fail:
    emerald_cleanup_module();
    return result;
}

int emerald_open (struct inode *inode, struct file *filp)
{
    int num = MINOR(inode->i_rdev);
    emerald_device *dev; /* device information */

    /*  check the device number */
    if (num >= emerald_nr_devs) return -ENODEV;
    dev = emerald_devices + num;

    /* and use filp->private_data to point to the device data */
    filp->private_data = dev;

    if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
    emerald_read_config(dev);
    up(&dev->sem);

    MOD_INC_USE_COUNT;
    return 0;	/* success */
}

int emerald_release (struct inode *inode, struct file *filp)
{
    MOD_DEC_USE_COUNT;
    return 0;
}

/*
 * The ioctl() implementation
 */

int emerald_ioctl (struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{
    emerald_device* dev = filp->private_data;
    int err= 0, ret = 0;

    /* don't even decode wrong cmds: better returning  ENOTTY than EFAULT */
    if (_IOC_TYPE(cmd) != EMERALD_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd) > EMERALD_IOC_MAXNR) return -ENOTTY;

    /*
     * the type is a bitmask, and VERIFY_WRITE catches R/W
     * transfers. Note that the type is user-oriented, while
     * verify_area is kernel-oriented, so the concept of "read" and
     * "write" is reversed
     */
    if (_IOC_DIR(cmd) & _IOC_READ)
        err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err =  !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd));
    if (err) return -EFAULT;

    switch(cmd) {

      case EMERALD_IOCGIOPORT:
        ret = put_user(dev->ioport, (unsigned long *) arg);
        break;

      case EMERALD_IOCGPORTCONFIG:	/* get port config */
        if (copy_to_user((emerald_config *) arg,&dev->config,
	      	sizeof(emerald_config)) != 0) ret = -EFAULT;
        break;
        
      case EMERALD_IOCSPORTCONFIG:	/* set port config */
	{
	    emerald_config tmpconfig;
	    if (copy_from_user(&tmpconfig,(emerald_config *) arg,
	      	sizeof(emerald_config)) != 0) ret = -EFAULT;
	    if (!emerald_check_config(&tmpconfig)) ret = -EINVAL;
	    else {
		if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
	        emerald_write_config(dev,&tmpconfig);
		up(&dev->sem);
	    }
	}
        break;

      case EMERALD_IOCGEEPORTCONFIG:	/* get config from eeprom */
        {
	  emerald_config eeconfig;
	  if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
	  emerald_read_eeconfig(dev,&eeconfig);
	  up(&dev->sem);
	  if (copy_to_user((emerald_config *) arg,&eeconfig,
		  sizeof(emerald_config)) != 0) ret = -EFAULT;
	}
        break;
        
      case EMERALD_IOCSEEPORTCONFIG:	/* set config in eeprom */
	{
	    emerald_config eeconfig;
	    if (copy_from_user(&eeconfig,(emerald_config *) arg,
	      	sizeof(emerald_config)) != 0) ret = -EFAULT;
	    if (!emerald_check_config(&eeconfig)) ret = -EINVAL;
	    else {
		if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
	        emerald_write_eeconfig(dev,&eeconfig);
		up(&dev->sem);
	    }
	}
        break;

      case EMERALD_IOCEECONFIGLOAD:	/* load EEPROM config */
	{
	  if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
	  emerald_load_config_from_eeprom(dev);
	  up(&dev->sem);
	}
        break;
      case EMERALD_IOCPORTENABLE:
	{
	  if (down_interruptible(&dev->sem)) return -ERESTARTSYS;
	  emerald_enable_ports(dev);
	  up(&dev->sem);
	}
        break;

      default:  /* redundant, as cmd was checked against MAXNR */
        return -ENOTTY;
    }
    return ret;
}

struct file_operations emerald_fops = {
    /* llseek:     emerald_llseek, */
    /* read:       emerald_read, */
    /* write:      emerald_write, */
    ioctl:      emerald_ioctl,
    open:       emerald_open,
    release:    emerald_release,
};
						      
module_init(emerald_init_module);
module_exit(emerald_cleanup_module);
