//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <sstream>
#include <iomanip>
#include <math.h>
#include <unistd.h>	// swab

#include <dsc/DscUserClockedA2DSampler.h>
#include <dsc/DiamondUD.h>

#include <atdUtil/Logger.h>

using namespace dsc;
using namespace std;
using namespace atdUtil;
using namespace atdISFF;

dsc::UserClockedA2DSampler::UserClockedA2DSampler(int type,
	int ioPortAddr,int irq,TimerThread*tt):
  // virtual base class
  atdISFF::A2DSampler(DscBoard::makeName(type,ioPortAddr,irq)),
  atdISFF::UserClockedA2DSampler(DscBoard::makeName(type,ioPortAddr,irq),tt),
  dsc::A2DSampler(type,ioPortAddr,irq),
  maxRequestedSamplingRate(0.0)
{
  memset(&dscadscan,0,sizeof(dscadscan));

}

dsc::UserClockedA2DSampler::~UserClockedA2DSampler() {
  delete [] dscadscan.sample_values;
}

void dsc::UserClockedA2DSampler::addChannel(int userChannelNum, int vRangeIndex, float samplingRate) throw(IOException,InvalidParameterException) {

  dsc::A2DSampler::addChannel(userChannelNum,vRangeIndex);

  checkSamplingRate(samplingRate);

  requestedSamplingRates.push_back(samplingRate);
  decimateValueVec.push_back(1);

  for (unsigned int i = 0; i < decimateValueVec.size(); i++) {
    float srate = requestedSamplingRates[i];
    decimateValueVec[i] = (int)rint(maxRequestedSamplingRate / srate);
    // cerr << "decimateValues[" << i << "]=" << decimateValueVec[i] << endl;
  }

}

void dsc::UserClockedA2DSampler::checkSamplingRate(float sRate) throw(InvalidParameterException) {

  if (sRate > getMaximumSamplingRate()) {
    ostringstream os;
    os << "Sample rate=" << sRate << " Hz " <<
      " is greater than maximum allowed rate=" <<
      getMaximumSamplingRate() << " Hz";
    throw InvalidParameterException(getName(),"addChannel",os.str());
  }
  if (sRate > timer->getWakeupRate()) {
    ostringstream os;
    os << "Sample rate=" << sRate << " Hz " <<
      " is greater than wakeup rate of timer thread =" <<
      timer->getWakeupRate() << " Hz";
    throw InvalidParameterException(getName(),"addChannel",os.str());
  }

  float tmpMax = maxRequestedSamplingRate;  
  if (sRate > tmpMax) tmpMax = sRate;

  // check existing sample rates against the new maximum, and against
  for (unsigned int i = 0; i < requestedSamplingRates.size(); i++)
    checkSamplingRates(tmpMax,requestedSamplingRates[i]);

  // check this rate
  checkSamplingRates(tmpMax,sRate);
  // check against timer thread
  checkSamplingRates(timer->getWakeupRate(),sRate);
  maxRequestedSamplingRate = tmpMax;
}

/**
 * Check that rate divides (fairly) evenly into maxrate, else throw InvalidParameterException.
 */
void dsc::UserClockedA2DSampler::checkSamplingRates(float maxrate,float rate) throw(InvalidParameterException) {
  double ratio = maxrate / rate;
  if (fabs(ratio - rint(ratio)) > .001) {
    ostringstream os;
    os << "Sample rate=" << rate << " Hz (dt=" << 1.0/rate <<
      " sec) does not divide evenly into the maximum requested rate of " <<
      maxrate << " Hz (dt=" << 1.0/maxrate << " sec)" << 
      " remainder=" << fabs(ratio - rint(ratio));
    throw InvalidParameterException(getName(),"addChannel",os.str());
  }
}


void dsc::UserClockedA2DSampler::init() throw(IOException,InvalidParameterException)
{
    
  dsc::A2DSampler::init();

  deltatMsec = (int)(rint(1000.0f / maxRequestedSamplingRate));

  int timerMsec = timer->getWakeupIntervalMsec();
  scanEvery = 1;

  cerr << "deltatMsec=" << deltatMsec << " timerMsec=" << timerMsec << endl;

  if (abs(timerMsec - deltatMsec) > 2) {
    if (deltatMsec < timerMsec || (deltatMsec % timerMsec) > 2) {
      ostringstream os;
      os << "Sample deltaT=" << deltatMsec <<
      	" msec is not compatible with timer deltaT" << timerMsec << " msec";
      throw InvalidParameterException(getName(),"init",os.str());
    }
    scanEvery = (int)rint((float)deltatMsec / timerMsec);
  }

  dscadscan.low_channel = getMinChannelScanned();
  dscadscan.high_channel = getMinChannelScanned() + getNumChannelsScanned() - 1; 
  dscadscan.sample_values = new DSCSAMPLE[getNumChannelsScanned()];
}

int dsc::UserClockedA2DSampler::run() throw(atdUtil::Exception) {

  tlast = 0;
  int eps = deltatMsec / 10;
  if (eps == 0) eps = 1;
  tdiffcheck = deltatMsec + eps;

  int nwake = 0;

  for (;;) {
    if (amInterrupted() || timer->isInterrupted()) {
      Logger::getInstance()->log(LOG_INFO,"%s",
          "UserClockedA2DSampler::run interrupted");
      break;
    }
    timer->lock();
    timer->wait();
    if (amInterrupted() || timer->isInterrupted()) {
      Logger::getInstance()->log(LOG_INFO,"%s",
          "UserClockedA2DSampler::run timer interrupted");
      timer->unlock();
      break;
    }
    timer->unlock();

#ifdef DEBUG
    cerr << "A2D woken, nwake=" << nwake << " scanEvery=" << scanEvery << endl;
#endif

    if ((++nwake % scanEvery)) continue;
    nwake = 0;
    doConversions();
  }
  return RUN_OK;
}

void dsc::UserClockedA2DSampler::doScan() throw(IOException)
{
  const int len = sizeof(short);
  BYTE result;
  if ((result =
      dscADScan(getDSCB(),&dscadscan,dscadscan.sample_values)) != DE_NONE) {
    throw IOException(getName(),"dscADScan",dscGetErrorString(result));
  }

  isff_sys_time_t dtime = getCurrentTimeInMillis();

  if (dtime > tlast + tdiffcheck)
    missedSamples += (size_t)(tlast - dtime) / deltatMsec;

  sample_time_t tt = sample_time(dtime,0);

  short* p1 = dscadscan.sample_values;
  for (int isamp = 0; isamp < getNumChannelsScanned(); isamp++,p1++) {
    int ichan = channelMap[isamp];
    if (ichan >= 0) {
      if (++sampleCounters[ichan] == decimateValues[ichan]) {
	RawSample *sample =
		RawSamplePool::getInstance()->getRawSample(len);
	sample->tt = tt;
	sample->chan = userChannelNumbers[ichan];
#ifdef DEBUG
	cerr << "tt=" << tt << " chan=" << userChannelNumbers[ichan] <<
	    " sampleCounters[ichan]=" << sampleCounters[ichan] <<
	    " decimateValues[ichan]=" << decimateValues[ichan] << endl;
#endif
	sample->len = (unsigned char) len;
#if __BYTE_ORDER == __LITTLE_ENDIAN
	// swap to big-endian
	/*
	sample->data[0] = *((char*)p1+1);
	sample->data[1] = *((char*)p1);
	*/
	::swab(p1,sample->data,len);
#else
	::memcpy(sample->data,p1,len);
#endif
	distribute(sample);
	sampleCounters[ichan] = 0;
      }
    }
  }
}

void dsc::UserClockedA2DSampler::doConversions() throw(IOException)
{

  const int len = sizeof(short);
  BYTE result;

  for (int isamp = 0; isamp < numChannelsScanned; isamp++) {
    int ichan = channelMap[isamp];
    if (ichan < 0) continue;

    int vindex = requestedVoltageRangeIndices[ichan];
    int channelNum = minChannelScanned + isamp;

    dscadsettings.current_channel = channelNum;
    dscadsettings.gain = gainSettings[vindex];
    dscadsettings.range = rangeSettings[vindex];
    dscadsettings.polarity = polaritySettings[vindex];

    if( ( result = dscADSetSettings(getDSCB(), &dscadsettings ) ) != DE_NONE )
    {
      ERRPARAMS errorParams;
      dscGetLastError(&errorParams);
      throw IOException(getName(),"dscADSetSettings",
	  errorParams.errstring);
    }

    short val;
    if ((result = dscADSample(getDSCB(),&val)) != DE_NONE)
	throw IOException(getName(),"dscADSample",dscGetErrorString(result));

    isff_sys_time_t dtime = getCurrentTimeInMillis();

    if (dtime > tlast + tdiffcheck)
      missedSamples += (size_t)(tlast - dtime) / deltatMsec;

    sample_time_t tt = sample_time(dtime,0);

    if (++sampleCounters[ichan] == decimateValues[ichan]) {
      RawSample *sample =
	      RawSamplePool::getInstance()->getRawSample(len);
      sample->tt = tt;
      sample->chan = userChannelNumbers[ichan];
#ifdef DEBUG
      cerr << "tt=" << tt << " chan=" << userChannelNumbers[ichan] <<
	  " sampleCounters[ichan]=" << sampleCounters[ichan] <<
	  " decimateValues[ichan]=" << decimateValues[ichan] << endl;
#endif
      sample->len = (unsigned char) len;
#if __BYTE_ORDER == __LITTLE_ENDIAN
      // swap to big-endian
      /*
      sample->data[0] = *((char*)(&val)+1);
      sample->data[1] = *((char*)(&val));
      */
      ::swab(&val,sample->data,len);
#else
      ::memcpy(sample->data,&val,len);
#endif
      distribute(sample);
      sampleCounters[ichan] = 0;
    }
  }
}

