//
//              Copyright 2004 (C) by UCAR
//

#ifndef DSC_EXTCLOCKEDA2DSAMPLER_H
#define DSC_EXTCLOCKEDA2DSAMPLER_H

#include <dsc/DscA2DSampler.h>
#include <atdISFF/ExtClockedA2DSampler.h>
#include <atdISFF/Time.h>
#include <atdISFF/TimeTagSmoother.h>

namespace dsc {

/**
 * Partial implementation of an atdISFF::ExtClockedA2DSampler using
 * Diamond Systems Corp Universal Driver calls.
 * It uses the dsc::A2DSampler implementation of atdISFF::A2DSampler,
 * and adds some implementation of the
 * atdISFF::ExtClockedA2DSampler interface.
 */
class ExtClockedA2DSampler :
	public virtual dsc::A2DSampler,
	public atdISFF::ExtClockedA2DSampler {

public:
  ExtClockedA2DSampler(int type,int ioPortAddr, int irq)
		throw(atdUtil::IOException);

  /**
   * Destructor.
   */
  virtual ~ExtClockedA2DSampler();


  const std::vector<float>& getAvailableSamplingRates() const {
    return availableSamplingRates;
  }

  void addChannel(int channelNum, int vRangeIndex, int samplingRateIndex)
  	throw(atdUtil::IOException,atdUtil::InvalidParameterException);

  /**
   * must be called after adding all channels, and before start;
   */
  void init() throw(atdUtil::IOException,atdUtil::InvalidParameterException);

  /**
   * Where the work gets done.
   */
  int run() throw(atdUtil::Exception);

  float getSamplingRate(int userChannelNum) const {
    return availableSamplingRates[requestedSamplingRateIndices[channelMapVec[userChannelNum-100]]];
  }

  float getMaxRequestedSamplingRate() const {
    return availableSamplingRates[indexOfMaxSamplingRate];
  }


protected:

  /**
   * Size of FIFO on this DSC A2D.
   */
  virtual int getFifoDepth() const { return fifoDepth; }

  /**
   * How many conversions to do before calling the user interrupt
   * function.
   */
  virtual int getDumpThreshold() const { return dumpThreshold; }

  /**
   * Total size of conversion buffer. Should be at least
   * twice getDumpThreshold (double buffering).
   */
  virtual int getNumConversions() const { return numConversions; }

  /**
   * Implemented by derived classes to return a pointer
   * to an appropriate user interrupt function.
   */
  virtual DSCUserInterruptFunction getStaticDscUserIntFunc() = 0;

  /**
   * Called by the static DSCUserInterruptFunction.
   */
  virtual void dscUserIntFunc(atdISFF::isff_sys_time_t time);

  void setupUserInterruptFunction() throw(atdUtil::IOException);

  /**
   * Check that the sRateIndex is within range. Re-computes
   * indexOfMaxSamplingRate.
   */
  float checkSamplingRate(int sRateIndex) throw(atdUtil::InvalidParameterException);

  /**
   * Check that rate divides evenly into maxrate.
   */
  void checkSamplingRates(float maxrate,float rate) throw(atdUtil::InvalidParameterException);

  void processBuffer(atdISFF::isff_sys_time_t dtime, register const short* p1, int nsamp);

  // structure containing I/O interrupt settings
  DSCAIOINT dscaioint;

  // structure containing interrupt-based sampling status information
  DSCS dscStatus;

  /**
   * What sampling rates are available on this A2D. Must be
   * defined by derived classes.
   */
  std::vector<float> availableSamplingRates;

  /**
   * Indices into availableSamplingRates of rates that have been
   * requested.
   */
  std::vector<int> requestedSamplingRateIndices;

  /**
   * Index of maximum requested sampling rate.
   */
  int indexOfMaxSamplingRate;

  /**
   * Time in milliseconds between conversion scans.
   * Used to compute time tags when samples from more than
   * one conversion time are in a buffer.
   */
  double deltatMsec;

  double deltatMsecAdj;

  /**
   * size of user sample buffer
   */
  int numConversions;
  
  /**
   * size of A2D fifo
   */
  int fifoDepth;
  
  /**
   * user interrupt function is called every dumpThreshold samples.
   */
  int dumpThreshold;
  
  /**
   * Index of where to get next group of samples in sample buffer.
   */
  int bufferIndex;

  /**
   * used to detect whether the user interrupt function is keeping up.
   */
  bool interruptFunctionActive;

  /**
   * number of FIFO overflows
   */
  unsigned int overflows;

  unsigned int transLast;

  int nscansPerDump;

  atdISFF::TimeTagSmoother ttsmooth;

};
}

#endif
