//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <sstream>
#include <iomanip>
#include <math.h>
#include <unistd.h>	// swab

#include <dsc/DscExtClockedA2DSampler.h>
#include <dsc/DscA2DSampler.h>
// #include <dsc/DiamondUD.h>
#include <atdISFF/Time.h>
#include <unistd.h>	// for pause

using namespace dsc;
using namespace atdISFF;
using namespace std;

dsc::ExtClockedA2DSampler::ExtClockedA2DSampler(int type,
	int ioPortAddr, int irq)
	throw(atdUtil::IOException) :
  // virtual base class
  dsc::A2DSampler(type,ioPortAddr,irq),
  // virtual base class
  atdISFF::A2DSampler(dsc::A2DSampler::getName()),
  atdISFF::ExtClockedA2DSampler(dsc::A2DSampler::getName()),
  indexOfMaxSamplingRate(0),bufferIndex(0),overflows(0),
  nscansPerDump(0)
{
  memset(&dscaioint,0,sizeof(dscaioint));
}

dsc::ExtClockedA2DSampler::~ExtClockedA2DSampler() {
  delete [] dscaioint.sample_values;
}

void dsc::ExtClockedA2DSampler::addChannel(int channelNum, int vRangeIndex,
	int sRateIndex)
	throw(atdUtil::IOException,atdUtil::InvalidParameterException) {

  dsc::A2DSampler::addChannel(channelNum,vRangeIndex);

  float maxSamplingRate = checkSamplingRate(sRateIndex);
  requestedSamplingRateIndices.push_back(sRateIndex);
  decimateValueVec.push_back(1);

  for (unsigned int i = 0; i < decimateValueVec.size(); i++) {
    float srate = availableSamplingRates[requestedSamplingRateIndices[i]];
    decimateValueVec[i] = (int)rint(maxSamplingRate / srate);
#ifdef DEBUG
    cerr << "decimateValues[" << i << "]=" << decimateValueVec[i] << endl;
#endif
  }

}

float dsc::ExtClockedA2DSampler::checkSamplingRate(int sRateIndex) throw(atdUtil::InvalidParameterException) {
  if (sRateIndex < 0 || (unsigned) sRateIndex >= availableSamplingRates.size()) {
    ostringstream os;
    os << "sRateIndex=" << sRateIndex << " out of range, max=" <<
      availableSamplingRates.size() - 1;
    throw atdUtil::InvalidParameterException(getName(),"addChannel",os.str());
  }
  float maxSamplingRate = availableSamplingRates[sRateIndex];
  indexOfMaxSamplingRate = sRateIndex;
  float srate;
  for (unsigned int i = 0; i < requestedSamplingRateIndices.size(); i++) {
    srate = availableSamplingRates[requestedSamplingRateIndices[i]];
    if (srate > maxSamplingRate) {
      maxSamplingRate = srate;
      indexOfMaxSamplingRate = requestedSamplingRateIndices[i];
    }
  }

  // check existing sample rates against the new maximum
  for (unsigned int i = 0; i < requestedSamplingRateIndices.size(); i++) {
    srate = availableSamplingRates[requestedSamplingRateIndices[i]];
    checkSamplingRates(maxSamplingRate,srate);
  }

  // check this rate
  checkSamplingRates(maxSamplingRate,availableSamplingRates[sRateIndex]);

  return maxSamplingRate;

}

/**
 * Check that rate divides (fairly) evenly into maxrate, else throw atdUtil::InvalidParameterException.
 */
void dsc::ExtClockedA2DSampler::checkSamplingRates(float maxrate,float rate) throw(atdUtil::InvalidParameterException) {
  double ratio = maxrate / rate;
  if (fabs(ratio - rint(ratio)) > .001) {
    ostringstream os;
    os << "Sample rate=" << rate << " Hz (dt=" << 1.0/rate <<
      " sec) does not divide evenly into the maximum requested rate of " <<
      maxrate << " Hz (dt=" << 1.0/maxrate << " sec)" << 
      " remainder=" << fabs(ratio - rint(ratio));
    throw atdUtil::InvalidParameterException(getName(),"addChannel",os.str());
  }
}

void dsc::ExtClockedA2DSampler::setupUserInterruptFunction() throw(atdUtil::IOException) {
                 
  DSCUSERINTFUNCTION uif;
  uif.func = getStaticDscUserIntFunc();
  uif.int_mode = USER_INT_AFTER;
  dscSetUserInterruptFunction(getDSCB(),&uif);

  //=========================================================================
  // IV. I/O INTERRUPT SETTINGS INITIALIZATION
  //
  //	   Initialize the structure containing the analog I/O interrupt
  //	   settings.
  //
  //	   NOTE: You must allocate space for the buffer holding the returned
  //		     sample values. Also, be generous in allocating storage.
  //			 Allocating insufficient memory to hold sample data will result
  //			 in improper behavior of the driver, such as hanging interrupt
  //			 operations or assertion errors.
  //
  //     STEPS TO FOLLOW:
  //
  //	   1. set the number of conversions (must be a multiple of the fifo
  //		  depth)
  //	   2. set the conversion rate (must be less than 100 kHz)
  //	   3. set the cycle flag
  //	   4. set the internal clock flag
  //	   5. set the low channel (must be between 0-15)
  //	   6. set the high channel (must be between 0-15)
  //	   7. set the external gate enable flag
  //	   8. set the internal clock gate flag
  //	   9. set the fifo enable flag
  //	   10. set the fifo depth (must be 46 or less )
  //	   11. allocate memory for the sample values
  //=========================================================================

  dscaioint.num_conversions = getNumConversions();
  cerr << "dscaiioint.num_conversions=" << dscaioint.num_conversions << endl;

  dscaioint.conversion_rate = availableSamplingRates[indexOfMaxSamplingRate];
  cerr << "dscaiioint.conversion_rate=" << dscaioint.conversion_rate << endl;

  // dscaioint.cycle = (BYTE)TRUE;
  dscaioint.cycle = 1;

  // internal clock generates interrupts
  // dscaioint.internal_clock = (BYTE)TRUE;
  dscaioint.internal_clock = 1;

  dscaioint.low_channel = getMinChannelScanned();
  dscaioint.high_channel = getMinChannelScanned() + getNumChannelsScanned() - 1; 
  // cerr << "dscaiioint.high_channel=" << (int)dscaioint.high_channel << endl;

  /* external_gate_enable defines the control bit C2 on boards DMM,
   * DMM-16, DMM-AT and DMM-16-AT. It is used to enable an external
   * signal on pin 29 (In0-) of the boards I/O connector to provide
   * a gating control for A/D sampling when an external clock is
   * used to control the sampling rate. If enabled, a high level on In0-
   * will enable the sampling to run, and a low level will halt sampling.
   * TRUE = use the external In0- to gate the external sampling clock.
   * FALSE  = no gating is enabled and sampling continues uninterrupted. (sic)
   */
  // dscaioint.external_gate_enable = (BYTE)FALSE;
  dscaioint.external_gate_enable = 0;

  /* internal_clock_rate defines the control bit C0 on boards DMM,
   * DMM-16, DMM-AT and DMM-16-AT. It is used to enable an external
   * signal on pin 48 (Din0) of the boards I/O connector to provide
   * a gating control for A/D sampling when the internal clock is
   * used to control the sampling rate. If enabled, a high level on Din0
   * will enable the sampling to run, and a low level will halt sampling.
   * On the DMM-32-AT this will get GT12EN high.
   * TRUE = use an external signal to gate the internal sampling clock
   * FALSE  = no gating is enabled and sampling continues uninterrupted. (sic)
   */
  // dscaioint.internal_clock_gate = (BYTE)FALSE;
  dscaioint.internal_clock_gate = 0;

  // dscaioint.fifo_enab = (BYTE)TRUE;
  dscaioint.fifo_enab = 1;

  dscaioint.fifo_depth = getFifoDepth();
  cerr << "dscaiioint.fifo_depth=" << dscaioint.fifo_depth << endl;

  /*
   * number of A/D conversions to perform before copying the
   * driver's sample buffer to the user-accessible buffer
   * (and calling the "interrupt" function).
   */
  dscaioint.dump_threshold = getDumpThreshold();
  cerr << "dscaiioint.dump_threshold=" << dscaioint.dump_threshold << endl;

  /* prometheus only: selects the clock source for the counter/timer
   * 1 during D/A conversions. 1=10MHz,0=100KHz
  */
  dscaioint.clksource = 0;

  dscaioint.sample_values = new DSCSAMPLE[dscaioint.num_conversions];
  bufferIndex = 0;
  //=========================================================================
  // V. SCANNING AND OUTPUT
  //
  //    Perform the actual sampling and then output the results. To calculate
  //	  the actual input voltages, we must convert the sample code (which
  //	  must be cast to a short to get the correct code) and then plug it
  //	  into one of the formulas located in the manual for your board (under
  //	  "A/D Conversion Formulas"). For example, if you are using a bipolar
  //	  input range and 32-bit signed integers:
  //
  //	  Input voltage = (AD Code / 32768) x Full-Scale Voltage
  //
  //	  For interrupt-based scanning, we perform scan several channels per
  //	  interrupt.
  //
  //    STEPS TO FOLLOW:
  //
  //	  1. reset the number of transfers to 0
  //	  2. set the operation type to interrupt-based
  //    3. get the samples from the board
  //	  4. wait for the board to signal that the operation has finished
  //    5. convert the results to a signed short to obtain a readout from
  //       -32768 to +32767 which scales to a range from -10V to +10V and
  //	     output the sample codes and the actual voltages to console
  //    6. calculate the input voltages by converting the sample code and
  //		 output the results
  //	  7. repeat steps 1-6 until a key is pressed
  //=========================================================================


  BYTE result; // returned error code
  cerr << "calling dscADScanInt" << endl;
  if( ( result = dscADScanInt(getDSCB(), &dscaioint ) ) != DE_NONE )
  {
    ERRPARAMS errorParams;
    dscGetLastError(&errorParams);
    throw atdUtil::IOException(getName(),"dscADScanInt", 
    	string(errorParams.errstring) + " (make sure single-ended/differential option matches jumpers)");
  }
  cerr << "called dscADScanInt" << endl;

}

void dsc::ExtClockedA2DSampler::init() throw(atdUtil::IOException,atdUtil::InvalidParameterException) {
  // cerr << "dsc::ExtClockedA2DSampler::init()" << endl;

  dsc::A2DSampler::init();

  // indexOfMaxSamplingRate is maintained by the addChannel method,
  // so it should be valid at ::init time.
  deltatMsecAdj = deltatMsec = 1000.0 / availableSamplingRates[indexOfMaxSamplingRate];
  cerr << "deltatMsec=" << deltatMsec << endl;

  nscansPerDump = getDumpThreshold()/getNumChannelsScanned();
}

int dsc::ExtClockedA2DSampler::run() throw(atdUtil::Exception) {

  for (;;) {
    if (amInterrupted()) break;
    pause();
  }

  // The run method has been interrupted. cleanup

  BYTE result; // returned error code
  if( (result = dscCancelOp(getDSCB())) != DE_NONE)
  {
    ERRPARAMS errorParams;
    dscGetLastError(&errorParams);
    throw atdUtil::IOException(getName(),"dscCancelOp", errorParams.errstring);
  }

  // This must be called after dscCancelOp
  if( (result = dscClearUserInterruptFunction(getDSCB())) != DE_NONE)
  {
    ERRPARAMS errorParams;
    dscGetLastError(&errorParams);
    throw atdUtil::IOException(getName(),"dscClearUserInterruptFunction", errorParams.errstring);
  }

  return RUN_OK;
}

void dsc::ExtClockedA2DSampler::dscUserIntFunc(isff_sys_time_t dtime)
{
  userInterrupts++;

  // structure containing interrupt-based sampling status information
  // check interrupt status
  // dscStatus.transfers = 0;
  // dscStatus.overflows = 0;
  // dscStatus.op_type = OP_TYPE_INT;

  /*
  struct DSCS
    total_transfers: the total number of A/D conversions that
  	have taken place since the interrupt operation began.
	When scans are occuring the variable increments by
	the scan size.
    overflows:
      Cumulative number of times the FIFO has overflowed during an interrupt operation.
      Non-zero means A/D data was missed.
  */

  BYTE result; // returned error code
  if ( ( result = dscGetStatus(getDSCB(), &dscStatus) ) != DE_NONE) {
    ERRPARAMS errorParams;
    dscGetLastError(&errorParams);
    atdUtil::IOException ioe("Diamond Sys Corp IO","dscGetStatus",errorParams.errstring);
    cerr << ioe.toString() << endl;
  }
  else {
    if (dscStatus.overflows != overflows) {
      cerr << " Fifo Overflows:" << dscStatus.overflows <<
	" A/D Int Trans in progress:" << dscStatus.transfers <<
	" Total Trans:" <<  dscStatus.total_transfers << 
	" bufferIndex=" << bufferIndex << endl;
      int nover = dscStatus.overflows - overflows;
      missedSamples += nover;
      cumulativeMissedSamples += nover;
      overflows = dscStatus.overflows;
    }
    if (dscStatus.op_type != OP_TYPE_INT) cerr << " wrong op_type" << endl;
  }

  int ntrans;
  // correct for rollover
  if (dscStatus.total_transfers < transLast)
      ntrans = dscStatus.total_transfers + (ULONG_MAX - transLast) + 1;
  else
      ntrans = dscStatus.total_transfers - transLast;

  if ((ntrans % getDumpThreshold())) {
      cerr << "ntrans=" << ntrans << " getDumpThreshold=" << getDumpThreshold() << endl;
      int transmod = ntrans % getDumpThreshold();
      ntrans -= transmod;
  }

  if (ntrans != getDumpThreshold()) cerr << "ntrans=" << ntrans << endl;

  int nscans = ntrans / getNumChannelsScanned();
  dtime = isff_sys_time_add(dtime,-deltatMsecAdj * nscans);

  for (int i = 0; i < ntrans / getDumpThreshold(); i++) {
      processBuffer(dtime,dscaioint.sample_values+bufferIndex,getDumpThreshold());
      dtime = isff_sys_time_add(dtime,deltatMsecAdj * nscansPerDump);
      bufferIndex = (bufferIndex + getDumpThreshold()) % getNumConversions();
  }

  if (dtime > statisticsTime) resetStatistics(dtime);
  transLast += ntrans;
}

void dsc::ExtClockedA2DSampler::processBuffer(isff_sys_time_t dtime, register const short* p1, int npts)
{
    /* dtime is the system time when the A2D buffer was available */
    // TODO: we should include the per-channel scan time * nchans
    assert((npts % getNumChannelsScanned()) == 0);
    int nsamps = npts / getNumChannelsScanned();
    int nchans = getNumChannelsRequested();

#ifdef DEBUG
    isff_sys_time_t dtimesave = dtime;
#endif

    dtime = ttsmooth.getTime(dtime,nsamps,deltatMsecAdj);

#ifdef DEBUG
    cerr << "dtime=" << dtimesave << ' ' << dtime <<
    	" diff=" << dtime - dtimesave <<
	" dt=" << deltatMsecAdj << endl;
#endif

    register const short* p2 = p1 + npts;
    for (int isamp = 0; isamp < nsamps; isamp++) {
	RawSample *sample =
		RawSamplePool::getInstance()->getRawSample(nchans*3);
	sample->tt = sample_time(dtime,isamp * deltatMsecAdj);
	sample->chan = 1;
	unsigned char* outp = (unsigned char*)sample->data;
	for (int ichan = 0; ichan < getNumChannelsScanned(); ichan++) {
	// for ( ; p1 < p2; p1++)
	    int chan = channelMap[ichan];	// -1 means ignore sample
	    if (chan >= 0) {
		if (++sampleCounters[chan] == decimateValues[chan]) {
		    *outp++ = userChannelNumbers[chan];
		    // TODO: add conversion deltaT if it's appreciable
#ifdef DEBUG
		    cerr << "sample->tt=" << sample->tt <<
		    	" chan=" << userChannelNumbers[chan] << 
			" sampleCounters[chan]=" << sampleCounters[chan] <<
			" decimateValues[chan]=" << decimateValues[chan] <<
			endl;
#endif
#if __BYTE_ORDER == __LITTLE_ENDIAN
		    // swap to big-endian
		    ::swab(p1,outp,sizeof(short));
#else
		    ::memcpy(outp,p1,len);
#endif
		    outp += sizeof(short);
		    p1++;
		    sampleCounters[chan] = 0;
		}
	    }
	}
	sample->len = outp - (unsigned char *)sample->data;
	distribute(sample);

    }
    assert(p1==p2);

#ifdef DEBUG
    printf( "\nSample readouts:" );

    for( i = dscaioint.low_channel; i < dscaioint.high_channel + 1; i++)
	printf( " %d", (short)dscaioint.sample_values[i] );

    printf( "\nActual voltages:" );

    for( i = dscaioint.low_channel; i < dscaioint.high_channel + 1; i++) {
	float voltage = (short)dscaioint.sample_values[i] *
		getConversionSlope(i) + getConversionOffset(i);
	printf( " %5.3fV", voltage);
    }

#endif
} 
