//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <sstream>
#include <iomanip>

#include <dsc/DscBoard.h>
#include <dsc/DiamondUD.h>

using namespace dsc;
using namespace std;

/* static */
string DscBoard::getTypeName(int type)
{
    if (type == DSC_PROM) return "prometheus";
    if (type == DSC_DMM16AT) return "mm16at";
    if (type == DSC_DMM32XAT) return "mm32xat";
    if (type == DSC_IR104) return "ir104";
    return "unknown";
}

/* static */
string DscBoard::makeName(int type,int ioPortAddr,int irq)
{

    ostringstream os;
    os << getTypeName(type) << "(ioport=0x" <<
	 std::setw(3) << std::setfill('0') << std::hex << ioPortAddr <<
	",irq=" << std::dec << irq << ")";
    return os.str();
}

/* static */
void DscBoard::parseName(const string& name,int &type, int& ioPortAddress,
	int& irq)
    throw(atdUtil::ParseException)
{
    cerr << "begin parseName" << endl;
    unsigned int oparen = name.find('(');
    if (oparen == string::npos)
    	throw atdUtil::ParseException(name,"no open parenthesis found");

    string field = name.substr(0,oparen);
    cerr << "field=" << field << endl;
    if (!field.compare("prometheus")) type = DSC_PROM;
    else if (!field.compare("mm16at")) type = DSC_DMM16AT;
    else if (!field.compare("mm32xat")) type = DSC_DMM32XAT;
    else if (!field.compare("ir104")) type = DSC_IR104;
    else throw atdUtil::ParseException(name,"board type not recognized");

    unsigned int pi = name.find("(ioport=0x",oparen);
    if (pi == string::npos)
    	throw atdUtil::ParseException(name,"\"(ioport=0x\" not found");
    cerr << "pi=" << pi << endl;
    pi += 10;

    unsigned int comma = name.find(",irq=",pi);
    if (comma == string::npos)
    	throw atdUtil::ParseException(name,"\",irq=\" not found");

    field = name.substr(pi,comma-pi);
    cerr << "field=" << field << endl;

    istringstream ist(field);
    ist >> hex >> ioPortAddress;
    if (ist.fail())
    	throw atdUtil::ParseException(name,"cannot parse hex ioport address");

    unsigned int cparen = name.find(')',comma);
    if (cparen == string::npos)
    	throw atdUtil::ParseException(name,"no close parenthesis found");

    cerr << "comma=" << comma << endl;
    comma += 5;
    field = name.substr(comma,cparen-comma);
    ist.clear();
    ist.str(field);
    ist >> dec >> irq;
    if (ist.fail())
    	throw atdUtil::ParseException(name,"cannot parse irq");
}

/* static */
std::map<std::string,DscBoard*> DscBoard::boards;

/* static */
DscBoard* DscBoard::getInstance(int type,int ioPortAddress, int irq)
  	throw(atdUtil::IOException)
{
    DscBoard* brd = boards[makeName(type,ioPortAddress,irq)];
    cerr << "brd=" << brd << endl;

    if (brd) return brd;

    // can't use auto_ptr here, since ~DscBoard is protected. pity
    brd = new DscBoard();
    cerr << "new brd=" << brd << endl;

    try {
	brd->open(type,ioPortAddress,irq);
    }
    catch(const atdUtil::IOException& e) {
        delete brd;
	throw e;
    }
    cerr << "brd->getName" << brd->getName() << endl;

    boards[brd->getName()] = brd;
    return brd;
}

/* static */
DscBoard* DscBoard::getInstance(const string& name)
  	throw(atdUtil::IOException,atdUtil::ParseException)
{
    int type = 0;
    int ioPortAddress = 0;
    int irq = 0;

    parseName(name,type,ioPortAddress,irq);
    return getInstance(type,ioPortAddress,irq);
}

/* static */
void DscBoard::release(DscBoard* brd) throw(atdUtil::IOException)
{
    if (!brd) return;
    if (brd->decRefCount() == 0) {
	boards[brd->getName()] = 0;
	brd->close();
	delete brd;
    }
}

DscBoard::DscBoard(): refCount(1)
{
    memset(&dsccb,0,sizeof(dsccb));
}

DscBoard::~DscBoard() {}

void DscBoard::open(int btype,int ioPortAddress, int irq)
  	throw(atdUtil::IOException)
{

    if (!DiamondUD::initDone)
	throw atdUtil::IOException(getName(),"dscInit","missing");

    //====================================================================
    // II. BOARD INITIALIZATION
    //
    //	   Initialize the PROMETHEUS board. This function passes the various
    //	   hardware parameters to the driver and resets the hardware.
    //
    //     STEPS TO FOLLOW:
    //
    //	   1. set the board type to DSC_PROM for PROMETHEUS board
    //	   2. set the base address to 0x280
    //	   3. set the interrupt level 5
    //	   4. intialize and register the board with the driver, after which
    //		  the struct, dscb, now holds the handle for the board
    //=========================================================================

    type = btype;
    name = makeName(type,ioPortAddress,irq);

    cerr << "name=" << name << endl;
    
    cerr << "type=" << type <<
    	hex << " ioPortAddress=" << ioPortAddress <<
	dec << " irq=" << irq << endl;

    dsccb.base_address = ioPortAddress;
    dsccb.int_level = irq;

    BYTE result;
    if((result = dscInitBoard(type, &dsccb, &dscb)) != DE_NONE) {
	cerr << "dscInitBoard failure:" << dscGetErrorString(result) << endl;
	throw atdUtil::IOException(getName(),
		"dscInitBoard",dscGetErrorString(result));
    }
    cerr << "dscInitBoard done" << endl;
}

int DscBoard::incRefCount()
{
    atdUtil::Synchronized autoMutex(refMutex);
    return ++refCount;
}

int DscBoard::decRefCount()
{
    atdUtil::Synchronized autoMutex(refMutex);
    return --refCount;
}

void DscBoard::close() throw(atdUtil::IOException)
{
    if(dscFreeBoard(dscb)!= DE_NONE) {
	ERRPARAMS errorParams;
	dscGetLastError(&errorParams);
	throw atdUtil::IOException(getName(),"dscFreeBoard",errorParams.errstring);
    }
}
