//
//              Copyright 2004 (C) by UCAR
//

#ifndef DSC_DSCA2DSAMPLER_H
#define DSC_DSCA2DSAMPLER_H

#include <string>

#include <dsc/DscBoard.h>

#include <atdUtil/IOException.h>
#include <atdISFF/A2DSampler.h>
#include <atdISFF/Time.h>

namespace dsc {

/**
 * Base class for Diamond Systems Corp (DSC) A2Ds which use the
 * Universal Driver (DSCUD).  Implement the part of
 * atdISFF::A2DSampler that is common to all DSC A2Ds.
 * Derived classes should set the appropriate voltage
 * ranges, gains, polarities, etc.
 */
class A2DSampler : public virtual atdISFF::A2DSampler {

public:
  A2DSampler(int type,int ioPortAddr, int irq);

  /**
   * Destructor.
   */
  ~A2DSampler();

  void open() throw(atdUtil::IOException);

  void close() throw(atdUtil::IOException);

  const std::string& getName() const
  { 
      return name;
  }

  DSCB getDSCB() const
  {
      if (board) return board->getDSCB();
      return -1;
  }

  /**
   * Request a channel of this A2D.  The userChannelNum is the
   * A2D channel + 100.  User channel 107 is port 7 on this A2D.
   * The A2D must have been opened.
   */
  void addChannel(int userChannelNum, int vRangeIndex) throw(atdUtil::IOException,atdUtil::InvalidParameterException);

  /**
   * How many channels do we need to scan.  This can be different
   * from the number of channels requested if the user
   * skips channels.
   */
  int getNumChannelsScanned() const { return numChannelsScanned; }

  int getNumChannelsRequested() const { return (int) userChannelNumberVec.size(); }

  /**
   * must be called after adding all channels, and before start;
   */
  void init() throw(atdUtil::IOException,atdUtil::InvalidParameterException);

  /**
   * Implementation of atdISFF::A2DSampler.  vRanges must be
   * initialized by derived classes.
   */
  const std::vector<const float*>& getAvailableVoltageRanges() const {
    return vRanges;
  }

  /**
   * Implementation of atdISFF::A2DSampler.
   */
  const std::vector<int>& getUserChannelNumbers() const {
    return userChannelNumberVec;
  }

  /**
   * Implementation of atdISFF::A2DSampler.
   */
  const float* getVoltageRange(int userChannelNum) const {
    return vRanges[requestedVoltageRangeIndices[channelMapVec[userChannelNum-100]]];
  }

  /**
   * Implementation of atdISFF::A2DSampler.
   */
  float getConversionSlope(int userChannelNum) const {
    return conversionSlopes[requestedVoltageRangeIndices[channelMapVec[userChannelNum-100]]];
  }

  /**
   * Implementation of atdISFF::A2DSampler.
   */
  float getConversionIntercept(int userChannelNum) const {
    return conversionIntercepts[requestedVoltageRangeIndices[channelMapVec[userChannelNum-100]]];
  }

  size_t getMissedSamples() const { return missedSamples; }

  size_t getCumulativeMissedSamples() const { return cumulativeMissedSamples; }
  

protected:

  void resetStatistics(atdISFF::isff_sys_time_t tnow);

  int boardType;
  
  int boardIoPortAddr;
  
  int boardIrq;

  std::string name;

  // The Diamond systems board.
  DscBoard* board;

  // structure containing A/D conversion settings
  DSCADSETTINGS dscadsettings;

  /**
   * Minimum channel number to be scanned, numbered from 0.
   */
  int getMinChannelScanned() const {
    return minChannelScanned;
  }

  /**
   * Minimum channel number to be scanned, numbered from 0.
   */
  int minChannelScanned;

  /**
   * Number of a2d channels to be scanned.
   */
  int numChannelsScanned;

  /**
   * Mapping from the channel numbers to
   * the sequence that they were requested, where -1 
   * means the port was not requested.
   * User may not have requested every channel in the
   * range minChannelScanned to (minChannelScanned+numChannelsScanned-1)
   * Example:
   *   User has done an addChannel of user channels 101,104,107,103
   *   in that order.  They are A2D channels 1,4,7,3
   * minChannelScanned=1,
   * numChannelsScanned = 7;
   * channelMapVec={-1,0,-1,3,1,-1,-1,2}
   * These are the indices starting from 0.
   * channel 1 was the 0th port requested, port 2 was not requested,
   * 3 is the 3th port requested, 4 is the 1th port, 5&6 not requested,
   * 7 is the 2th port requested.
   * userChannelNumberVec = { 100,104,107,103 };
   */
  std::vector<int> channelMapVec;

  /**
   * The requested channel numbers.
   */
  std::vector<int> userChannelNumberVec;

  /**
   * report every decimateValueVec[i]th sample
   */
  std::vector<int> decimateValueVec;

  /**
   * Vector of available voltage ranges on this A2D.
   * Each element is a float array of length 3:
   * range[0]= minimum input voltage
   * range[1]= maximum input voltage
   * range[2]= voltage resolution in microVolts
   */
  std::vector<const float*> vRanges;

  /**
   * For each requested channel, the index into vRanges that
   * was requested.
   */
  std::vector<int> requestedVoltageRangeIndices;

  /**
   * The values for DSCADSETTINGS.gain for the available voltage ranges.
   */
  std::vector<int> gainSettings;

  /**
   * The values for DSCADSETTINGS.range for the available voltage ranges.
   */
  std::vector<int> rangeSettings;

  /**
   * Slope of conversion from counts to volts:
   * volts = counts * slope + intercept;
   */
  std::vector<float> conversionSlopes;

  /**
   * Intercept of conversion from counts to volts:
   * volts = counts * slope + intercept;
   */
  std::vector<float> conversionIntercepts;

  /**
   * The values for DSCADSETTINGS.polarity for the available voltage ranges.
   */
  std::vector<int> polaritySettings;

  /*
   * optimized version of channelMapVec. Leading -1's are removed.
   */
  int *channelMap;

  /**
   * optimized userChannelNumberVec
   */
  int* userChannelNumbers;

  /**
   * cycling sample counter for each channel.
   */
  int* sampleCounters;

  /**
   * optimized decimateValueVec
   */
  int* decimateValues;
  
  /**
   * total number of missed A2D samples since startup.
   * Samples are missed because either the DSC fifo overflowed
   * or the user interrupt function was not keeping up.
   */
  size_t cumulativeMissedSamples;

  /**
   * number of missed A2D samples in latest statistics period.
   */
  size_t missedSamples;

  /**
   * number of times our user "interrupt routine" has been called.
   */
  int userInterrupts;

};
}

#endif
