//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>
#include <sstream>
#include <iomanip>
#include <math.h>

#include <dsc/DscA2DSampler.h>
#include <dsc/DiamondUD.h>
#include <atdISFF/RawSamplePool.h>
#include <atdISFF/Time.h>
#include <unistd.h>	// for sleep

using namespace dsc;
using namespace std;
using namespace atdUtil;
using namespace atdISFF;


dsc::A2DSampler::A2DSampler(int type,int ioPortAddress, int irq):
    atdISFF::A2DSampler(DscBoard::makeName(type,ioPortAddress,irq)),
    boardType(type),boardIoPortAddr(ioPortAddress),boardIrq(irq),
    name(DscBoard::makeName(type,ioPortAddress,irq)),
    board(0),
    numChannelsScanned(0),channelMap(0),userChannelNumbers(0),
    sampleCounters(0),decimateValues(0),
    cumulativeMissedSamples(0),missedSamples(0),userInterrupts(0)
{

  memset(&dscadsettings,0,sizeof(dscadsettings));

  resetStatistics(getCurrentTimeInMillis());

  // Initialize minChannelScanned. Ideally one could set it to
  // getMaxNumberOfChannels.  However that is set
  // in the subclass constructor, so we don't have it at
  // this point.
  minChannelScanned = -1;
  numChannelsScanned = 0;
}

dsc::A2DSampler::~A2DSampler() {
  delete [] userChannelNumbers;
  delete [] sampleCounters;
  delete [] decimateValues;
  delete [] channelMap;
  if (board) DscBoard::release(board);
}

void dsc::A2DSampler::open() throw(atdUtil::IOException)
{
    board = DscBoard::getInstance(boardType,boardIoPortAddr,boardIrq);
    name = board->getName();
}

void dsc::A2DSampler::close() throw(atdUtil::IOException)
{
    if (board) DscBoard::release(board);
    board = 0;
}

void dsc::A2DSampler::addChannel(int userChannelNum, int vRangeIndex) throw(IOException,InvalidParameterException) {

  // the Nth channel requested, numbered from 0
  int channelIndex = getNumChannelsRequested();

  // which A2D channel on this board
  int channelNum = userChannelNum - 100;

  if (channelNum >= getMaxNumberOfChannels()) {
    ostringstream os;
    os << getMaxNumberOfChannels() + 100 - 1;
    throw InvalidParameterException(getName(),"addChannel",
	string("maximum allowed channel number is ") +  os.str());
  }

  /*
   * example: first userChannel 104, channelNum 4 was requested:
   * channelMapVec = { -1,-1,-1,-1,0 },
   *	minChannelScanned=4, numChannelsScanned=1
   *
   * then userChannel=102, channelNum 2 was requested:
   * channelMapVec = { -1,-1, 1,-1, 0 },
   *	minChannelScanned=2, numChannelsScanned=3
   *
   * then userChannel=106, channelNum 6, was requested:
   * channelMapVec = { -1,-1, 1,-1, 0, -1, 2 }, 
   *	minChannelScanned=2, numChannelsScanned=5
   * channelMap is the same as channelMapVec, except leading
   * -1's are removed.
   */

  // minChannelScanned is initialized as maxNumberOfChannels
  if (minChannelScanned < 0) minChannelScanned = getMaxNumberOfChannels();

  if (channelNum < minChannelScanned) minChannelScanned = channelNum;

  for (int i = channelMapVec.size(); i <= channelNum; i++)
    channelMapVec.push_back(-1);

  channelMapVec[channelNum] = channelIndex;

  numChannelsScanned = channelMapVec.size() - minChannelScanned;

  if (vRangeIndex < 0 || (unsigned) vRangeIndex > vRanges.size()) {
    ostringstream os;
    os << "vRangeIndex=" + vRangeIndex << " out of range, max=" <<
	vRanges.size();
    throw InvalidParameterException(getName(),"addChannel",os.str());
  }

  dscadsettings.current_channel = channelNum;
  dscadsettings.gain = gainSettings[vRangeIndex];
  dscadsettings.range = rangeSettings[vRangeIndex];
  dscadsettings.polarity = polaritySettings[vRangeIndex];

  /* 2002 Diamond Systems Corp. Universal Driver Software Reference Guide
   * V5.6 PAGE 154
   * scan_interval: determines the time delay between consecutive
   * A/D conversions during an A/D scan operation on Diamond-MM-32-AT.
   * When you perform an A/D scan, the individual A/D samples will be
   * spaced apart by exactly this amount of time. The time delay affects
   * the maximum possible sample rate. For example, if you want to
   * achieve a 100KHz sample rate, the maximum scan interval is
   * 1/100KHz = 10 microseconds.
   * 0 20 microseconds (default value)
   * 1 15 microseconds
   * 2 10 microseconds
   * 3 5 microseconds 
   */
  switch (boardType) {
  case DSC_DMM32XAT:
  case DSC_DMM32AT:
      dscadsettings.scan_interval = 0;
      break;
  default:
      dscadsettings.scan_interval = 0;
      break;
  }

  dscadsettings.load_cal = (BYTE)TRUE;

  //=========================================================================
  // III. AD SETTINGS INITIALIZATION
  //
  //	    Initialize the structure containing the AD conversion settings and
  //		then pass it to the driver.
  //
  //      STEPS TO FOLLOW:
  //
  //		1. set the polarity (must be BIPOLAR or UNIPOLAR)
  //		2. set the gain (must be GAIN_1, GAIN_2, GAIN_4, or GAIN_8)
  //		3. set the current channel to be sampled (must be between 0-15)
  //		4. initialize the AD conversion with these settings
  //=========================================================================
  BYTE result; // returned error code

  cerr << "dscadsettings.current_channel=" << (int)dscadsettings.current_channel << endl;
  cerr << "dscadsettings.gain=" << (int)dscadsettings.gain << endl;
  cerr << "dscadsettings.range=" << (int)dscadsettings.range << endl;
  cerr << "dscadsettings.polarity=" << (int)dscadsettings.polarity << endl;
  cerr << "dscadsettings.load_cal=" << (int)dscadsettings.load_cal << endl;
  cerr << "dscadsettings.scan_interval=" << (int)dscadsettings.scan_interval << endl;
  // addiff is for Hercules only, software settable diff vs SE
  // cerr << "dscadsettings.addiff=" << (int)dscadsettings.addiff << endl;
#ifdef DEBUG
#endif

  if( ( result = dscADSetSettings(getDSCB(), &dscadsettings ) ) != DE_NONE )
  {
    ERRPARAMS errorParams;
    dscGetLastError(&errorParams);
    throw IOException(getName(),"dscADSetSettings",
    	errorParams.errstring);
  }

  userChannelNumberVec.push_back(userChannelNum);
  requestedVoltageRangeIndices.push_back(vRangeIndex);
}

void dsc::A2DSampler::init() throw(IOException,InvalidParameterException) {

  delete [] userChannelNumbers;
  delete [] decimateValues;
  delete [] sampleCounters;
  int nchan = getNumChannelsRequested();
  // create arrays from vector<>s
  userChannelNumbers = new int[nchan];
  decimateValues = new int[nchan];
  sampleCounters = new int[nchan];

  copy(userChannelNumberVec.begin(),userChannelNumberVec.end(),userChannelNumbers);
  copy(decimateValueVec.begin(),decimateValueVec.end(),decimateValues);
  copy(decimateValueVec.begin(),decimateValueVec.end(),sampleCounters);

  // set sampleCounters to decimateValue[i] - 1
  unsigned int i;
  for (i = 0; i < (unsigned) nchan; i++) sampleCounters[i]--;

  // find first non-negative channel in channelMapVec
  for (i = 0; i < channelMapVec.size(); i++)
  	if (channelMapVec[i] >= 0) break;

  delete [] channelMap;
  channelMap = new int[channelMapVec.size() - i];
  copy(channelMapVec.begin() + i,channelMapVec.end(),channelMap);
}

void dsc::A2DSampler::resetStatistics(isff_sys_time_t tnowMsec)
{
  statisticsTime += statisticsPeriod;
  if (statisticsTime < tnowMsec)
    statisticsTime = ((tnowMsec / statisticsPeriod) + 1) * statisticsPeriod;

  // statisticsPeriod is in milliseconds, hence the factor of 1000.
  totalObservedSamplingRate = (float)nsamples / statisticsPeriod * 1000.;

  userInterruptRate = (float) userInterrupts / statisticsPeriod * 1000.;

  missedSamples = 0;
  nsamples = 0;
  userInterrupts = 0;
}

