//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/relays/RrelayListener.h>
#include <atdUtil/Logger.h>

#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <errno.h>
#include <stdio.h>

using namespace std;
using namespace atdISFF;
using namespace atdUtil;

/* static */
// const int RrelayListener::RRELAY_PORT = 8101;
/**
 * Open socket for listening on port 8101.
 */
RrelayListener::RrelayListener(RelayProgram& rly,RawSampleSource* src)
	throw(IOException) :
	Thread("RrelayListener"),
	relayProgram(rly),sampleSource(src)
{
  blockSignal(SIGINT);
  blockSignal(SIGHUP);
  blockSignal(SIGTERM);
  unblockSignal(SIGUSR1);

  if ((listenfd = socket(AF_INET,SOCK_STREAM,0)) < 0)
    throw IOException("RrelayListener socket","open",errno);

  int i = 1;
  if (setsockopt(listenfd,SOL_SOCKET,SO_REUSEADDR,
	      (char *)&i,sizeof i) < 0)
    throw IOException("RrelayListener socket","setsockopt",errno);

  struct sockaddr_in myAddr;
  memset(&myAddr,0,sizeof myAddr);
  myAddr.sin_family = AF_INET;
  myAddr.sin_port = htons(RRELAY_PORT);

  if (bind(listenfd, (struct sockaddr *) & myAddr, sizeof(myAddr)) < 0) {
    ::close(listenfd);
    listenfd = -1;
    throw IOException("RrelayListener socket","bind",errno);
  }

  if (listen(listenfd, 10) < 0) {
    ::close(listenfd);
    listenfd = -1;
    throw IOException("RrelayListener socket","listen",errno);
  }
}

RrelayListener::~RrelayListener()
{
  // socket is closed at end of run method
  // delete any remaining connections
  list<RrelayConnection*>::iterator ic;
  for (ic = connections.begin(); ic != connections.end(); ++ic)
    delete *ic;
}

/*
 * In run method, do accept, then create RrelayConnection.
 * RrelayConnection then creates an iostream out of
 * the file descriptor, creates an AirCOAOutputstream
 * with it, and starts a run method.
 * run method of RrelayConnection just reads from the iostream.
 */
int RrelayListener::run() throw(atdUtil::Exception)
{
  struct sockaddr_in clientAddr;

#ifdef USE_SELECT
  fd_set readfdset;
  FD_ZERO(&readfdset);
  FD_SET(listenfd,&readfdset);
  int selectn = listenfd + 1;
  struct timeval tout;
#endif
 
  for (;;) {

    // check for old connections that aren't running anymore
    list<RrelayConnection*>::iterator ic;
    {
	Synchronized autolock (connectionLock);
	for (ic = connections.begin(); ic != connections.end(); ) {
	  RrelayConnection* conn = *ic;
	  if (!conn->isRunning()) {
	    cerr << "deleting " << conn->getName() << endl;
	    try {
	      conn->join();
	    }
	    catch (Exception& e) {
	      Logger::getInstance()->log(
		    LOG_ERR,"Exception from RrelayConnection::join : %s",
			    e.what());
	    }
	    delete conn;
	    ic = connections.erase(ic);
	  }
	  else ++ic;
	}
    }

    if (amInterrupted()) break;

    socklen_t client_len = sizeof(clientAddr);
    cerr << "accept" << endl;
    int fd = accept(listenfd, (struct sockaddr *) & clientAddr, &client_len);
    cerr << "accept done, fd=" << fd << endl;

    if (amInterrupted()) break;

    if (fd < 0) {
      if (errno == EINTR) break;	// should only occur if amInterrupted
      throw IOException("RrelayListener socket","accept",errno);
    }

    connectionLock.lock();
    connections.push_back(new RrelayConnection(fd,relayProgram,sampleSource));
    connectionLock.unlock();
  }
  cerr << "RrelayListener run returning" << endl;
  ::close(listenfd);
  return RUN_OK;
}

void
RrelayListener::interrupt() {
  cerr << "RrelayListener::interrupt" << endl;
  list<RrelayConnection*>::iterator ic;
  connectionLock.lock();
  for (ic = connections.begin(); ic != connections.end(); ++ic) {
    RrelayConnection* conn = *ic;
    if (conn->isRunning()) conn->cancel();
  }
  connectionLock.unlock();
  Thread::interrupt();
  cerr << "RrelayListener::interrupt done" << endl;
}
