//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/relays/RelayProgram.h>
#include <atdISFF/Time.h>
#include <atdUtil/Logger.h>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

RelayProgram::RelayProgram(): relay(0),timer(0),
	repNum(0),seqNum(0),setNum(0),
	interrupted(false),
	nsamples(0),override(false),override_end(false),channel(164)
{
}

void RelayProgram::setOverride(bool val) throw(IOException) { 

  // detect if we're ending the override state - need to set
  // the relays on next wakeup
  override_end = !val && override;
  override = val;
  // relay->setOverride(val);
}

int RelayProgram::run() throw(atdUtil::Exception)
{
  unsigned int programDuration = getDuration();
  cerr << "relay program duration = " << programDuration << endl;

  programDuration *= 1000;
  atdISFF::isff_sys_time_t dtime = timer->currentTimeMsec();

#ifdef TIME_IS_LONG_LONG
  atdISFF::isff_sys_time_t progStart = dtime / programDuration;
#else
  atdISFF::isff_sys_time_t progStart =
  	(atdISFF::isff_sys_time_t)floor(dtime / programDuration);
#endif

  // When would this program have started last.
  progStart *= programDuration;

  atdISFF::isff_sys_time_t nextTime = progStart + getSettingDuration() * 1000;

  // step through program without setting relays until
  // we get to where we should be, then set them.
  while (dtime >= nextTime) {
    step();	// shouldn't get to end
    nextTime += getSettingDuration() * 1000;
  }

  relay->setRelayBits(getSetting());
  try {
    relay->setRelays();
  }
  catch (IOException& ioe) {
    // log it and re-throw.  Otherwise an exception from
    // a run method isn't seen until the wait.
    Logger::getInstance()->log(LOG_ERR,"%s: %s",
      relay->getName().c_str(),ioe.what());
    throw ioe;
  }

  for (;;) {
    if (amInterrupted() || timer->isInterrupted()) {
      Logger::getInstance()->log(LOG_INFO,"%s",
          "RelayProgram::run interrupted");
      break;
    }
    timer->lock();
    timer->wait();
    if (amInterrupted() || timer->isInterrupted()) {
      Logger::getInstance()->log(LOG_INFO,"%s",
          "RelayProgram::run timer interrupted");
      timer->unlock();
      break;
    }
    timer->unlock();

    dtime = timer->currentTimeMsec();

    while (dtime >= nextTime) {
      step();
      
      if (!override) {
	relay->setRelayBits(getSetting());
	try {
	  relay->setRelays();
	}
	catch (IOException& ioe) {
	  // log it and re-throw.  Otherwise an exception from
	  // a run method isn't seen until the wait.
	  Logger::getInstance()->log(LOG_ERR,"%s: %s",
	    relay->getName().c_str(),ioe.what());
	  throw ioe;
	}
      }
      nextTime += getSettingDuration() * 1000;
    }
      
    if (override_end) {
      relay->setRelayBits(getSetting());
      try {
	relay->setRelays();
      }
      catch (IOException& ioe) {
	Logger::getInstance()->log(LOG_ERR,"%s: %s",
	  relay->getName().c_str(),ioe.what());
	throw ioe;
      }
      override_end = false;
    }

    BitArray& relaybits = relay->getRelayBits();
    const int rlen = relaybits.getLengthInBytes();

    // If relay is over-ridden, set extra bit.
    int nbits = relaybits.getLength();
    bool rover = getOverride();
    if (rover) nbits++;
    const int len = (nbits + 7) / 8;

    sample_time_t tt = sample_time(dtime,0);
                                                                                
    RawSample *sample =
                RawSamplePool::getInstance()->getRawSample(len);
    sample->tt = tt;
    sample->chan = getChannel();
    sample->len = (unsigned char) len;
    ::memset(sample->data,0,len);
    ::memcpy(sample->data,relaybits.getPtr(),rlen);
    if (rover) sample->data[len-1] |= 1 << ((nbits - 1) % 8);
    distribute(sample);
  }
  relay->openRelays();
  return atdUtil::Thread::RUN_OK;
}

/**
 * step forward in program. Return false if at end of program.
 * Also, reset the relay if at end.
 */
bool RelayProgram::step() {
  bool eop = false;
  if (++setNum == getSequence(seqNum).size()) {
    setNum = 0;
    if (++repNum == getRepeat(seqNum)) {
      repNum = 0;
      if (++seqNum == size()) {
        eop = true;
	seqNum = 0;
	relay->reset();
      }
    }
  }
  return !eop;
}

/**
 * Reset program to beginning.
 */
void RelayProgram::reset() {
  setNum = repNum = seqNum = 0;
}

const RelaySetting& RelayProgram::getSetting() const {
  return getSequence(seqNum).getSetting(setNum);
}

int RelayProgram::getSettingDuration() const {
  return getSequence(seqNum).getDuration(setNum);
}

