//
//              Copyright 2004 (C) by UCAR
//

#include <iostream>

#include <stdio.h>
#include <string.h>
#include <errno.h>


#include <atdISFF/relays/RelayParser.h>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

RelayParser::~RelayParser()
{
  if (fp) fclose(fp);
}

/**
 * Read next line in file. Skip blank lines and comment lines.
 */
void RelayParser::readLine() throw(IOException,EOFException) {
  for (;;) {
    lineno++;
    if (fgets(linebuf,sizeof(linebuf),fp) == 0) {
      if (feof(fp)) {
        popfile();	// throws EOFException if no files left
	continue;
      }
      throw IOException(filename,"read",errno);
    }

#ifdef DEBUG
    cerr << "sizeof(linebuf)=" << sizeof(linebuf) << " strlen=" << strlen(linebuf) << 
    	" linebuf=" << (char*)linebuf << endl;
#endif

    curptr = linebuf;

    // skip white space. If reach end of line, or a # comment, then read again
    if (skipTilEOL()) continue;

    char word[128];
    int nchar;
    if (sscanf(curptr,"%s%n",word,&nchar) > 0 && !strcmp(word,"include")) {
      curptr += nchar;
      char nextname[128];
      // no spaces allowed in file name.
      if (sscanf(curptr,"%s%n",nextname,&nchar) < 1)
	throw ParseException(filename,"error parsing include statement",lineno);
      curptr += nchar;
      if (!skipTilEOL())
	throw ParseException(filename,"error parsing include statement",lineno);
      pushfile(nextname);
      continue;
    }
    break;
  }
}

/**
 * Skip blanks, tabs or carriage-returns.  Does not skip anything else
 * including newlines or nulls '\0'.
 */
const char* RelayParser::skipWhite() {
  for ( ; *curptr == ' ' || *curptr == '\t' || *curptr == '\r'; curptr++);
  return curptr;
}

/**
 * Skip blanks, tabs or carriage-returns.  Return a bool true if
 * the the next character is a '#', newline or NULL, otherwise return false.
 */
bool RelayParser::skipTilEOL() {
  skipWhite();
  return (*curptr == '\n' || *curptr == '#' || *curptr == '\0');
}

void RelayParser::pushfile(const string& fname) throw(IOException)
{
  // push previous file on stack
  if (fp) {
    fps.push_back(fp);
    linenos.push_back(lineno);
    filenames.push_back(filename);
  }
  filename = fname;
  fp = fopen(filename.c_str(),"r");
  if (!fp) throw IOException(filename,"open",errno);
  lineno = 0;
}

void RelayParser::popfile() throw(EOFException)
{
  fclose(fp);
  fp = 0;
  int ifile = (signed)fps.size() - 1;
  // cerr << "popfile, ifile=" << ifile << endl;
  if (ifile < 0) throw EOFException(filename,"parse");
  fp = fps[ifile];
  lineno = linenos[ifile];
  filename = filenames[ifile];

  fps.pop_back();
  linenos.pop_back();
  filenames.pop_back();
}


void RelayParser::parse(const string& fname) throw(IOException,ParseException)
{
  pushfile(fname);

  for (;;) {
    try {
      readLine();
    }
    catch (EOFException& eof) {
      break;
    }

    char word[128];

    int nchar;
    if (sscanf(curptr,"%s%n",word,&nchar) < 1)
      throw ParseException(filename,
      	"expecting \"settings\", \"sequence\" or \"program\"",lineno);
    curptr += nchar;

    if (!strcmp(word,"settings")) parseSettings();
    else if (!strcmp(word,"sequence")) parseSequence();
    else if (!strcmp(word,"program")) parseProgram();
    else throw ParseException(filename,
    	"expecting \"settings\", \"sequence\" or \"program\"",lineno);
  }
}

void RelayParser::parseSettings()
	throw (IOException,ParseException) {

  // first line is just the word "settings", so check that rest
  // of line is just whitespace and a newline.

  if (!skipTilEOL())
    throw ParseException(filename,"error parsing settings",lineno);

  int nchar;
  for (;;) {
    /*
     * format: name followed by 0 and 1s
     * 1mfastflow              1  0  0  0  0  0  1  0  0  0  0  0  1  0  1
     */

    readLine();

    char word[128];

    if (sscanf(curptr,"%s%n",word,&nchar) < 1)
      throw ParseException(filename,"expecting a setting name",lineno);
    curptr += nchar;

    // scan ahead for "end settings"
    if (strcmp(word,"end") == 0) {
      char word2[128];
      if (sscanf(curptr,"%s%n",word2,&nchar) > 0 && !strcmp(word2,"settings")) {
        curptr += nchar;
	break;
      }
    }
    string settingName(word);

    vector<int> relayVec;
    int relayval;
    for (int i = 0; sscanf(curptr,"%d%n",&relayval,&nchar) > 0; i++) {
      relayVec.insert(relayVec.begin(),relayval);	// insert at beginning
      curptr += nchar;
    }

    RelaySetting setting(relayVec.size());
    setting.setName(settingName);
    for (unsigned int i = 0; i < relayVec.size(); i++) setting.setBit(i,relayVec[i]);

    if (!skipTilEOL())
      throw ParseException(filename,"error parsing settings",lineno);

    settings.insert(pair<string, RelaySetting>(settingName,setting));
  }
}

void RelayParser::parseSequence()
	throw (IOException,ParseException) {

  RelaySequence sequence;

  /*
   * format: "sequence" name
   * 	sequence test
   */

  int nchar;
  char name[128];
  if (sscanf(curptr,"%s%n",name,&nchar) < 1)
      throw ParseException(filename,"error parsing sequence name",lineno);
  curptr += nchar;

  sequence.setName(name);

  // rest of line should be empty, or comments
  if (!skipTilEOL())
    throw ParseException(filename,
    	string("error parsing sequence \"") + name + "\"",lineno);

  for (;;) {
    /*
     * format: name seconds
     * 1mfastflow    30
     */
    readLine();
    char word[128];

    if (sscanf(curptr,"%s%n",word,&nchar) < 1)
      throw ParseException(filename,"expecting a setting name",lineno);
    curptr += nchar;

    // check for "end sequence"
    if (!strcmp(word,"end")) {
      char word2[128];
      if (sscanf(curptr,"%s%n",word2,&nchar) > 0 && !strcmp(word2,"sequence")) {
	curptr += nchar;
        break;
      }
    }

    string settingName(word);

    map<string,RelaySetting>::iterator si = settings.find(settingName);
    if (si == settings.end())
      throw ParseException(filename,
	string("can't find setting named \"") + settingName + "\"", lineno);

    int seconds;
    int is;
    if ((is = sscanf(curptr,"%d%n",&seconds,&nchar)) < 1)
      throw ParseException(filename,"error parsing seconds field",lineno);
    curptr += nchar;

    if (!skipTilEOL())
      throw ParseException(filename,
	  string("error parsing sequence \"") + name + "\"",lineno);

    sequence.addSetting(si->second,seconds);
  }
  if (!skipTilEOL())
    throw ParseException(filename,
	string("error parsing sequence \"") + sequence.getName() + "\"",lineno);
  sequences.insert(pair<string,RelaySequence>(sequence.getName(),sequence));
}

void RelayParser::parseProgram()
	throw (IOException,ParseException)
{

  int nchar;
  char name[128];
  if (sscanf(curptr,"%s%n",name,&nchar) < 1)
      throw ParseException(filename,"error parsing sequence name",lineno);
  curptr += nchar;

  RelayProgram program;
  program.setName(name);

  if (!skipTilEOL())
    throw ParseException(filename,
    	string("error parsing program \"") + name + "\"",lineno);

  for (;;) {
    /*
     * format: "program" name
     * 	program test
     */

    readLine();

    char word[128];

    if (sscanf(curptr,"%s%n",word,&nchar) < 1)
      throw ParseException(filename,
      	"expecting a program entry, \"sequence\" sequence_name repeat_val",
		lineno);
    curptr += nchar;

    // check for "end program"
    if (!strcmp(word,"end")) {
      char word2[128];
      if (sscanf(curptr,"%s%n",word2,&nchar) > 0 && !strcmp(word2,"program")) {
	curptr += nchar;
        break;
      }
    }

    if (!strcmp(word,"sequence")) {
      if (sscanf(curptr,"%s%n",word,&nchar) < 1)
	throw ParseException(filename,
	  "expecting a program entry, \"sequence\" sequence_name repeat_val",
		lineno);
      curptr += nchar;

      string sequenceName(word);

      map<string,RelaySequence>::iterator si = sequences.find(sequenceName);
      if (si == sequences.end())
	throw ParseException(filename,
	  string("can't find sequence named \"") + sequenceName + "\"", lineno);

      int repeat;
      int is;
      if ((is = sscanf(curptr,"%d%n",&repeat,&nchar)) < 1)
	throw ParseException(filename,"error parsing repeat field",lineno);
      curptr += nchar;

      program.addSequence(si->second,repeat);

      if (!skipTilEOL())
	throw ParseException(filename,
	    string("error parsing program \"") + name + "\"",lineno);
    }
    else throw ParseException(filename,
    	string("unknown statement:") + word, lineno);
  }
  if (!skipTilEOL()) 
    throw ParseException(filename,
	string("error parsing program \"") + program.getName() + "\"",lineno);

  programs.insert(pair<string, RelayProgram>(program.getName(),program));
  programNames.push_back(program.getName());
}

/* static */
int RelayParser::usage(const char* argv0)
{
  cerr << "Usage:" << endl << argv0 << " filename " << endl;
  return 1;
}

/**
 * static main routine for testing.
 */
/* static */
int RelayParser::main(int argc, char** argv)
{
  if (argc < 2) return usage(argv[0]);

  const char* filename = argv[1];

  RelayParser parser;
  try {
    parser.parse(filename);
  }
  catch (Exception& e) {
    cerr << e.what() << endl;
  }

  std::vector<std::string> pnames = parser.getProgramNames();

  for (unsigned int i = 0; i < pnames.size(); i++) {
    RelayProgram& prog = parser.getProgram(pnames[i]);
    cerr << "program name=" << prog.getName() << " duration=" <<
    	prog.getDuration() << endl;
    do {
      cerr << "duration=" << prog.getSettingDuration() << endl;
      cerr << "relays=" << prog.getSetting().toString() << endl;
    } while(prog.step());
  }
  return 0;
}
