//
//              Copyright 2004 (C) by UCAR
//

#include <config.h>

#include <atdISFF/licor820/AirCOASampler.h>
#include <atdISFF/licor820/Licor820SensorPort.h>
#include <atdISFF/SampleFormatter.h>

#include <atdISFF/relays/RelayParser.h>

#ifdef ENABLE_DSCUD
#include <dsc/UserClockedPrometheusA2DSampler.h>
#include <dsc/DiamondUD.h>
#endif

#include <atdISFF/RawSampleSocket.h>
#include <atdISFF/RawSampleOutputStream.h>
#include <atdUtil/Logger.h>

#include <errno.h>

#ifdef GPP_2_95_2
#include <strstream>
#else
#include <sstream>
#endif

#include <iostream>

// for mlockall
#include <sys/mman.h>
#include <sys/resource.h>

using namespace std;
using namespace atdISFF;
using namespace atdUtil;
using namespace atdTermio;

/* static */
const string AirCOASampler::_version = "v1";

/* static */
const std::string& AirCOASampler::getVersion() { return _version; }

AirCOASampler::AirCOASampler():
	Sampler(),_name("aircoa"),_quit(false),
	_sphandler(0),_a2dsampler(0),_licorPrompter(0),
	_rsbuffer(0),_timer(0),
	_outputSocket(0),_sampleFormatter(0),_archiver(0),
	_outputSerialPort(0),_outputBuffer(0),
	_outputStream(0),_aircoaOutputStream(0),
	_relayProgram(0),_relayThread(0),_relay(0),
	_privileged(geteuid()==0),_sampleRate(1.0)
{
}

AirCOASampler::~AirCOASampler() {

  delete _timer;

  delete _relayThread;

  delete _relay;

  delete _licorPrompter;

  // cerr << "in ~AirCOASampler, deleting _sphandler" << endl;
  delete _sphandler;

  // cerr << "in ~AirCOASampler, deleting _a2dsampler" << endl;
  delete _a2dsampler;

  // cerr << "in ~AirCOASampler, deleting _rsbuffer" << endl;
  delete _rsbuffer;

  // cerr << "in ~AirCOASampler, deleting _sampleFormatter" << endl;
  delete _sampleFormatter;

  delete _archiver;

  // cerr << "in ~AirCOASampler, deleting _outputStream" << endl;
  delete _aircoaOutputStream;
  delete _outputStream;
  delete _outputBuffer;
  delete _outputSerialPort;

  delete _outputSocket;

}

void AirCOASampler::init() throw(Exception)
{

  int msecDeltat = (int)rint(1000 / getSampleRate());

  time(&_restartTime);

  _sampleFormatter = new SampleFormatter();
  _sampleFormatter->setDeltatMsec(msecDeltat);

  _rsbuffer = new RawSampleBuffer((int)(msecDeltat * 1.5));
  if (_privileged) _rsbuffer->setRealTimeFIFOPriority(40);
  _rsbuffer->blockSignal(SIGINT);
  _rsbuffer->blockSignal(SIGHUP);
  _rsbuffer->blockSignal(SIGTERM);
  _rsbuffer->setThreadSignalFactor(1);
  _rsbuffer->start();

  _timer = new TimerThread("AirCOA timer");
  if (_privileged) _timer->setRealTimeFIFOPriority(50);
  _timer->blockSignal(SIGINT);
  _timer->blockSignal(SIGHUP);
  _timer->blockSignal(SIGTERM);
  _timer->setWakeupIntervalMsec(msecDeltat);
  // don't start it yet.
  // std::cerr << "timer->getWakeupIntervalMsec=" << _timer->getWakeupIntervalMsec() << std::endl;

  startLicor820Serial("/dev/ttyS0");
  _sampleFormatter->addVariable(SampleFormatter::LICOR820,"co2",200);
  _sampleFormatter->addVariable(SampleFormatter::LICOR820,"cellpres",200);
  _sampleFormatter->addVariable(SampleFormatter::LICOR820,"celltemp",200);

  if (_relayProgram) {
    _relayProgram->setTimerThread(_timer);
    _relayProgram->addRawSampleClient(_rsbuffer);

    _relayThread =
      new ThreadRunnable("Relay",_relayProgram);
    if (_privileged) _relayThread->setRealTimeFIFOPriority(45);
    _relayThread->blockSignal(SIGINT);
    _relayThread->blockSignal(SIGHUP);
    _relayThread->blockSignal(SIGTERM);
    _relayThread->start();
  }
  _sampleFormatter->addVariable(SampleFormatter::RELAY,"relay",164);

  startA2D();

  // Now that we've added all the variables to the _sampleFormatter
  // we're ready to accept samples. Note that some samples
  // could be generated by the licor sensor before the
  // timer thread starts - if it responds to the initialization 
  // commands.
  _rsbuffer->addRawSampleClient(_sampleFormatter);

  // rrelay runs at non-real-time priority
  _rrelayListener = new RrelayListener(*_relayProgram,_sampleFormatter);
  _rrelayListener->blockSignal(SIGINT);
  _rrelayListener->blockSignal(SIGHUP);
  _rrelayListener->blockSignal(SIGTERM);
  _rrelayListener->start();

  if (_servername.size() > 0) {
    _outputSocket = new RawSampleSocket(_servername);
    _outputSocket->reopen();
    // send formatted samples on the outputSocket
    _sampleFormatter->addRawSampleClient(_outputSocket);
  }

  if (_ttyOutDev.size() > 0) {
    _outputSerialPort = new SerialPort(_ttyOutDev);
    _outputSerialPort->setBaudRate(9600);
    _outputSerialPort->setDataBits(8);
    _outputSerialPort->setStopBits(1);
    _outputSerialPort->setParity(atdTermio::SerialPort::NONE);
    _outputSerialPort->setFlowControl(atdTermio::SerialPort::NOFLOWCONTROL);
    _outputSerialPort->setRaw(false);
    _outputSerialPort->setLocal(true);
    _outputSerialPort->open(O_WRONLY | O_NOCTTY);

#if __GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ > 3)
    _outputBuffer = new __gnu_cxx::stdio_filebuf<char>(_outputSerialPort->fd(),
    	ios_base::out,128);
#else
    _outputBuffer = new __gnu_cxx::stdio_filebuf<char>(_outputSerialPort->fd(),
    	ios_base::out,false,128);
#endif

    _outputStream = new std::ostream(_outputBuffer);
    _aircoaOutputStream = new AsciiRawSampleOutputStream(*_outputStream);

    _sampleFormatter->addRawSampleClient(_aircoaOutputStream);
  }

  if (archiveDir.size() > 0) createArchiver();

  _timer->start();
}

void AirCOASampler::startLicor820Serial(const std::string& ttyDev)
	throw(Exception)
{
  _sphandler = new SensorPortHandler();
  if (_privileged) _sphandler->setRealTimeFIFOPriority(45);
  _sphandler->blockSignal(SIGINT);
  _sphandler->blockSignal(SIGHUP);
  _sphandler->blockSignal(SIGTERM);
  _sphandler->start();

  _licorSensor = new Licor820SensorPort(ttyDev,200);
  SerialPortConfig cfg;
  cfg.setChannel(200);
  cfg.setBaud(9600);
  cfg.setDataBits(8);
  cfg.setStopBits(1);
  cfg.setParity(atdTermio::SerialPort::NONE);
  cfg.setFlowControl(atdTermio::SerialPort::NOFLOWCONTROL);
  cfg.setRecordLength(0);
  cfg.setRecordDelimiter(string(1,'\n'));


  // SensorPortHandler will now own the SensorPort and
  // will close/delete it.
  _sphandler->addSensorPort(_licorSensor,cfg);
  _licorSensor->addRawSampleClient(_rsbuffer);

  if (_licor820InitFile.size() > 0) {
    _licorSensor->setInitFile(_licor820InitFile);
    /* _licorSensor->open does the licorSensor->init(), which
     * does the sendInitFile(). 
     * We don't want to send the init file until
     * the sphandler is ready to read the output. */
    _licorSensor->sendInitFile();
  }

  _licorPrompter = new Licor820Prompter(_licorSensor,_timer);
  if (_privileged) _licorPrompter->setRealTimeFIFOPriority(45);
  _licorPrompter->blockSignal(SIGINT);
  _licorPrompter->blockSignal(SIGHUP);
  _licorPrompter->blockSignal(SIGTERM);
  _licorPrompter->start();
}


#if defined(ENABLE_DSCUD)
void AirCOASampler::setIR104Relay(RelayProgram* relayProg, int ioport)
	throw(IOException)
{
  _relayProgram = relayProg;

  // ioport = 0x260;
  int irq = 0;
  _relay = new dsc::IR104(ioport,irq);

  _relayProgram->setRelay(_relay);
}
#endif

void AirCOASampler::startA2D() throw(atdUtil::Exception) {

  _a2dsampler = 0;

#ifdef ENABLE_DSCUD

#define A2D_IRQ 4
  int irq = A2D_IRQ;
  bool differential = false;
  bool bipolar = true;

  _a2dsampler = new dsc::UserClockedPrometheusA2DSampler(
  	0x280,irq, differential,bipolar,_timer);

#endif

  if (_a2dsampler) {
    _a2dsampler->open();
    if (_privileged) _a2dsampler->setRealTimeFIFOPriority(45);
    _a2dsampler->blockSignal(SIGINT);
    _a2dsampler->blockSignal(SIGHUP);
    _a2dsampler->blockSignal(SIGTERM);

    float rate = 1000.0f / _timer->getWakeupIntervalMsec();

    if (_a2dConfigFile.length() > 0) {
        FILE* ifp = fopen(_a2dConfigFile.c_str(),"r");
        if (ifp == NULL) throw IOException(_a2dConfigFile,"open",errno);

        int irec = 0;
        for (;;)
        {
                char vname[64];
                int chanId;
                float vMin,vMax;

                if (irec++ > 0) fscanf(ifp,"%*[^\n]");

                if (fscanf(ifp,"%63s",vname) != 1) {
                    if (feof(ifp)) break;
                    Logger::getInstance()->log(LOG_ERR,
                        "Error: %s: cannot parse line %d",
                            _a2dConfigFile.c_str(),irec);
                    break;
                }
                if (vname[0] == '#') continue;

                if (fscanf(ifp,"%d %f %f",&chanId,&vMin,&vMax) != 3) {
                    if (feof(ifp)) break;
                    Logger::getInstance()->log(LOG_ERR,
                        "Error: %s: cannot parse line %d",
                            _a2dConfigFile.c_str(),irec);
                    break;
                }
                addA2DChannel(vname,chanId,vMin,vMax,rate);
        }
        fclose(ifp);
    }
    else {
        addA2DChannel("T",108,-1.25,1.25,rate);
        addA2DChannel("RH",100,-1.25,1.25,rate);
        addA2DChannel("flow0",101,-5,5,rate);
        addA2DChannel("flow1",104,-5,5,rate);
        addA2DChannel("flow2",111,-5,5,rate);
        addA2DChannel("flow3",103,-5,5,rate);
        addA2DChannel("flow4",110,-5,5,rate);
    }

    _a2dsampler->addRawSampleClient(_rsbuffer);
    _a2dsampler->init();
    _a2dsampler->start();
  }
}

void AirCOASampler::setArchiveValues(std::string& aDir,
		std::string& aFormat,
		int aLengthSecs)
{
  archiveDir = aDir;
  archiveFormat = aFormat;
  archiveLengthSecs = aLengthSecs;
}

void AirCOASampler::createArchiver()
{
  _archiver = new AsciiSampleArchiver();
  _archiver->setDirectoryName(archiveDir);
  _archiver->setFileNameFormat(archiveFormat);
  _archiver->setFileLength(archiveLengthSecs);
  _sampleFormatter->addRawSampleClient(_archiver);
}

void AirCOASampler::addVariable(SampleFormatter::varType type,const string& name,int num,float a2dslope, float a2dintercept) {
}

void AirCOASampler::addA2DChannel(const string& name,int num,float vmin,
	float vmax, float rate) throw(atdUtil::IOException,atdUtil::InvalidParameterException) {

  // search for closest voltage range than includes vmin,vmax
  const vector<const float*> vranges = _a2dsampler->getAvailableVoltageRanges();
  float mindiff = 1.e10;
  int minvrindex = 0;
  for (unsigned int i = 0; i < vranges.size(); i++) {
    float vdiff = fabs(vranges[i][0] - vmin) + fabs(vranges[i][1] - vmax);
    if (vdiff < mindiff && vranges[i][0] <= vmin && vranges[i][1] >= vmax) {
      mindiff = vdiff;
      minvrindex = i;
    }
  }

  _a2dsampler->addChannel(num,minvrindex,rate);
  _sampleFormatter->addVariable(SampleFormatter::A2D,name,num,
  	_a2dsampler->getConversionSlope(num),
	_a2dsampler->getConversionIntercept(num));
}

float AirCOASampler::getMaximumA2DSamplingRate() const {
  return _a2dsampler->getMaximumSamplingRate();
}

const std::vector<const float*>& AirCOASampler::getAvailableA2DVoltageRanges() const {
  return _a2dsampler->getAvailableVoltageRanges();
}

/* static */
time_t AirCOASampler::_startTime = 0;

void AirCOASampler::quit() {
  // cerr << "in AirCOASampler::quit()" << endl;
  _quit = true;
  interrupt();
  // cerr << "end of AirCOASampler::quit()" << endl;
}
  
void AirCOASampler::wait() {
  // cerr << "in AirCOASampler::wait()" << endl;

  cerr << "timer->join()" << endl;
  _timer->join();

  cerr << "relayThread->join()" << endl;
  _relayThread->join();

  cerr << "licorPrompter->join()" << endl;
  _licorPrompter->join();

  cerr << "rrelayListener->join()" << endl;
  _rrelayListener->join();

  cerr << "sphandler->join()" << endl;
  _sphandler->join();

  // cerr << "in AirCOASampler::wait(), _sphandler joined" << endl;
  cerr << "a2dsampler->join()" << endl;
  if (_a2dsampler) _a2dsampler->join();

  cerr << "rsbuffer->join()" << endl;
  _rsbuffer->join();

  cerr << "end of AirCOASampler::wait()" << endl;

}
  
void AirCOASampler::interrupt() {

  _timer->interrupt();

  _licorSensor->removeRawSampleClient(_rsbuffer);
  _a2dsampler->removeRawSampleClient(_rsbuffer);
  if (_relayProgram) _relayProgram->removeRawSampleClient(_rsbuffer);

  _rsbuffer->removeRawSampleClient(_sampleFormatter);

  if (_archiver) _sampleFormatter->removeRawSampleClient(_archiver);

  if (_aircoaOutputStream)
    _sampleFormatter->removeRawSampleClient(_aircoaOutputStream);

  if (_outputSocket)
    _sampleFormatter->removeRawSampleClient(_outputSocket);

  _rrelayListener->kill(SIGUSR1);

  _relayThread->interrupt();

  _licorPrompter->interrupt();

  if (_a2dsampler) _a2dsampler->interrupt();

  _sphandler->kill(SIGUSR1);

  _rsbuffer->interrupt();

}

class Runstring {
public:
  Runstring(int argc, char** argv):
    syslogit(true),sampleRate(1.0)
{
    
    extern char *optarg;		/* set by getopt() */
    extern int optind;			/* "  "     "     */
    int opt_char;			/* option character */
    a2dConfigFile = default_a2dConfigFile;
    while ((opt_char = getopt(argc, argv, "a:c:er:R:s:St:x:")) != -1) {
      switch (opt_char) {
      case 'a':
	{
	  archiveDir = optarg;
	  // archive file name format must be next argument
	  if (optind == argc || argv[optind][0] == '-')
		  usage(argv[0]);
	  archiveFormat = argv[optind++];
	  // file length in secs is next arg
	  if (optind == argc || argv[optind][0] == '-')
		  usage(argv[0]);
	  archiveLengthSecs = atoi(argv[optind++]);
	}
	break;
      case 'c':
	a2dConfigFile = optarg;
        break;
      case 'e':
	syslogit = false;
	break;
      case 'r':
	relayProgFile = optarg;
	if (optind == argc) usage(argv[0]);
	relayProgName = argv[optind++];
	break;
      case 'R':
	{
	    istringstream ist(optarg);
	    ist >> sampleRate;
	    if (ist.fail()) usage(argv[0]);
	}
	break;
      case 's':
	nodeName = optarg;
	if (optind == argc) usage(argv[0]);
	serverName = argv[optind++];
	break;
      case 'S':
	syslogit = true;
	break;
      case 't':
	ttyOutDev = optarg;
	// if (optind == argc) usage(argv[0]);
	// ttyOptions = argv[optind++];
	break;
      case 'x':
	xmlInitFile = optarg;
	break;
      case '?':
	usage(argv[0]);
	break;
      }
    }
  }

  string relayProgFile;
  string relayProgName;
  string nodeName;
  string serverName;
  string xmlInitFile;
  string ttyOutDev;
  static string default_a2dConfigFile;
  string a2dConfigFile;
  // string ttyOptions;
  bool syslogit;
  float sampleRate;

  string archiveDir;
  string archiveFormat;
  int archiveLengthSecs;

  static void usage(const char* argv0) {
    cerr << "Usage: " << argv0 << "[-a adir afile alen] [-c a2dconfig] [-e]\n\
    	-r relayFileName relayProgName [-R rate]\n\
  [-s nodename servername] [-t ttyOutDev]\n\
  [-x licor820XMLInitfile]" << endl;
    cerr << "\
  -a adir afile alen: archive directory, file and file length(secs)\n\
      file can contain strftime time format descriptors (do man strftime)\n\
      Example: -a /data cme_%Y%m%d_%H%M%S.dat 86400\n\
        creates day-long files named like /data/cme_20050519_000000.dat\n\
  -c a2dconfig\n\
      file containing a space-separated A2D configuration like so:\n\
      varname A2Dchannel minV maxV\n\
      Default file is " << default_a2dConfigFile << "\n\
      Example contents:\n\
T       8  -1.25  1.25\n\
RH      0  -1.25  1.25\n\
flow0   1  -5     5\n\
flow1   4  -5     5\n\
flow2  11  -5     5\n\
flow3   3  -5     5\n\
flow4  10  -5     5\n\
\n\
  -e: status messages to stderr, otherwise to syslog\n\
  -r: read relayFile, execute relayProg to set relays on IR104\n\
  -R rate: sample rate in samples/sec, default 1.0\n\
  -s: create ISFF socket from this nodename to servername (optional)\n\
  -t: generate output on tty (optional).\n\
      -s and/or -t or neither can be specified\n\
  -x: send contents of XML as initialization for licor820 (optional)\n\
  " << endl;
    exit(1);
  }
};

/* static */
string  Runstring::default_a2dConfigFile =
    "/home/isff/aircoa/aircoa_config.txt";

/* static */
int AirCOASampler::main(int argc, char** argv) {

  Runstring runstring(argc, argv);

  if (runstring.syslogit)
    Logger::createInstance("AirCOASampler",LOG_CONS | LOG_PID, LOG_LOCAL4,(char *)0);
  else Logger::createInstance(stderr);

  int status = 0;
  time(&AirCOASampler::_startTime);

  try {
    bool quit = false;

    while (!quit) {
      // RelayParser owns the RelayProgram, so we
      // don't want to delete the parser until the
      // Sampler is finished.
      RelayParser* relayParser = new RelayParser();
      relayParser->parse(runstring.relayProgFile);
      RelayProgram* relayProgram = &relayParser->getProgram(
	      runstring.relayProgName);

      AirCOASampler* sampler = new AirCOASampler();

      if (runstring.serverName.size() > 0) {
        sampler->setDataServerName(runstring.serverName);
        sampler->setNodeName(runstring.nodeName);
      }
      if (runstring.ttyOutDev.size() > 0)
	sampler->setTtyOutDev(runstring.ttyOutDev);

      sampler->setLicor820InitFile(runstring.xmlInitFile);

      sampler->setA2DConfigFile(runstring.a2dConfigFile);

#ifdef ENABLE_DSCUD
      sampler->setIR104Relay(relayProgram,0x260);
#endif

      sampler->setSampleRate(runstring.sampleRate);

      if (runstring.archiveDir.size() > 0)
        sampler->setArchiveValues(runstring.archiveDir,
		runstring.archiveFormat,
		runstring.archiveLengthSecs);

      sampler->init();

      struct timespec stime;
      stime.tv_sec = 2;
      stime.tv_nsec = 0;
      nanosleep(&stime,0);

      if (mlockall(MCL_CURRENT) < 0) {
	Exception e("mlockall",Exception::errnoToString(errno));
	Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());

	struct rlimit mlimit;
	if (getrlimit(RLIMIT_MEMLOCK,&mlimit) < 0) {
	  Exception e("getrlimit",Exception::errnoToString(errno));
	  Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());
	}
	else
	  Logger::getInstance()->log(LOG_INFO,"getrlimit RLIMIT_MEMLOCK: soft limit=%d, hard limit=%d",mlimit.rlim_cur,mlimit.rlim_max);
      }

      sampler->wait();
      quit = sampler->getQuit();
      delete sampler;
      delete relayParser;
      if (!quit) sleep(5);
      if (munlockall() < 0) {
	Exception e("munlockall",Exception::errnoToString(errno));
	Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());
      }

    }
#if defined(ENABLE_DSCUD)
    dsc::DiamondUD::freeUD();
#endif
  }
  catch (UnknownHostException &uhe) {
    cerr << "UnknownHostException in main: " << uhe.toString() << endl;
    status = 1;
  }
  catch (IOException &se) {
    cerr << "IOException in main: " << se.toString() << endl;
    status = 1;
  }
  catch (InvalidParameterException &ipe) {
    cerr << "InvalidParameterException in main: " << ipe.toString() << endl;
    status = 1;
  }
  catch (Exception &e) {
    cerr << "Exception in main: " << e.toString() << endl;
    status = 1;
  }
  return status;
}
