//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/TimerThread.h>

#include <iostream>
// #include <unistd.h>	// usleep
#include <time.h>	// nanosleep
#include <math.h>
#include <errno.h>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

/**
 * Constructor for TimerThread. Default sleep value of 1000 mseconds.
 */
TimerThread::TimerThread(const string& name) : Thread(name),
	mSecSleep(1000)
{
  blockSignal(SIGINT);
  blockSignal(SIGHUP);
  blockSignal(SIGTERM);
}

TimerThread::~TimerThread()
{
}

void TimerThread::setWakeupIntervalMsec(unsigned int msec)
{
  mSecSleep = msec;
}

unsigned int TimerThread::getWakeupIntervalMsec() const
{
  return mSecSleep;
}

void TimerThread::setWakeupRate(float rate)
{
  mSecSleep = (unsigned int) rintf(1000.0f / rate);
}

float TimerThread::getWakeupRate() const
{
  return 1000.0f / mSecSleep;
}

void TimerThread::interrupt()
{
  Synchronized autosync(timerCond);
  Thread::interrupt();
  timerCond.broadcast();
}
int TimerThread::run() throw(atdUtil::Exception) {

  struct timespec sleepTime;

  /* sleep a bit so that we're on an even interval boundary */
  unsigned long mSecVal =
    mSecSleep - (unsigned long)(currentTimeMsec() % mSecSleep);

  sleepTime.tv_sec = mSecVal / 1000;
  sleepTime.tv_nsec = (mSecVal % 1000) * 1000000;
  if (nanosleep(&sleepTime,0) < 0) {
    if (errno == EINTR) return RUN_OK;
    throw Exception(string("nanosleep: ") + Exception::errnoToString(errno));
  }

  while (!amInterrupted()) {
    timerCond.lock();
    timerCond.broadcast();
    timerCond.unlock();
    mSecVal =
      mSecSleep - (unsigned long)(currentTimeMsec() % mSecSleep);
    sleepTime.tv_sec = mSecVal / 1000;
    sleepTime.tv_nsec = (mSecVal % 1000) * 1000000;
    if (nanosleep(&sleepTime,0) < 0) {
      if (errno == EINTR) break;
      throw Exception(string("nanosleep: ") + Exception::errnoToString(errno));
    }
  }
  cerr << "TimerThread run method returning, doing broadcast" << endl;
  // after a wait, clients should check that this thread interrupted.
  timerCond.lock();
  timerCond.broadcast();
  timerCond.unlock();
  return RUN_OK;
}
