//
//              Copyright 2004 (C) by UCAR
//

#ifndef ATDISFF_TIMETAGSMOOTHER_H
#define ATDISFF_TIMETAGSMOOTHER_H

#include <config.h>

#include <atdISFF/Time.h>

// #include <sys/time.h>
// #include <math.h>
#include <stdlib.h>

#include <iostream>

namespace atdISFF {

/**
 * Simple time tag smoother.
 * Smoothed time tags are generated using two
 * running averages computed from input, raw time tags.
 * One running average is kept of the differences
 * between the input time tags. This average
 * delta-T (adjusted as described below) is added
 * successively to generate smoothed time tags.
 *
 * Another shorter running average is kept of
 * the differences between these generated time
 * tags and the input raw time tags. A fraction
 * of this running difference (currently 10%)
 * is added to the generated time tags.
 */
class TimeTagSmoother
{
public:
    TimeTagSmoother():
    	T0_STATS(10),DT_STATS(100),TERR_STATS(10)
    {
        reset();
    }


    /**
     * Get a smoothed time tag.
     * @param t Raw time tag
     * @nsamp Number of individual samples between calls
     *  to getTime.
     * @deltat Delta-T for individual samples, in milliseconds.
     *  Initialize this value with the expected delta T.
     *  The returned value will be an estimate calculated
     *  from running averages of the differences of t.
     *
     * If getTime is called with the leading time tag of
     * a block of samples, as from an A2D, then nsamp
     * is the number of sub-samples in the block. Initialize
     * deltat with the expected delta T of the sub-samples.
     * getTime will return an adjusted deltat for the
     * sub-samples.
     */
    isff_sys_time_t getTime(isff_sys_time_t t,int nsamp,double & deltat);

    static int test(double var);

private:

    const int T0_STATS;
    const int DT_STATS;
    const int TERR_STATS;

    /**
     * Sum of T0_STATS initial times. Used in the  computation
     * if an estimate of t0 (initial time) from the initial times.
     */
    isff_sys_time_t tsum;
    int isum;
    isff_sys_time_t t0;

    double dtavg;
    double dtsave;
    ssize_t ntime;
    isff_sys_time_t tlast;
    isff_sys_time_t tcor;

    double terravg;
    int nerravg;

    inline void updateDtAvg(isff_sys_time_t t)
    {
	double dt = t - tlast;
	int istat;
	if (ntime < DT_STATS) istat = ntime++;
	else istat = DT_STATS;
	dtavg = (dtavg * (istat - 1) + dt) / istat;
	tlast = t;
    }

    inline void updateTerrAvg(double terr)
    {
	int istat;
	if (nerravg < TERR_STATS) istat = ++nerravg;
	else istat = TERR_STATS;
	terravg = (terravg * (istat - 1) + terr) / istat;
    }

    void reset()
    {
    	tsum = 0;
	isum = 0;
	dtavg = 0;
	ntime = 0;
	terravg = 0;
	nerravg = 0;
    }


};

}

#endif
