//
//              Copyright 2004 (C) by UCAR
//

#ifndef ATDISFF_TIME_H
#define ATDISFF_TIME_H

#include <config.h>

#include <sys/time.h>
#include <math.h>

namespace atdISFF {

#ifdef TIME_IS_LONG_LONG
  typedef long long isff_sys_time_t;
#else
  typedef double isff_sys_time_t;
#endif

#ifdef TIME_IS_LONG_LONG
  /**
   * Return the current unix system time, in milliseconds.
   */
  static isff_sys_time_t getCurrentTimeInMillis() {
    struct timeval tval;
    if (::gettimeofday(&tval,0) < 0) return 0L;   // shouldn't happen
    return (isff_sys_time_t)(tval.tv_sec) * 1000 +
    	(tval.tv_usec + 500) / 1000;
  }
#else
  /**
   * Return the current unix system time, in milliseconds.
   */
  static isff_sys_time_t getCurrentTimeInMillis() {
    struct timeval tval;
    if (::gettimeofday(&tval,0) < 0) return 0.0;   // shouldn't happen
    return (isff_sys_time_t)tval.tv_sec * 1000.0 + tval.tv_usec / 1000.0;
  }
#endif

  typedef unsigned long sample_time_t;

  /**
   * Inline function to create a sample_time_t from a isff_sys_time_t
   * plus an offset in milliseconds.
   */
  inline sample_time_t sample_time(isff_sys_time_t t,double offset) {
#ifdef TIME_IS_LONG_LONG
    return (sample_time_t)((t + (long)rint(offset)) % 86400000L);
#else
    return (sample_time_t)rint(fmod(t + offset,86400000.0));
#endif
  }

  inline sample_time_t sample_time(isff_sys_time_t t,float offset) {
#ifdef TIME_IS_LONG_LONG
    return (sample_time_t)((t + (long)rintf(offset)) % 86400000L);
#else
    return (sample_time_t)rint(fmod(t + offset,86400000.0));
#endif
  }

  inline sample_time_t sample_time(isff_sys_time_t t,int offset) {
#ifdef TIME_IS_LONG_LONG
    return (sample_time_t)((t + offset) % 86400000L);
#else
    return (sample_time_t)rint(fmod(t + offset,86400000.0));
#endif
  }

  inline isff_sys_time_t isff_sys_time_add(isff_sys_time_t t,double offset) {
#ifdef TIME_IS_LONG_LONG
    return t + (long)rint(offset);
#else
    return t + offset;
#endif
  }

  /**
   * compute a difference between two time tags.
   * Result in the range -43200000:43200000;
   */
  inline signed long sample_time_diff(sample_time_t t1, sample_time_t t2) {
    long diff = t1 - t2;
    if (diff > 43200000) diff -= 86400000;
    else if (diff <= -43200000) diff += 86400000;
    return diff;
  }

  inline isff_sys_time_t next_isff_sys_time(isff_sys_time_t t,unsigned long period) {
#ifdef TIME_IS_LONG_LONG
    return ((t / period) + 1) * period;
#else
    return (floor(t / period) + 1.0) * period;
#endif
  }

}

#endif
