#ifndef STEPPER_H_
#define STEPPER_H_

///The number of motors we are controlling
#define NMOTORS 3
/// ioport line number for universal control line for logic +5 feed to all
#define LOGICONOFF 12

#include <vector>
#include <atdUtil/Thread.h>
#include <atdUtil/IOException.h>
#include <dsc/PromDigitalIO.h>

namespace atdISFF {

  /// Control three stepper motors from an independent thread.
  ///
  /// Note: The selected motor number ranges from 1 to NMOTORS, in order
  /// to match the labelling on the hardware. Therefore, all indexing
  /// based on motor number will subtract one from the selected
  /// motor.
  ///
  /// There are two halves to this class:
  /// 1.) One half will be called from the ndaqmonitor thread,
  ///     initiated by an RPC call on a remote system.
  ///     These are the functions found in the lower half of
  ///     Stepper.cpp. They either immediately return a current 
  ///     property (such as status or position) of the stepper 
  ///     system, or they store the requested command, which 
  ///     will be executed by a different thread. The point is
  ///     to allow ndaqmonitor to continue executing, without
  ///     having it stall while we wait for motors to move, etc.
  ///     The member functions that execute in the ndaq thread
  ///     are doCmd(), status(), position(), move(), zero() and
  ///     reset(). Remember, these are called from the ndaqmonitor,
  ///     and either return a property immediately, or schedule
  ///     a command to be executed later.
  ///     
  /// 2.) The second half of this class is found in the upper
  ///     half of Stepper.cpp. Examine the run() function. This
  ///     function is executed as an independent thread. It 
  ///     waits for commands to be registered by ndaqmonitor, and
  ///     then executes them. The status result of all of these
  ///     command  is saved in _status[_motor-1]. 
  ///     The functions that implement the queued coammands, called 
  ///     from run(), are MoveCmd(), ZeroCmd(), and ResetCmd(). Note 
  ///     that these commands will operate on the currently selected
  ///     motor, which is saved in _motor.
  ///     Note that each of these internal methods will run to 
  ///     completion before returning to the run() loop, and so 
  ///     they must not get into infinite loops. Stands to reason.
  ///
  ///
  /// If an RPC command is not accepted, query the status to find out why. 
  /// In general, if the status is STEPPERSTATUS_IDLE, a command 
  /// will be acccepted. However, the user should always verify that
  /// the status is STEPPERSTATUS_IDLE, before registering a command. 
  /// 
  class Stepper: public atdUtil::Thread
    {
  
    public:
      /// The current state of the stepper
      typedef enum {
	/// Stepper is idle and ready to acept a command
	STEPPERSTATUS_IDLE, 
	/// Stepper is busy executing a command
	STEPPERSTATUS_BUSY, 
	/// Stepper encountered a fault. Try reset() to clear the fault
	STEPPERSTATUS_FAULT} StepperStatus;

      /// The current command being executed
      enum StepperCommand {
	/// Done
	STEPPERCMD_DONE,
	/// Move
	STEPPERCMD_MOVE,
	/// Zero
	STEPPERCMD_ZERO,
	/// Reset
	STEPPERCMD_RESET,
	/// get the status
	STEPPERCMD_STATUS
      };

      /// 
      /// _currentCommand can only be changed when _status is STEPPERSTATUS_IDLE.
      ///
      ///
      /// Functions which return a status will indicate that the device is 
      /// either idle, busy, or in a fault state. When the stepper is commanded
      /// to move, either by a reset(), zero() or move() command, a status 
      /// of busy, idle, or fault may be returned.
      ///
    public:
      ///
      Stepper(int ioport, int irq);
      ///
      ~Stepper();
      ///
      /// Run the polling loop which will respond to
      /// commands queued by the doCmd() function
      int run() throw(atdUtil::Exception);
      ///
      /// @return The current status
      StepperStatus status(int motorNumber);
      ///
      /// Register the move command to be executed by the run() loop.
      /// @return True if the command was accepted, false otherwise
      bool move(int motorNumber, double step);
      ///
      /// Register the zero command to be executed by the run() loop.
      /// @return True if the command was accepted, false otherwise
      bool zero(int motorNumber);
      ///    
      /// Register the reset command to be executed by the run() loop.
      /// @return True if the command was accepted, false otherwise
      bool reset(int motorNumber);
      ///
      /// Get the current position
      double position(int motorNumber);

    protected:
      ///
      void init(int ioport, int irq) throw(atdUtil::IOException);
      /// 
      //// Register any command to be executed by the run() loop.
      /// If it is a status command, just return the status without
      /// registering a command.
      bool doCmd(int motorNumber, StepperCommand cmd, double arg);
      bool doCmd(int motorNumber, StepperCommand cmd);
      ///
      /// Reset the position
      StepperStatus ResetCmd();
      ///
      /// Zero the position
      StepperStatus ZeroCmd();
      ///
      /// Move the probe
      StepperStatus MoveCmd(double step);
      ///
      /// Step the probe
      StepperStatus StepCmd(double step);
      ///
      void close();
      ///
      dsc::PromDigitalOut _out;
      ///
      dsc::PromDigitalIn  _in;
      ///
      atdUtil::BitArray   _outMask;
      ///
      atdUtil::BitArray   _inMask;
      ///
      atdUtil::BitArray   _outMotor;
      /// Our current status, for each motor
      std::vector<StepperStatus> _status;
      /// The current position, for each motor
      std::vector<double> _position;
      /// The pending command
      StepperCommand _currentCommand;
      /// Mutex to protect access for everything that needs protection
      atdUtil::Mutex _lock;
      /// The argument for the command
      double _cmdArg;
      /// set true by the command processor to force a reset of the status
      bool _resetFlag;
      /// The motor number that the current command applies to.
      int _motor; 

      struct LineDefs {	// NOTE the line#'s are 1 less than physical pin#'s
	int clock;
	int ccwlimit;
	int center;
	int cwlimit;
	int onoff;
	int direction;
      } _ioPins[NMOTORS];

    };
}

#endif

