// Drive the motor of the hotfilm probe
// I/O lines of the Prometheus are used to set motor direction and drive
// the motor 
// Other I/O lines are used to detect when the probe is at the maximum
// point for CCW and CW, as well as when the Center Position is reached. 
//
//
// Modifications:
// 22Feb06,jwm	- added struct for motor io lines
//		- modified the 'zero position' to use 'center' position
//		- modified the move routine to assume integer 'steps'
//		  based upon the double passed in.   NOTE this is yet to
//		  be firmed up and converted to degrees.  For now the
//		  divider circuit is working on divide counts of 1024
//		  resulting in ~.44 degrees per pules.  NOTE that this
//		  was working with an input drive voltage of 3.2
//		  resulting in ~19hz pulses.....don't know if this'd
//		  work with 5v input, and then we'd need the 4096
//		  divisor tap on the interface.
//		- added in _position values based on passed 'step'
//
// I/O Port definitions
// CORRECTED PER ACTUAL INTERFACE BOARD
// PortA,C=input	PORT-Physical-Pin-Numbers
// 			motor 1		motor 2		motor 3		
// Clock		C4-21		A4-5		A3-4	
// Left (CCW)		C6-23		A6-7		A1-2
// Center		C0-17		A0-1		A7-8
// Right (CW)		C2-19		A2-3		A5-6
//
// Motor on/off		B7-16		B3-12		B1-10
// Direction		B6-15		B2-11		B0-9
// Logic on/off	 	B4-13


// arguments get pass to the program to define if the probe needs to be
// moved to the zero position or moved to a new position. If new postion
// then the direction flag is also provided.
// If no arguments are passed then the program aborts
// argv[1]: position steps (sign determines direction

using namespace std;

#include "Stepper.h"
#include <iostream>
#include <unistd.h>
#include <time.h>
#include <math.h>

namespace atdISFF {

  //////////////////////////////////////////////////////////////////////////////////
  //
  // Constructors and initialization
  //
  //
  //--------------------------------------------------------------------------------
  Stepper::Stepper(int ioport, int irq):
    Thread("Stepper"),
    _out(ioport, irq),
    _in(ioport, irq),
    _outMask(_out.getNumberOfOutputs()),
    _inMask(_in.getNumberOfInputs()),
    _outMotor(_out.getNumberOfOutputs()),
    _currentCommand(STEPPERCMD_DONE),
    _lock("stepper"),
    _cmdArg(STEPPERCMD_DONE),
    _resetFlag(false),
    _motor(1)
  {

    // initialize status and position
    _status.resize(NMOTORS);
    _position.resize(NMOTORS);
    for (unsigned int i = 0; i < NMOTORS; i++) {
      _status[i] = STEPPERSTATUS_IDLE;
      _position[i] = 0.0;
    }
    
    // Set up the map of IO pins. This is ugly,
    // since it assumes NMOTORS == 3. If the number 
    // of motors ever becomes greater than 3, new
    // IO definitions will have to be added.
    if (NMOTORS > 0) {
      _ioPins[0].clock     = 20;
      _ioPins[0].ccwlimit  = 22;
      _ioPins[0].center    = 16;
      _ioPins[0].cwlimit   = 18;
      _ioPins[0].onoff     = 15;
      _ioPins[0].direction = 14;
    }

    if (NMOTORS > 1) {
      _ioPins[1].clock     = 4;
      _ioPins[1].ccwlimit  = 6;
      _ioPins[1].center    = 0;
      _ioPins[1].cwlimit   = 2;
      _ioPins[1].onoff     = 11;
      _ioPins[1].direction = 10;
    }

    if (NMOTORS > 2) {
      _ioPins[2].clock     = 3;
      _ioPins[2].ccwlimit  = 1;
      _ioPins[2].center    = 7;
      _ioPins[2].cwlimit   = 5;
      _ioPins[2].onoff     = 9;
      _ioPins[2].direction = 8;
    }

    // initialize the ioport
    init(ioport, irq);
    
    cout << "Stepper created\n";
  }

  //--------------------------------------------------------------------------------
  Stepper::~Stepper()
  {
    cout << "~Stepper called" << endl;
    _out.close();
    _in.close();
    cout << "~Stepper finished" << endl;
  }

  //--------------------------------------------------------------------------------
  // 
  void 
  Stepper::init(int ioport, int irq) throw(atdUtil::IOException)
  {
    cout << "_out.open" << endl;
    _out.open();
    cout << "_out.open done" << endl;

    cout << "_in.open" << endl;
    _in.open();

    cout << "_out.getNumberOfOutputs" << endl;
    _outMask.setBits(8,16,0xff);	// PORT-B
    _out.reserveOutputs(_outMask);

    cout << "Hello: Initializing Stepper" << endl;

    _inMask.setBits(0,8,0xff);		// PORT-A
    _inMask.setBits(16,24,0xff);	// PORT-C
    _in.reserveInputs(_inMask);
    for (unsigned int i = 0; i < NMOTORS; i++)
      _status[i] = STEPPERSTATUS_IDLE;
    cout << "init done" << endl;
  }
  ////////////////////////////////////////////////////////////////////////////////
  //
  // This section contains the run() loop, and the functions called by it to execute
  // a command request.
  //
  //--------------------------------------------------------------------------------
  int
  Stepper::run() throw(atdUtil::Exception)
  {
    // Main thread loop. Cycle here, waiting for a command to come in.
    // When the command is received, set our status to busy, excute the 
    // command, and then set status back to idle.

    struct timespec sleep200msec = { 0, 200000000 };

    for (;;) {
      // sleep for awhile. We don't want this loop using up the whole cpu
      nanosleep(&sleep200msec,0);

      // See if a command has been registered
      _lock.lock();
      StepperCommand nextCommand = _currentCommand;

      if (_resetFlag) {
	nextCommand = STEPPERCMD_RESET;
	_resetFlag = false;
      }
      _lock.unlock();

      if (nextCommand != STEPPERCMD_DONE) {

	// we got a command request. Execute it.
	// The command handler must return the value that 
	// status should be set to.

	StepperStatus cmdStatus = STEPPERSTATUS_IDLE;
	switch (nextCommand) {
	case STEPPERCMD_RESET: {
	  cmdStatus = ResetCmd();
	  break;
	}
	case STEPPERCMD_ZERO: {
	  cmdStatus = ZeroCmd();
	  break;
	}
	case STEPPERCMD_MOVE: {
	  cmdStatus = MoveCmd(_cmdArg);
	  break;
	}
	default: {
	  break;
	}
	}

	_lock.lock();
	_status[_motor-1] = cmdStatus;
	_currentCommand = STEPPERCMD_DONE;
	_lock.unlock();
      }
    }
  }

  //--------------------------------------------------------------------------------
  // StepCmd: function increment a motor to the desired location
  // input: the number of steps to make
  // output: 0 -> ok, -1 -> max switch turned on
  atdISFF::Stepper::StepperStatus
  Stepper::StepCmd(double Step)
  {
    int icw=0;
    int iccw=0;

    struct timespec sleep2msec = { 0, 2000000 };
    // initialize the return status to idle. It may
    // end up being something else
    StepperStatus status = STEPPERSTATUS_IDLE;
	
    atdUtil::BitArray inMotor(_out.getNumberOfOutputs());
	
    cout << "stepprobe for motor "<< _motor << endl;

    // turn on logic and then motor Vcc on
    _outMotor.setBit(LOGICONOFF,true);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(_ioPins[_motor-1].onoff,true);
    _out.write(_outMask,_outMotor);

    // First move the probe to one clock pulse before the final postion.
    // This is done so that we will stop immediately upon the rising
    // edge of the last pulse rather than waiting for the succeeding
    // 'low-level' as is done in this loop....perhaps this can be 
    // improved and condensed into 1 loop at some point....
    // While moving we need to check the limit switches to ensure
    // we do not burn up the motor.
    for (int i = (int) Step; i > 1;  --i ) {
      do {
	inMotor = _in.read(_inMask);		// read the input lines
	//cout << "stepprobe" << endl <<inMotor.toString() << endl;

	//  check to make sure we have not maxed out movement range
	// if so stop motor and return fault
	icw= inMotor.getBit(_ioPins[_motor-1].cwlimit);
	iccw=inMotor.getBit(_ioPins[_motor-1].ccwlimit); 
	if ( icw || iccw ) {
	  // turn motor off
	  _outMotor.setBit(_ioPins[_motor-1].onoff,false);
	  _out.write(_outMask,_outMotor);
	  _outMotor.setBit(LOGICONOFF,false);
	  _out.write(_outMask,_outMotor);
	  cout << "LimitHit: cw="<<icw<<" ccw="<<iccw << endl;
	  return STEPPERSTATUS_FAULT;
	}
	// wait for a clock signal to go high
	nanosleep(&sleep2msec,0);
      }  while(!(inMotor.getBit(_ioPins[_motor-1].clock)));	

      // now read the input lines and wait for the clock signal to go low
      do {
	inMotor = _in.read(_inMask);
	nanosleep(&sleep2msec,0);
      } while(inMotor.getBit(_ioPins[_motor-1].clock));
      //cout << "clock-low; step-" << inMotor.toString() << endl;
    }

    // Finally, move the probe to the last postion, stopping as
    // soon as the clock signal goes high.
    do {
      inMotor = _in.read(_inMask);		// read the input lines
      //cout << "Stepprobe (last step):" << endl << inMotor.toString() << endl;

      // check to make sure we have not maxed out movement range
      // if so stop motor and return -1
      if ((inMotor.getBit(_ioPins[_motor-1].ccwlimit)) ||
	  (inMotor.getBit(_ioPins[_motor-1].cwlimit))) {

	// turn motor off
	_outMotor.setBit(_ioPins[_motor-1].onoff,false);
	_out.write(_outMask,_outMotor);
	_outMotor.setBit(LOGICONOFF,false);
	_out.write(_outMask,_outMotor);
	return STEPPERSTATUS_FAULT;
      }
      // wait for a clock signal to go high
      nanosleep(&sleep2msec,0);
    } while(!(inMotor.getBit(_ioPins[_motor-1].clock)));	
	
    // turn motor off
    _outMotor.setBit(_ioPins[_motor-1].onoff,false);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(LOGICONOFF,false);
    _out.write(_outMask,_outMotor);

    // Update current position value
    _position[_motor-1] += Step;

    return status;
  }

  //--------------------------------------------------------------------------------
  // ZeroCmd:
  // Function moves probe to the starting position which is defined
  // as parallel with the sonic boom.
  // The probe is moved in a CW direction until either the 'Center' signal
  // is received in which case we're done; or until the CWstop signal is received
  // and then the motor direction is changed to CCW and the probe is moved back
  // until the 'Center' signal is found.
  // The number of clock signals is determined
  // by the returned pulses between 'CWstop' and 'Center'

  atdISFF::Stepper::StepperStatus
  Stepper::ZeroCmd()
  {	

    struct timespec sleep2msec = { 0, 2000000 };

    cout << "stepper zero for motor " << _motor << endl;
    StepperStatus status = STEPPERSTATUS_IDLE;
    atdUtil::BitArray inMotor(_in.getNumberOfInputs());

    int iccw=0;
    int icw=0;
    int ic=0;

    // set motor direction CW
    _outMotor.setBit(_ioPins[_motor-1].direction,true);
    _out.write(_outMask,_outMotor);

    // turn motor on, setting logic line first and then vcc
    _outMotor.setBit(LOGICONOFF,true);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(_ioPins[_motor-1].onoff,true);
    _out.write(_outMask,_outMotor);	

    // read the input lines and test until either the center
    // position or the CWStop is detected
    do {
      inMotor = _in.read(_inMask);
      ic=  inMotor.getBit(_ioPins[_motor-1].center);
      icw=inMotor.getBit(_ioPins[_motor-1].cwlimit); 
      nanosleep(&sleep2msec,0);
    }  while( !ic && !icw );

    // turn motor off
    _outMotor.setBit(_ioPins[_motor-1].onoff,false);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(LOGICONOFF,false);
    _out.write(_outMask,_outMotor);

    // if hit center THAT's IT!
    if ( ic==1 && !icw ) {	
      _position[_motor-1] = 0.0;
      cout << "Zero-Position centertriggered" << endl;
      // status will be STEPPERSTATUS_IDLE:
      return status;	
    }

    cout << "     CW EndLimit:" << endl;

    // now move the probe in the other direction 
    // to the Zero position

    // turn motor on
    _outMotor.setBit(_ioPins[_motor-1].direction,false);
    _out.write(_outMask,_outMotor);
    sleep(1);

    cout << "Move Probe back to Zero Position" << endl;
    _outMotor.setBit(LOGICONOFF,true);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(_ioPins[_motor-1].onoff,true);
    _out.write(_outMask,_outMotor);

    // Read the input lines and test until the 
    // CenterSwitch is detected
    do {
      // read the input lines
      inMotor = _in.read(_inMask);		
      ic=  inMotor.getBit(_ioPins[_motor-1].center);
      iccw=inMotor.getBit(_ioPins[_motor-1].ccwlimit); 
      nanosleep(&sleep2msec,0);
    } while( !ic && !iccw );

    // turn off motor
    _outMotor.setBit(_ioPins[_motor-1].onoff,false);
    _out.write(_outMask,_outMotor);
    _outMotor.setBit(LOGICONOFF,false);
    _out.write(_outMask,_outMotor);

    _position[_motor-1] = 0.0;
    cout << "Zero-Position; " << ic << " ccwlimit=" << iccw << endl;

    return status;
  }

  //--------------------------------------------------------------------------------
  // MoveCmd:  function moves the probe based on input parameters
  // Dir: "CW" or "CCW" (direction to move)
  // Step; number of steps to move
  atdISFF::Stepper::StepperStatus
  Stepper::MoveCmd(double Step)
  {

    cout << "stepper move " << Step << " for motor " << _motor << endl;

    StepperStatus status = STEPPERSTATUS_IDLE;
    bool dir = true;

    if (Step < 0)
      dir = false;				// set the direction bit to CW

    // set motor direction
    _outMotor.setBit(_ioPins[_motor-1].direction,dir);
    _out.write(_outMask,_outMotor);

    // now move the probe forward to the new position position
    status = StepCmd(fabs(Step));

    return status;
  }

  //--------------------------------------------------------------------------
  atdISFF::Stepper::StepperStatus
  Stepper::ResetCmd() {
    cout << "stepper reset for motor " << _motor << endl;
    return STEPPERSTATUS_IDLE;
  }

  /////////////////////////////////////////////////////////////////////
  ///
  /// The following functions provide the API for our users to
  /// register a command request.
  ///
  /// True will be returned if the command was accepted.
  /// False will be returned if the command was rejected.
  ///
  /// Notice that the command, motor number, and argument are
  /// saved here. The run() loop, in the Stepper thread, will
  /// use these values. The logic here will prevent these values
  /// from being overwritten until the status becomes idle.
  ///  
  bool
  Stepper::doCmd(int motorNumber, StepperCommand cmd, double arg) {

    // start out with a return status of true.
    bool s = true;

    // grab lock to Stepper variables.
    _lock.lock();
    
    if (motorNumber < 1 || motorNumber > NMOTORS) {
      s = false;
    }

    // If any motor is not idle, do not accept this command
    if (s) {
      for (unsigned int i = 0; i < NMOTORS; i++) {
	if (_status[i] != STEPPERSTATUS_IDLE)
	  s = false;
      }
    }

    // if the preceeding checks passed, then accept the command.
    if (s) {
      _currentCommand = cmd;
      _motor = motorNumber;
      _cmdArg = arg;
      _status[_motor-1] = STEPPERSTATUS_BUSY;
      s = true;
    }

    // release lock on Stepper variables.
    _lock.unlock();

    // return the status
    return s;
  }

  //--------------------------------------------------------------------------
  bool
  Stepper::doCmd(int motorNumber, StepperCommand cmd) {
    return doCmd(motorNumber, cmd, 0.0);
  }

  //--------------------------------------------------------------------------
  bool 
  Stepper::reset(int motorNumber) {
    _lock.lock();
    _resetFlag = true;
    _lock.unlock();
    return true;
  }
  
  //--------------------------------------------------------------------------
  bool 
  Stepper::zero(int motorNumber) {
    return doCmd(motorNumber, STEPPERCMD_ZERO);
  }
  
  //--------------------------------------------------------------------------
  bool 
  Stepper::move(int motorNumber, double steps) {
    return doCmd(motorNumber, STEPPERCMD_MOVE, steps);
  }

  //--------------------------------------------------------------------------
  double
  Stepper::position(int motorNumber) {
    double pos;
    _lock.lock();
    if (motorNumber < 1 || motorNumber > NMOTORS)
      pos = -1000000.0;
    else
      pos = _position[motorNumber-1];
    _lock.unlock();
    return pos;
  }
  
  //--------------------------------------------------------------------------
  atdISFF::Stepper::StepperStatus
  Stepper::status(int motorNumber) {
    StepperStatus s;
    _lock.lock();
    if (motorNumber < 1 || motorNumber > NMOTORS)
      s = STEPPERSTATUS_FAULT;
    else
      s = _status[motorNumber - 1];

    //    cout << "Status for motor " << motorNumber << " is " << s << std::endl;
    _lock.unlock();
    return s;
  }
}
