//
//              Copyright 2004 (C) by UCAR
//

#ifndef ATDISFF_SENSORPORTHANDLER_H
#define ATDISFF_SENSORPORTHANDLER_H

#include <atdISFF/SerialPortConfig.h>
#include <atdISFF/SensorPort.h>
#include <atdISFF/Rserial.h>
#include <atdUtil/Thread.h>
#include <atdUtil/ThreadSupport.h>

#include <sys/time.h>
#include <sys/select.h>

#include <vector>

namespace atdISFF {

class SensorPortHandler : public atdUtil::Thread {
public:

  SensorPortHandler();
  ~SensorPortHandler();

  /**
   * Creates either an AsciiSensorPort or BinarySensorPort,
   * depending on the value of cfg.getRecordLength().
   * The SensorPort then belongs to SensorPortHandler. If you
   * want to close it, call closeSensorPort(port);
   */
  SensorPort* createSensorPort(const std::string& devname,
  	const SerialPortConfig& cfg) throw(atdUtil::IOException);

  /**
   * Add an already created, but not opened SensorPort.
   * The SensorPort then belongs to SensorPortHandler. If you
   * want to close it, call closeSensorPort(port);
   */
  void addSensorPort(SensorPort* sensor,
  	const SerialPortConfig& cfg) throw(atdUtil::IOException);

  void addSensorPort(SensorPort *port);
  void closeSensorPort(SensorPort *port);

  void addRserialConnection(Rserial::Connection);
  void removeRserialConnection(int fd);

  void setTimeoutMsec(int val);
  int getTimeoutMsec() const;

  void setTimeoutWarningMsec(int val);
  int getTimeoutWarningMsec() const;

  void calcStatistics(isff_sys_time_t);

  void setStatisticsPeriodInSecs(int val) { statisticsPeriod = val * 1000; }
  int getStatisticsPeriodInSecs() const { return statisticsPeriod / 1000; }

  int getSelectErrors() const { return selectErrors; }
  int getRserialListenErrors() const { return rserialListenErrors; }

  /**
   * Thread function.
   */
  virtual int run() throw(atdUtil::Exception);

protected:

  void handleChangedPorts();

  atdUtil::Mutex portsMutex;
  std::vector<int> pendingSerialPortFds;
  std::vector<SensorPort*> pendingSerialPorts;
  std::vector<SensorPort*> pendingSerialPortClosures;

  std::vector<int> activeSerialPortFds;
  std::vector<SensorPort*> activeSerialPorts;

  atdUtil::Mutex rserialFdsMutex;
  std::vector<int> pendingRserialFds;
  std::vector<SensorPort*> pendingRserialPorts;
  std::vector<int> pendingRserialClosures;
  std::vector<int> activeRserialFds;
  std::vector<SensorPort*> activeRserialPorts;

  bool portsChanged;
  bool rserialFdsChanged;

  struct timeval tval;
  fd_set readfdset;
  int selectn;

  Rserial rserial;

  int selectErrors;
  int rserialListenErrors;

  size_t timeoutMsec;
  size_t timeoutSec;
  size_t timeoutUsec;
  size_t timeoutWarningMsec;

  isff_sys_time_t statisticsTime;

  unsigned long statisticsPeriod;

};
}
#endif
