//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/SensorPortHandler.h>
#include <atdISFF/AsciiSensorPort.h>
#include <atdISFF/BinarySensorPort.h>
#include <atdUtil/Logger.h>

#include <errno.h>
#include <signal.h>

using namespace std;
using namespace atdISFF;
using namespace atdUtil;

SensorPortHandler::SensorPortHandler() :
  Thread("SensorPortHandler"),portsChanged(false),
  rserialFdsChanged(false),statisticsPeriod(300000)
{
  /* start out with a 1/10 second select timeout.
   * While we're adding sensor ports, we want select to
   * timeout fairly quickly, so when new SensorPorts are opened
   * and added that they are read from without much time delay.
   * Otherwise if you add a sensor that isn't transmitting,
   * then one which is generating alot of output, the buffers for
   * the second sensor may fill before the first timeout.
   * Later it can be increased, but there may not be much to 
   * gain from increasing it.
   */
  setTimeoutMsec(100);
  setTimeoutWarningMsec(300000);
  FD_ZERO(&readfdset);
  statisticsTime = next_isff_sys_time(getCurrentTimeInMillis(),
  	statisticsPeriod);
  blockSignal(SIGINT);
  blockSignal(SIGHUP);
  blockSignal(SIGTERM);
  unblockSignal(SIGUSR1);
}

/**
 * Close any remaining serial ports. Before this is called
 * the run method should be finished.
 */
SensorPortHandler::~SensorPortHandler()
{
  for (unsigned int i = 0; i < activeSerialPorts.size(); i++)
    delete activeSerialPorts[i];
}

SensorPort* SensorPortHandler::createSensorPort(
	const string& devname,const SerialPortConfig& cfg)
	throw(atdUtil::IOException)
{
  SensorPort* sensor;
  if (cfg.getRecordLength() == 0) {
    sensor = new AsciiSensorPort(devname,cfg.getChannel());
    sensor->setRawLength(8);
    char eom = cfg.getRecordDelimiter().c_str()[0];
    sensor->setMessageSeparator(string(1,eom));
  }
  else {
    sensor = new BinarySensorPort(devname,cfg.getChannel());
    sensor->setMessageSeparator(cfg.getRecordDelimiter().c_str());
  }

  sensor->setBaudRate(cfg.getBaud());
  sensor->setParity(cfg.getParity());
  sensor->setDataBits(cfg.getDataBits());
  sensor->setStopBits(cfg.getStopBits());
  sensor->setFlowControl(cfg.getFlowControl());
  sensor->setLocal(true);
  sensor->setRaw(true);
  sensor->setBlocking(true);
  sensor->setMessageLength(cfg.getRecordLength());
  sensor->setSeparatorAtEOM(cfg.getDelimiterAtEOM());

  sensor->open(O_RDWR | O_NOCTTY | O_NONBLOCK);
  sensor->flushInput();
  addSensorPort(sensor);
  return sensor;
}

void SensorPortHandler::addSensorPort(
	SensorPort* sensor,const SerialPortConfig& cfg)
	throw(atdUtil::IOException)
{

  sensor->setBaudRate(cfg.getBaud());
  sensor->setParity(cfg.getParity());
  sensor->setFlowControl(cfg.getFlowControl());
  sensor->setLocal(true);
  sensor->setRaw(true);
  sensor->setBlocking(true);
  sensor->setMessageLength(cfg.getRecordLength());
  sensor->setMessageSeparator(cfg.getRecordDelimiter().c_str());
  sensor->setSeparatorAtEOM(cfg.getDelimiterAtEOM());

  sensor->open(O_RDWR | O_NOCTTY | O_NONBLOCK);
  sensor->flushInput();
  addSensorPort(sensor);
}

void SensorPortHandler::setTimeoutMsec(int val) {
  timeoutMsec = val;
  timeoutSec = val / 1000;
  timeoutUsec = (val % 1000) * 1000;
}

int SensorPortHandler::getTimeoutMsec() const {
  return timeoutSec * 1000 + timeoutUsec / 1000;
}

void SensorPortHandler::setTimeoutWarningMsec(int val) {
  timeoutWarningMsec = val;
}

int SensorPortHandler::getTimeoutWarningMsec() const {
  return timeoutWarningMsec;
}

void SensorPortHandler::calcStatistics(isff_sys_time_t tnowMsec)
{
  statisticsTime += statisticsPeriod;
  if (statisticsTime < tnowMsec)
    statisticsTime = next_isff_sys_time(tnowMsec,statisticsPeriod);

  for (unsigned int ifd = 0; ifd < activeSerialPortFds.size(); ifd++) {
    SensorPort *port = activeSerialPorts[ifd];
    port->calcStatistics(statisticsPeriod);
  }
}

/**
 * Thread function, select loop.
 */
int SensorPortHandler::run() throw(atdUtil::Exception)
{

  /* ignore SIGPIPE signals (they come from rserial) */
  struct sigaction act;
  memset(&act,0,sizeof (struct sigaction));
  act.sa_flags = 0;
  act.sa_handler = SIG_IGN;
  sigaction(SIGPIPE,&act,(struct sigaction *)0);

  isff_sys_time_t rtime = 0;
  struct timeval tout;
  unsigned long timeoutSumMsec = 0;
  for (;;) {
    if (amInterrupted()) break;

    if (portsChanged || rserialFdsChanged)
    	handleChangedPorts();

    fd_set rset = readfdset;
    fd_set eset = readfdset;
    tout.tv_sec = timeoutSec;
    tout.tv_usec = timeoutUsec;
    int nfdsel = ::select(selectn,&rset,0,&eset,&tout);
    if (amInterrupted()) break;

    if (nfdsel <= 0) {
      if (nfdsel < 0) {
	// Create and report but don't throw IOException.
	// Likely this is a EINTR (interrupted) error, in which case
	// the signal handler has set the amInterrupted() flag, which
	// will be caught at the top of the loop.
	IOException e("SensorPortHandler","select",errno);
	Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str());
	if (errno != EINTR) selectErrors++;
	timeoutSumMsec = 0;
      }
      else {
	timeoutSumMsec += timeoutMsec;
	if (timeoutSumMsec > timeoutWarningMsec) {
	  Logger::getInstance()->log(LOG_INFO,
	  	"SensorPortHandler select timeout %d msecs",timeoutSumMsec);
	  timeoutSumMsec = 0;
	}
      }
      rtime = getCurrentTimeInMillis();
      if (rtime > statisticsTime) calcStatistics(rtime);
      continue;
    }

    timeoutSumMsec = 0;

    int nfd = 0;
    int fd;

    for (unsigned int ifd = 0; ifd < activeSerialPortFds.size(); ifd++) {
      fd = activeSerialPortFds[ifd];
      if (FD_ISSET(fd,&rset)) {
	SensorPort *port = activeSerialPorts[ifd];
	try {
	  rtime = port->read();
	}
	// log the error but don't exit
	catch (IOException &ioe) {
	  Logger::getInstance()->log(LOG_ERR,"%s",ioe.toString().c_str());
	}
	if (++nfd == nfdsel) break;
      }
      // log the error but don't exit
      if (FD_ISSET(fd,&eset)) {
	SensorPort *port = activeSerialPorts[ifd];
	Logger::getInstance()->log(LOG_ERR,
	      "SensorPortHandler select reports exception for %s",
	      port->getName().c_str());
	if (++nfd == nfdsel) break;
      }
    }
    if (rtime > statisticsTime) calcStatistics(rtime);

    if (nfd == nfdsel) continue;

    for (unsigned int ifd = 0; ifd < activeRserialFds.size(); ifd++) {
      fd = activeRserialFds[ifd];
      if (FD_ISSET(fd,&rset)) {
	SensorPort *port = activeRserialPorts[ifd];
	try {
	  port->readRserial(fd);
	}
	// log the error but don't exit
	catch (IOException &ioe) {
	  Logger::getInstance()->log(LOG_ERR,"rserial: %s",ioe.toString().c_str());
	}
	if (++nfd == nfdsel) break;
      }
      if (FD_ISSET(fd,&eset)) {
	Logger::getInstance()->log(LOG_ERR,"%s",
	  "SensorPortHandler select reports exception for rserial socket ");
	if (++nfd == nfdsel) break;
      }
    }
    if (nfd == nfdsel) continue;
    fd = rserial.getListenSocketFd();
    if (FD_ISSET(fd,&rset)) {
      try {
	Rserial::Connection rsconn = rserial.acceptConnection();
	addRserialConnection(rsconn);
      }
      catch (IOException &ioe) {
	Logger::getInstance()->log(LOG_ERR,
		"SensorPortHandler rserial: %s",ioe.toString().c_str());
	rserialListenErrors++;
      }
      if (++nfd == nfdsel) continue;
    }
    if (FD_ISSET(fd,&eset)) {
      Logger::getInstance()->log(LOG_ERR,"%s",
	"SensorPortHandler select reports exception for rserial listen socket ");
	rserialListenErrors++;
      if (++nfd == nfdsel) continue;
    }
  }

  Logger::getInstance()->log(LOG_INFO,
      "SensorPortHandler finished, closing remaining %d serial ports ",activeSerialPorts.size());

  rserialFdsMutex.lock();
  std::vector<int> tfds = pendingRserialFds;
  rserialFdsMutex.unlock();
	
  for (unsigned int i = 0; i < tfds.size(); i++)
    removeRserialConnection(tfds[i]);

  portsMutex.lock();
  std::vector<SensorPort*> tports = pendingSerialPorts;
  portsMutex.unlock();

  for (unsigned int i = 0; i < tports.size(); i++)
    closeSensorPort(tports[i]);

  portsChanged = true;
  handleChangedPorts();

  return RUN_OK;
}

/**
 * Called from the main thread
 */
void SensorPortHandler::addSensorPort(SensorPort *port)
{

  int fd = port->fd();

  portsMutex.lock();

  for (unsigned int i = 0; i < pendingSerialPorts.size(); i++) {
    // new file descriptor for this port
    if (pendingSerialPorts[i] == port) {
      pendingSerialPortFds[i] = fd;
      portsChanged = true;
      portsMutex.unlock();
      return;
    }
  }
  pendingSerialPortFds.push_back(fd);
  pendingSerialPorts.push_back(port);
  portsChanged = true;
  portsMutex.unlock();

  port->setHandler(this);
}

void SensorPortHandler::closeSensorPort(SensorPort *port)
{

  port->setHandler(0);
  Synchronized autosync(portsMutex);

  for (unsigned int i = 0; i < pendingSerialPorts.size(); i++) {
    if (pendingSerialPorts[i] == port) {
      pendingSerialPortFds.erase(pendingSerialPortFds.begin()+i);
      pendingSerialPorts.erase(pendingSerialPorts.begin()+i);
      break;
    }
  }
  pendingSerialPortClosures.push_back(port);
  portsChanged = true;
}

/**
 * Protected method to add an Rserial connection
 */
void SensorPortHandler::addRserialConnection(Rserial::Connection con)
{
  Synchronized autosync(portsMutex);
  for (unsigned int i = 0; i < pendingSerialPorts.size(); i++) {
    if (pendingSerialPorts[i]->getChannel() == con.channel) {
      rserialFdsMutex.lock();
      pendingRserialFds.push_back(con.socketfd);
      pendingRserialPorts.push_back(pendingSerialPorts[i]);
      pendingSerialPorts[i]->addRserialSockFd(con.socketfd);
      rserialFdsChanged = true;
      rserialFdsMutex.unlock();
      Logger::getInstance()->log(LOG_INFO,
	"added rserial connection for channel %d",con.channel);
      break;
    }
  }
}

/**
 * this is called by a SensorPort from this (SensorPortHandler) thread.
 */
void SensorPortHandler::removeRserialConnection(int fd)
{
  Synchronized autosync(rserialFdsMutex);
  for (unsigned int i = 0; i < pendingRserialFds.size(); i++) {
    if (pendingRserialFds[i] == fd) {
      pendingRserialFds.erase(pendingRserialFds.begin()+i);
      pendingRserialPorts.erase(pendingRserialPorts.begin()+i);
    }
  }
  pendingRserialClosures.push_back(fd);
  rserialFdsChanged = true;
  Logger::getInstance()->log(LOG_INFO,"%s",
      "SensorPortHandler::removeRserialConnection");
}


void SensorPortHandler::handleChangedPorts() {
  if (portsChanged) {
    Synchronized autosync(portsMutex);
    activeSerialPortFds = pendingSerialPortFds;
    activeSerialPorts = pendingSerialPorts;
    // close any ports
    for (unsigned int i = 0; i < pendingSerialPortClosures.size(); i++)
      delete pendingSerialPortClosures[i];
    pendingSerialPortClosures.clear();
    portsChanged = false;
  }

  selectn = 0;
  FD_ZERO(&readfdset);
  for (unsigned int i = 0; i < activeSerialPortFds.size(); i++) {
    if (activeSerialPortFds[i] > selectn) selectn = activeSerialPortFds[i];
    FD_SET(activeSerialPortFds[i],&readfdset);
  }

  if (rserialFdsChanged) {
    Synchronized autosync(rserialFdsMutex);
    activeRserialFds = pendingRserialFds;
    activeRserialPorts = pendingRserialPorts;
    // close any ports
    for (unsigned int i = 0; i < pendingRserialClosures.size(); i++)
      ::close(pendingRserialClosures[i]);
    pendingRserialClosures.clear();
    rserialFdsChanged = false;
  }

  for (unsigned int i = 0; i < activeRserialFds.size(); i++) {
    if (activeRserialFds[i] > selectn) selectn = activeRserialFds[i];
    FD_SET(activeRserialFds[i],&readfdset);
  }

  int fd = rserial.getListenSocketFd();
  FD_SET(fd,&readfdset);
  if (fd > selectn) selectn = fd;

  selectn++;
}

