/*              Copyright 2004 (C) by UCAR
 *
 * File       : SensorPort.h,v
 * Revision   : 1.13
 * Directory  : /code/cvs/isa/src/lib/atdISFF/SensorPort.h,v
 * System     : PAM
 * Date       : 2005/11/16 18:17:49
 *
 * Description:
 *
 * SensorPort.h,v
 * Revision 1.13  2005/11/16 18:17:49  maclean
 * fixed a bunch of compiler warnings
 *
 * Revision 1.12  2005/01/16 01:34:24  maclean
 * added copyright
 *
 * Revision 1.11  2004/11/07 14:12:54  maclean
 * updated statistics code
 *
 * Revision 1.10  2004/10/09 04:20:13  maclean
 * fix bug in statistics calc when a sensor quits reporting
 *
 * Revision 1.9  2004/08/29 23:13:04  maclean
 * shiftBuffer now does samplePointers.clear
 *
 * Revision 1.8  2004/08/12 20:20:09  maclean
 * added support for report overflow counts
 *
 * Revision 1.7  2004/08/12 20:12:32  maclean
 * don't log so many buffer overflows
 *
 * Revision 1.6  2004/06/26 05:37:04  maclean
 * *** empty log message ***
 *
 * Revision 1.5  2004/06/23 14:25:27  maclean
 * reorg. Added RawSampleClient and RawSampleSource, and a source can have multiple clients
 *
 * Revision 1.4  2004/06/18 17:43:17  maclean
 * valgrind fixes
 *
 * Revision 1.3  2004/05/25 16:45:01  maclean
 * Initial version for Brit Stephens, OHATS test
 *
 * Revision 1.2  2004/04/27 16:43:32  maclean
 * *** empty log message ***
 *
 * Revision 1.1.1.1  2004/03/18 14:49:51  maclean
 * first import
 *
 */

#ifndef ATDISFF_SENSORPORT_H
#define ATDISFF_SENSORPORT_H

#include <atdUtil/ThreadSupport.h>
#include <atdTermio/SerialPort.h>
#include <atdISFF/RawSampleSource.h>
#include <atdISFF/Time.h>

#include <vector>
#include <list>

namespace atdISFF {

class SensorPortHandler;
class SensorPort : public atdTermio::SerialPort, public RawSampleSource {

public:
  SensorPort(const std::string& portName,int channel)
    throw(atdUtil::IOException);

  virtual ~SensorPort();

  virtual int open(int mode) throw(atdUtil::IOException);

  void setHandler(SensorPortHandler *val) { sphandler = val; }

  isff_sys_time_t read() throw(atdUtil::IOException);

  void addRserialSockFd(int fd);
  void removeRserialSockFd(int fd);
  void readRserial(int fd) throw(atdUtil::IOException,atdUtil::IOException);

  virtual void processInput(isff_sys_time_t time,const char *) throw(atdUtil::IOException) = 0;

  virtual void setBufferSize(int val);
  int getBufferSize() const { return bufsize; }

  void setChannel(int val) { chan = (unsigned char) val; }
  int getChannel() const { return chan; }

  /* sampleLength is the main distinction between BinarySensorPorts
   * and AsciiSensorPorts. For AsciiSensorPorts, sampleLength is zero,
   * and after we find an end of message character, then the next character
   * represents the start of a record, and we immediately start
   * looking for the EOR character (i.e. the EOM character cannot be
   * found in the data of the message).
   *
   * Example SensorPort messages, with a T above showing when a
   * timetag is assigned (CR=carriage return, NL=newline)
   *
   * Simple ASCII message, terminated by carriage-return, line-feed
   *      T                  T
   *  CRNL1.0 2.0 3.0 4.0CRNL  1.1 2.1 3.1 4.1CRNL
   *			setSampleLength(0);
   *			setMessageSeparator(string(1,'\n'));
   *			setSeparatorATEOM(true);	// default
   *
   * Simple ASCII message with incorrect message separator (set to CR,
   * should be NL):
   *    T                  T                    T
   *  CRNL1.0 2.0 3.0 4.0CRNL  1.1 2.1 3.1 4.1CRNL
   *			setSampleLength(0);
   *			setMessageSeparator(string(1,'\r'));
   *			setSeparatorATEOM(true);	// default
   *  Since the message separator was set to CR ('\r'), then
   *  the newline is mistaken as the first character of the
   *  next message. No data is lost, but timetags are slightly wrong.
   *
   * Binary message with hex "\x55\xaa" as end of message separator,
   * and length 10 bytes (Campbell sonic):
   *   hex, with "x" being any hex character:
   *                  T                             T
   *   xxxxxxxxxxx55aaxxxxxxxxxxxxxxxxxxxxyyyyyy55aaxxxxx...
   *           byte:  1 2 3 4 5 6 7 8 9 10
   * In the above example, "yyyyyy" are an extra 3 bytes in the
   * record.  0x55aa may appear in the data, but it will
   * be ignored in the first 10 bytes after the previous
   * 0x55aa.
   */
  virtual int getMessageLength() const { return messageLength; }
  virtual void setMessageLength(int val) {
    messageLength = val;
  }

  virtual void setMessageSeparator(const std::string& val)
	throw (atdUtil::IOException) { messageSeparator = val; }

  virtual const std::string& getMessageSeparator() const { return messageSeparator; }

  /**
   * static method to convert a string to a human readable format, with
   * non-printable characters as hex \0xhh, newlines as \n,
   * carriage returns as \r, etc.
   */
  static std::string printableMessageSeparator(const std::string&);

  /**
   * return the message separator in human-readable format. with
   */
  std::string getPrintableMessageSeparator() const { 
    return printableMessageSeparator(getMessageSeparator());
  }

  /*
   * Is the separator at the end of the message (true) or
   * at the beginning (false). This is used in order to
   * recognize the first byte of a message, so that
   * we can grab the system time when the first byte is
   * received.  If the separator is at the EOM, then
   * the first byte is the next byte after the EOM.
   * If the separator is not at the EOM (i.e. the BOM) then
   * the first byte is the message separator.
   */
  virtual void setSeparatorAtEOM(bool val) { separatorAtEOM = val; }
  bool getSeparatorAtEOM() const { return separatorAtEOM; }
  bool isSeparatorAtEOM() const { return separatorAtEOM; }

  virtual void init() throw(atdUtil::IOException);

  void initStatistics();
  void calcStatistics(unsigned long periodMsec);

  float getReadRate() const;

  void setSamplingRate(float val);
  float getSamplingRate() const;
  float getObservedSamplingRate() const;

  int getMinReadLength() const {
    return minReadLength[reportStatsIndex];
  }
  int getMaxReadLength() const {
    return maxReadLength[reportStatsIndex];
  }
  int getReadErrorCount() const { return readErrorCount[0]; };
  int getWriteErrorCount() const { return writeErrorCount[0]; };
  int getCumulativeReadErrorCount() const { return readErrorCount[1]; };
  int getCumulativeWriteErrorCount() const { return writeErrorCount[1]; };

  int getBufferOverflowCount() const { return bufferOverflowCount[0]; }
  int getCumulativeBufferOverflowCount() const { return bufferOverflowCount[1]; }

protected:

  /**
   * shift contents of buffer to beginning of buffer. 
   */
  void shiftBuffer();

  unsigned char chan;
  int bufsize;
  char *buffer;

  /**
   * one past the end of the current buffer
   */
  char *eodP;

  /**
   * bytes available for reading in the current buffer.
   */
  int bytesAvail;

  const char* scanPtr;
  
  std::string messageSeparator;

  bool separatorAtEOM;

  int messageLength;

  float msecsPerChar;

  SensorPortHandler *sphandler;

  /**
   * Start time. Used in computing initial statistics.
   */
  time_t initialTimeSecs;
  int minReadLength[2];
  int maxReadLength[2];
  int readErrorCount[2];	// [0] is recent, [1] is cumulative
  int writeErrorCount[2];	// [0] is recent, [1] is cumulative
  int currStatsIndex;
  int reportStatsIndex;
  int nreads;
  int nsamples;

  int bufferOverflowCount[2];	// [0] is recent, [1] is cumulative

  /**
   * Number of reads from the serial port per second.
   */
  float readRate;	

  /**
   * Observed number of samples per second.
   */
  float sampleRateObs;

  std::vector<int> activeRserialSockFds;
  std::vector<int> pendingRserialSockFds;
  bool rserialSockFdsChanged;
  std::vector<int>::iterator rserialFdIterator;
  char rserialBuf[256];
  atdUtil::Mutex rserialSockFdsMutex;

  /**
   * When more than one sample are read at a time, we scan
   * through the buffer, and determine the pointers to
   * the beginning of each sample.
   */
  std::list <const char*> samplePointers;

  /**
   * observed deltaT between samples.
   */
  float deltatms;
  
  /**
   * Time of last sample in previous buffer. Used by derived
   * classes to compute deltatms
   */
  isff_sys_time_t lastSampleTime;

  /**
   * Whether last sample in previous buffer was a partial sample.
   */
  bool partialLastSample;

  /**
   * How many zero length reads in a row have we done.
   * Paranoid check. After several (currenty 5) zero length
   * reads in a row, we close the port.
   */
  int read0Len;

};
}

#endif
