//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/SensorPort.h>
#include <atdISFF/SensorPortHandler.h>
#include <atdUtil/Logger.h>
#include <sys/time.h>
#include <errno.h>
#include <iomanip>

using namespace atdUtil;
using namespace atdISFF;
using namespace atdTermio;
using namespace std;

SensorPort::SensorPort(const string& name,
    int channel)
  	throw (IOException) :
	SerialPort(name),chan(channel),bufsize(0),buffer(0),
	scanPtr(0),
	separatorAtEOM(true),
	messageLength(0),
	sphandler(0),
	rserialSockFdsChanged(false),
	deltatms(0.0f),lastSampleTime(0),partialLastSample(false),
	read0Len(0)
{
  setBufferSize(1024);
  initStatistics();
  reportStatsIndex = currStatsIndex;
}
SensorPort::~SensorPort() {
  delete [] buffer;
}

int SensorPort::open(int mode) throw(IOException) {
  int res = SerialPort::open(mode);
  init();
  msecsPerChar = (getDataBits() + getStopBits() + 1) /
    	(double) getBaudRate() * 1000.0;
  return res;
}

void SensorPort::init() throw(IOException) {}

void SensorPort::setBufferSize(int val) {
  delete [] buffer;
  bufsize = val;
  buffer = new char[bufsize];
  eodP = buffer;
  bytesAvail = bufsize;
}

void SensorPort::addRserialSockFd(int fd) {
  rserialSockFdsMutex.lock();
  pendingRserialSockFds.push_back(fd);
  rserialSockFdsChanged = true;
  rserialSockFdsMutex.unlock();
  // cerr << "SensorPort::addRserialSockFd" << endl;
}

void SensorPort::removeRserialSockFd(int fd) {
  // cerr << "start of SensorPort::removeRserialSockFd" << endl;
  rserialSockFdsMutex.lock();
  for (vector<int>::iterator i = pendingRserialSockFds.begin(); 
  	i < pendingRserialSockFds.end(); ) {
    if (*i == fd) i = pendingRserialSockFds.erase(i);
    else ++i;
  }
  rserialSockFdsChanged = true;
  rserialSockFdsMutex.unlock();
  if (sphandler) sphandler->removeRserialConnection(fd);
  // cerr << "SensorPort::removeRserialSockFd" << endl;
}

/**
 * This method is called on this SensorPort when bytes are available on
 * the serial port.  It is typically called after a select() indicates
 * a read would succeed on this file descriptor.  This method reads
 * the bytes into a buffer and then calls processInput() which a sub-class
 * of SensorPort must implement to handle data.
 */

isff_sys_time_t SensorPort::read() throw(IOException)
{
  
  int rlen;
  // bytesAvail == 0 is a symptom of a mis-configured or mis-operating
  // sensor. It means that we aren't finding the end-of-message
  // character anywhere in the output.
  // The reason could be wrong baud rate, wrong parity,
  // wrong number of data bits, binary sensor configured as ASCII or v.v.,
  // or wrong end-of-message string.
  if (bytesAvail == 0) {
    bufferOverflowCount[1]++;		// cumulative counter
    if (!(bufferOverflowCount[0]++ % 1000))	// count in last stats period
      Logger::getInstance()->log(LOG_WARNING,"%s: %d buffer overflows",
      	getName().c_str(),bufferOverflowCount[1]);
    flushInput();
    eodP = buffer;
    bytesAvail = bufsize;
  }
  try {
    if ((rlen = atdTermio::SerialPort::read(eodP,bytesAvail)) == 0)  {
      minReadLength[currStatsIndex] = 0;
      if (read0Len++ > 3) {
	Logger::getInstance()->log(LOG_WARNING,
		"%s: %d zero length reads in a row. Closing port",
		getName().c_str(),read0Len);
	sphandler->closeSensorPort(this);
      }
      return 0;
    }
  }
  catch (IOException &ioe) {
    readErrorCount[0]++;
    readErrorCount[1]++;
    throw ioe;
  }

  /* get time of read completion */
  isff_sys_time_t rtime = getCurrentTimeInMillis();

  read0Len = 0;


  /*
  cerr << "readlen=" << rlen <<
    " avail=" << bytesAvail << 
    " notavail=" << (int)(eodP - buffer) <<
    " total=" << bytesAvail + (int)(eodP - buffer) <<
    endl;
  */

  if (rlen < minReadLength[currStatsIndex])
      minReadLength[currStatsIndex] = rlen;
  if (rlen > maxReadLength[currStatsIndex])
      maxReadLength[currStatsIndex] = rlen;

  nreads++;

  if (rserialSockFdsChanged) {
    rserialSockFdsMutex.lock();
    activeRserialSockFds = pendingRserialSockFds;
    rserialSockFdsChanged = false;
    rserialSockFdsMutex.unlock();
    // cerr << "rserialSockFdsChanged in " << getName() << " size=" <<
	// activeRserialSockFds.size() << endl;
  }

  for (rserialFdIterator = activeRserialSockFds.begin();
  	rserialFdIterator < activeRserialSockFds.end(); ) {
    int rfd = *rserialFdIterator;
    int rstat;
    // note that we don't handle situation of writing less
    // than rlen bytes to rserial socket
    if ((rstat = ::write(rfd,eodP,rlen)) < 0)  {
      IOException e("rserial socket","write",errno);
      cerr << e.toString() << endl;
      removeRserialSockFd(rfd);
    }
    ++rserialFdIterator;
  }

  char *prevP = eodP;
  eodP += rlen;
  bytesAvail -= rlen;

  processInput(rtime,prevP);
  return rtime;
}

void SensorPort::shiftBuffer() {
  // shift data back to beginning of buffer
  if (scanPtr > buffer + 20) {
    bytesAvail += scanPtr - buffer;
    register char* cp2;
    for (cp2 = buffer; scanPtr < eodP; ) *cp2++ = *scanPtr++;
    eodP = cp2;
    scanPtr = buffer;
  }
  samplePointers.clear();
  // sanity check, could be assert
  if (bytesAvail != buffer + bufsize - eodP)
        std::cerr << "bad buffer arith:" <<
        " bytesAvail=" << bytesAvail <<
        " buffer + bufsize - eodP" << buffer + bufsize - eodP << std::endl;
}


void SensorPort::readRserial(int rfd) throw(IOException) {

  int i = ::read(rfd,rserialBuf,sizeof(rserialBuf));
  if (i <= 0) {
    removeRserialSockFd(rfd);
    if (i < 0) throw IOException("rserial socket","read",errno);
  }
  else if (i > 0) {
    try {
      // don't handle situation of writing less than i bytes
      // to serial port
      write(rserialBuf,i);
    }
    catch (IOException& ioe) {
      writeErrorCount[0]++;
      writeErrorCount[1]++;
      throw ioe;
    }
  }
}

void SensorPort::initStatistics()
{
  currStatsIndex = reportStatsIndex = 0;
  maxReadLength[0] = maxReadLength[1] = 0;
  minReadLength[1] = minReadLength[currStatsIndex] = 999999999;

  readRate = 0.0;
  sampleRateObs = 0.0;

  nreads = nsamples = 0;
  readErrorCount[0] = writeErrorCount[0] = bufferOverflowCount[0] = 0;
  readErrorCount[1] = writeErrorCount[1] = bufferOverflowCount[1] = 0;
  initialTimeSecs = time(0);
}

void SensorPort::calcStatistics(unsigned long periodMsec)
{
  reportStatsIndex = currStatsIndex;
  currStatsIndex = (currStatsIndex + 1) % 2;
  maxReadLength[currStatsIndex] = 0;
  minReadLength[currStatsIndex] = 999999999;

  // periodMsec is in milliseconds, hence the factor of 1000.
  readRate = (float)nreads / periodMsec * 1000.;
  sampleRateObs = (float)nsamples / periodMsec * 1000.;

  nreads = nsamples = 0;
  readErrorCount[0] = writeErrorCount[0] = bufferOverflowCount[0] = 0;
}

float SensorPort::getObservedSamplingRate() const {
  if (reportStatsIndex == currStatsIndex)
      return (float)nsamples/(time(0) - initialTimeSecs);
  else return sampleRateObs;
}

float SensorPort::getReadRate() const {
  if (reportStatsIndex == currStatsIndex)
      return (float)nreads/(time(0) - initialTimeSecs);
  else return readRate;
}

/* static */
string SensorPort::printableMessageSeparator(const string& inputstr) {

  string result;
  bool printable = true;
  // check if all are printable
  for (unsigned int i = 0; i < inputstr.size(); i++)
    if (!isprint(inputstr[i])) { printable = false; break; }

  if (!printable) result.append("\\0x");

  for (unsigned int i = 0; i < inputstr.size(); i++) {
    char c = inputstr[i];
    if (printable) {
      if (isspace(c)) {
	switch (c) {
	case '\f':
	  result.append("\\f");
	  break;
	case '\n':
	  result.append("\\n");
	  break;
	case '\r':
	  result.append("\\r");
	  break;
	case '\t':
	  result.append("\\t");
	  break;
	case '\v':
	  result.append("\\v");
	  break;
	case ' ':
	default:
	  result += c;
	  break;
	}
      }
      else result += c;
    }
    else {
      char hexstring[4];
      sprintf(hexstring,"%02x",(unsigned char)c);
      result.append(hexstring);
    }
  }
  return result;
}

