//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/SampleFormatter.h>
#include <atdISFF/Time.h>

#include <iostream>
#include <iomanip>

#ifdef GPP_2_95_2
#include <strstream>
#else
#include <sstream>
#endif

using namespace atdISFF;
using namespace std;

SampleFormatter::SampleFormatter():
	data(0),
	relayBits(0),relayBitsLen(0),
	deltatMsec(1000),deltatMsec2(deltatMsec/2),
	nextOutputTime(-1000)
{
}
SampleFormatter::~SampleFormatter() {
  delete [] data;
}

void SampleFormatter::addVariable(varType type,const string &name, int channel) {
  addVariable(type,name,channel,0.0,0.0);
}

void SampleFormatter::addVariable(varType type,const string &name, int channel,float slope,float intercept) {
  types.push_back(type);
  varnames.push_back(name);
  channelNumbers.push_back(channel);
  if (data) delete [] data; data = 0;
  a2dslope.push_back(slope);
  a2dintercept.push_back(intercept);
}

/**
 * Method called when an upstream class has a sample available for me.
 */
bool SampleFormatter::receive(const RawSample *sample) throw(atdUtil::IOException) {
  
  if (!data) init();

  unsigned long time = sample->tt;

  if (nextOutputTime < 0)
      nextOutputTime = (time / deltatMsec) * deltatMsec + deltatMsec2;

  // If this is new data, format out previous data.
  // cerr << "time=" << time << " nextOutputTime=" << nextOutputTime <<
  // 	" diff=" << sample_time_diff(time,nextOutputTime) << endl;

  if (sample_time_diff(time,nextOutputTime) >= 0) {
    format();
    nextOutputTime = sample_time(nextOutputTime,deltatMsec);
  }

  int chan = sample->chan;
  varType type = LICOR820;
  int ivar;
  int nvars = channelNumbers.size();
  for (ivar = 0; ivar < nvars; ivar++) {
    if (chan == channelNumbers[ivar]) {
      type = types[ivar];
      break;
    }
  }

  if (ivar == nvars) return 0;	// un-recognized channel number

  // cerr << ((type == LICOR820) ? "licor" : "other") << endl;

  switch (type) {
  case LICOR820:
    {
      int nfloat = sample->len / sizeof(float);
      for (int i = 0; i < nfloat; ) {
	// cerr << "ivar=" << ivar << " i=" << i << endl;
        data[ivar] = ((float *)(sample->data))[i];
	if (++i == nfloat) break;			// scan for next
	for (ivar++ ; ivar < nvars; ivar++)
	  if (chan == channelNumbers[ivar]) break;	// look for next
	if (ivar == nvars) break;	// un-recognized channel number
      }
    }
    break;
  case A2D:
    {
      int nshort = sample->len / sizeof(short);
      for (int i = 0; i < nshort; ) {

#if __BYTE_ORDER == __LITTLE_ENDIAN
	union {
	  short s;
	  char c[2];
	} val;
	val.c[0] = sample->data[i*2+1];
	val.c[1] = sample->data[i*2];
        data[ivar] = val.s;
#else
        data[ivar] = ((short *)(sample->data))[i];
#endif
#ifdef DEBUG
	cerr << "chan=" << chan << " ivar=" << ivar << " i=" << i <<
		" val=" << data[ivar] <<
		" slope=" << a2dslope[ivar] << 
		" intercept=" << a2dintercept[ivar] << endl;
#endif
        data[ivar] *= a2dslope[ivar];
	data[ivar] += a2dintercept[ivar];

	if (++i == nshort) break;

	// scan for next variable which may also have this channel number
	for (ivar++ ; ivar < nvars; ivar++)
	  if (chan == channelNumbers[ivar]) break;
	if (ivar == nvars) break;	// un-recognized channel
      }
    }
    break;
  case RELAY:
    {
      int len = sample->len;
      if (len != relayBitsLen) {
        delete [] relayBits;
	relayBits = new unsigned char[len];
	relayBitsLen = len;
      }
      ::memcpy(relayBits,sample->data,len);
    }
    break;
  }
  return true;
}

void SampleFormatter::format() throw(atdUtil::IOException) {

#ifdef GPP_2_95_2
  ostrstream ost;
#else
  ostringstream ost;
#endif

  // setfill persists
  // setw doesn't
  // if fixed is in effect, then setprecision sets num of digits after dp

  ost << fixed;

  for (unsigned int i = 0; i < channelNumbers.size(); i++) {
    switch (types[i]) {
    case RELAY:
      ost << setfill('0') << hex;
      // print them out high order bits first
      for (int ib = relayBitsLen - 1; ib >= 0; ib--) 
	ost << setw(2) << (unsigned int)relayBits[ib];
      ost << dec << ' ';
      break;
    default:
      if (isnan(data[i])) ost << setw(8) << setfill(' ') << "NA " << ' ';
      else ost << setprecision(3) << setw(8) << setfill(' ') << data[i] << ' ';
      break;
    }
  }

  string ostring = ost.str();
  int len = ostring.size() + 1;
  if (len > 255) len = 255;	// sample length limit

  RawSample *sample = 
    RawSamplePool::getInstance()->getRawSample(len);
  sample->tt = sample_time(nextOutputTime,-deltatMsec2);
  sample->chan = 201;
  sample->len = (unsigned char) len;
  ::memcpy(sample->data,ostring.c_str(),len-1);
  sample->data[len-1] = '\0';

  distributeDiscOnExc(sample);
  initData();
}

void SampleFormatter::init() {
  data = new float[varnames.size()];

  // initial value of relay - we don't have
  // a special missing data value for the relay bits,
  // so we'll just set them to 0.  Also, this
  // assumes a length of 3 bytes.
  if (relayBitsLen == 0) {
    relayBitsLen = 3;
    relayBits = new unsigned char[relayBitsLen];
    ::memset(relayBits,0,relayBitsLen);
  }
  initData();
}

void SampleFormatter::initData() {
  // we won't set the relay values to anything - assume
  // they remain at their previous value.
  for (unsigned int i = 0; i < varnames.size(); i++) data[i] = nanf("");
}

