//
//              Copyright 2004 (C) by UCAR
//

#include <vector>

#include <atdISFF/RawSampleSocket.h>
#include <atdUtil/Logger.h>

#include <iostream>

using namespace atdISFF;
using namespace atdUtil;

RawSampleSocket::RawSampleSocket(const std::string& serverName):
    FieldDataSocket(serverName,8000),_lostSamples(0),_nwriteErr(0),
    _opener(0)
{
  setMaxMilliSecondsBetweenWrites(5000);
  // setMaxMilliSecondsBetweenWrites(100);
  setMinMilliSecondsBetweenWrites(1);
  // Logger::getInstance()->log(LOG_INFO,"getSendBufferSize()=%d",
  // 	getSendBufferSize());
}
void RawSampleSocket::open() throw(UnknownHostException,IOException) {
  FieldDataSocket::open();
  setNonBlocking();
  // setSendBufferSize(32768);
}

RawSampleSocket::~RawSampleSocket()
{
    if (_opener && _opener->isRunning()) _opener->kill(SIGUSR1);
}

/**
 * A RawSampleSource is passing a sample to me. Write it
 * to the socket.
 *
 * return true: OK
 * 	false: socket blocked, sample not written
 */
bool RawSampleSocket::receive(const RawSample *sample) throw(IOException) {

  if (!isConnected()) {
    _lostSamples++;
    return false;
  }
  else _opener = 0;

  // since we do not keep the sample, no need to sample->holdReference()
  const char *wptrs[3];
  int lens[3];

  // don't alter sample, other clients may want it
  unsigned long tt = htonl(sample->tt);	// convert to big-endian
  wptrs[0] = (const char *) &tt;
  lens[0] = sizeof(tt);

  wptrs[1] = sample->getHeaderPtr() + sizeof(tt);
  lens[1] = RawSample::headLen - sizeof(tt);

  wptrs[2] = (const char *)sample->data;
  lens[2] = sample->len;

  bool result = true;
  try {
    if (write(wptrs,lens,3) == 0 && !(_lostSamples++ % 1000)) {
	Logger::getInstance()->log(LOG_WARNING,
		"%s: lost %d samples, temp unavail=%d, partial=%d %d",
		getName().c_str(),_lostSamples,
	      getNumWriteTempUnavailable(),
	      _partialWrites[0],_partialWrites[1]);
	result = false;
    }
  }
  catch (IOException &ioe) {
    result = false;
    _lostSamples++;
    if ((_nwriteErr++ % 100) == 0)
      Logger::getInstance()->log(LOG_ERR,"%s",ioe.toString().c_str());
    // schedule a thread to re-open this socket
    try {
      close();
      reopen();
    }
    catch(Exception& e) {
      throw IOException(getName(),"reopen",e.what());
    }
  }
  return result;
}
void RawSampleSocket::reopen() throw(Exception)
{
  /* The ReOpener is a detached thread, which will be deleted
   * at the end of the run method, so this isn't a memory leak.
   */
  _opener = new RawSampleSocketReOpener(this);
  _opener->blockSignal(SIGINT);
  _opener->blockSignal(SIGHUP);
  _opener->blockSignal(SIGTERM);
  _opener->start();
}

