//
//              Copyright 2004 (C) by UCAR
//

#include <vector>

#include <atdISFF/RawSampleBuffer.h>
#include <atdISFF/Time.h>
#include <atdUtil/Logger.h>

#include <unistd.h>	// for sleep
#include <iostream>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

RawSampleBuffer::RawSampleBuffer(int buflenInMilliSec_a) :
    Thread("RawSampleBuffer"),buflenInMillisec(buflenInMilliSec_a),
    samplesAvail("samplesAvail"),threadSignalFactor(1000)
{
  blockSignal(SIGINT);
  blockSignal(SIGHUP);
  blockSignal(SIGTERM);
}

RawSampleBuffer::~RawSampleBuffer() {
  RawSampleSet::iterator iv;
  for (iv = samples.begin(); iv != samples.end(); iv++) {
    const RawSample *s = *iv;
    s->freeReference();
  }
  samples.clear();
}

/**
 * Thread function.
 */
int RawSampleBuffer::run() throw(Exception) {

  for (;;) {

    // If another thread is interrupting us, we don't want the 
    // interrupt() method to finish between the time of the
    // check for amInterrupted and the wait(), otherwise we will
    // wait without seeing the interrupt. Therefore we check
    // for the interrupt after the lock.
    samplesAvail.lock();
    if (amInterrupted()) {
      samplesAvail.unlock();
      break;
    }
    samplesAvail.wait();

    if (amInterrupted()) {
      samplesAvail.unlock();
      break;
    }

    RawSampleSet::const_reverse_iterator latest = samples.rbegin();

    if (latest == samples.rend()) { // empty set
	samplesAvail.unlock();
	continue;
    }
    dummy.tt = (*latest)->tt;
    if (dummy.tt < buflenInMillisec) dummy.tt += 86400000;
    dummy.tt -= buflenInMillisec;
    // std::cout << "awoke in thread func, samples.size()=" << samples.size() <<
    //	" tt=" << dummy.tt << std::endl;

    RawSampleSet::const_iterator rsb = samples.begin();

    // get iterator pointing at first sample not less than dummy
    RawSampleSet::const_iterator rsi = samples.lower_bound(&dummy);

    // grab the samples before the iterator
    std::vector<const RawSample*> agedsamples(rsb,rsi);

    // remove samples from sorted multiset
    samples.erase(rsb,rsi);

    // free the lock
    samplesAvail.unlock();

    // loop over the aged samples
    std::vector<const RawSample *>::iterator iv;
    for (iv = agedsamples.begin(); iv < agedsamples.end(); iv++) {
      const RawSample *s = *iv;
      try {
	distributeDiscOnExc(s);
      }
      catch(IOException& ioe) {
	Logger::getInstance()->log(LOG_INFO,"RawSampleBuffer::distribute: %s",
		ioe.what());
      }
    }
  }
  Logger::getInstance()->log(LOG_INFO,"%s",
    "RawSampleBuffer::run interrupted");
  return RUN_OK;
}

void RawSampleBuffer::interrupt() {
  // cerr << "RawSampleBuffer::interrupt" << endl;
  Synchronized autosync(samplesAvail);
  Thread::interrupt();
  samplesAvail.signal();
  // cerr << "RawSampleBuffer::interrupt done" << endl;
}

bool RawSampleBuffer::receive(const RawSample *s) throw () {
  static int nsamp = 0;
  Synchronized autosync(samplesAvail);
  samples.insert(samples.end(),s);
  s->holdReference();

  // signal the handler thread every threadSignalFactor
  // number of samples
  nsamp = (++nsamp % threadSignalFactor);
  if (!nsamp) samplesAvail.signal();
  return true;
}

