//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/RawSampleArchiver.h>
#include <atdUtil/Logger.h>

#include <netinet/in.h>		// htonl()

#include <stdio.h>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

const int RawSampleArchiver::MSECS_IN_DAY = 86400000;
const int RawSampleArchiver::MSECS_IN_5MIN = 300000;

RawSampleArchiver::RawSampleArchiver() :
  fileLengthMsecs(8*3600*1000),nextFileTime(0),cfp(0),badTimeTagCount(0)
{
  initTime();
}

RawSampleArchiver::~RawSampleArchiver()
{
  if (cfp) ::fclose(cfp);
}

bool RawSampleArchiver::receive(const RawSample* samp)
	throw(IOException)
{
  isff_sys_time_t fullTT = expandTimeTag(samp->tt);
  if (fullTT > 0) {
    if (fullTT > nextFileTime) openFile(fullTT);
    write(fullTT,samp);
    return true;
  }
  return false;
}

void RawSampleArchiver::write(isff_sys_time_t fullTT,const RawSample* samp)
	throw(IOException)
{
  // don't write 0 length samples
  if (samp->len == 0) return;

  // since we do not keep the sample, no need to samp->holdReference()
  unsigned long tt = htonl(samp->tt);	// convert to big-endian

  int nintr = 0;

  // On Aug 25 04:20 OHATS, saw the following. Don't know
  // what the source of the interruption was - assume some signal:
  // Aug 24 22:20:11 ragwort Sampler[97]: RawSampleBuffer::distribute:
  //   IOException: /data/lrag_040825_000000.dat: write: Interrupted system call
  // However this could have been due trying to write a len==0 sample
  // in which case it wasn't really an error.

  while (::fwrite(&tt,sizeof(tt),1,cfp) < 1) {
    if (ferror(cfp)) {
      if (errno != EINTR || nintr++ > 5)
      	throw IOException(fileName,"write",errno);
      else Logger::getInstance()->log(LOG_WARNING,
	      "Interrupted write to %s",fileName.c_str());
    }
    else
      throw IOException(fileName,"write","unknown error");
  }

  size_t l = RawSample::headLen - sizeof(samp->tt);
  while (::fwrite(samp->getHeaderPtr() + sizeof(samp->tt),l,1,cfp) < 1) {
    if (ferror(cfp)) {
      if (errno != EINTR || nintr++ > 5)
      	throw IOException(fileName,"write",errno);
      else Logger::getInstance()->log(LOG_WARNING,
	      "Interrupted write to %s",fileName.c_str());
    }
    else
      throw IOException(fileName,"write","unknown error");
  }

  while (::fwrite(samp->data,samp->len,1,cfp) < 1) {
    if (ferror(cfp)) {
      if (errno != EINTR || nintr++ > 5)
      	throw IOException(fileName,"write",errno);
      else Logger::getInstance()->log(LOG_WARNING,
	      "Interrupted write to %s",fileName.c_str());
    }
    else
      throw IOException(fileName,"write","unknown error");
  }
}

void RawSampleArchiver::openFile(isff_sys_time_t fullTT)
	throw(IOException)
{

  if (cfp) {
    Logger::getInstance()->log(LOG_INFO,"Closing %s",fileName.c_str());
    ::fclose(cfp);
    cfp = 0;
  }

  time_t timet = (time_t) (fullTT / 1000);	// convert to secs
  struct tm tm;
  gmtime_r(&timet,&tm);

  char tmpstr[1024];
  strftime(tmpstr,sizeof(tmpstr),getFileNameFormat().c_str(),&tm);

  shortName = string(tmpstr);

  fileName = getDirectoryName() + '/' + shortName;

  Logger::getInstance()->log(LOG_INFO,"Opening %s",fileName.c_str());
  if ((cfp = ::fopen(fileName.c_str(),"w")) == NULL) {
    cerr << "open failed" << endl;
    throw IOException(fileName,"open",errno);
  }

#ifdef TIME_IS_LONG_LONG
  nextFileTime = fullTT - (fullTT % fileLengthMsecs) + fileLengthMsecs;
#else
  nextFileTime = fullTT - fmod(fullTT,fileLengthMsecs) + fileLengthMsecs;
#endif
}


void RawSampleArchiver::initTime()
{
  t0day = getCurrentTimeInMillis();

#ifdef TIME_IS_LONG_LONG
  t0day -= t0day % MSECS_IN_DAY;
#else
  t0day -= fmod(t0day,MSECS_IN_DAY);
#endif
  t0day -= MSECS_IN_DAY;
  lastTT = MSECS_IN_DAY + 1;
}

isff_sys_time_t RawSampleArchiver::expandTimeTag(unsigned long tt) {

  isff_sys_time_t fullTT = t0day + tt;

  /*
   * tt is the number of seconds since midnight GMT.
   * Since the input data is supposed to be sorted,
   * tt can be less than lastTT in the following situations:
   * 1. Normal midnight rollover.  lastTT is the time of the last
   *    sample just before midnight say 86399000, and tt
   *    is a few milliseconds after midnight, say 200.
   * 2. Wacko sample time from a serial sensor. If a 
   *    serial sensor is unplugged when the data system
   *    has read a partial sample. When the serial sensor
   *    is plugged in again, you might get a backwards
   *    time tag on the screwed up sample.  The trick
   *    is to not confuse it with a midnight rollover.
   */

  if (tt < lastTT) {
    // We'll assume we're running in real-time and that
    // the system clock is well behaved.
    isff_sys_time_t tnow = getCurrentTimeInMillis();
    while (fullTT < tnow - MSECS_IN_DAY + MSECS_IN_5MIN) {
      // looks like a midnight rollover
      t0day += MSECS_IN_DAY;
      fullTT += MSECS_IN_DAY;
    }
#ifdef TIME_IS_LONG_LONG
    if (labs(fullTT-tnow) > MSECS_IN_5MIN)
#else
    if (fabs(fullTT-tnow) > MSECS_IN_5MIN)
#endif
    {
      badTimeTagCount++;
      return 0;
    }
  }
  else if (tt > lastTT + MSECS_IN_5MIN) {
    isff_sys_time_t tnow = getCurrentTimeInMillis();
#ifdef TIME_IS_LONG_LONG
    if (labs(fullTT-tnow) > MSECS_IN_5MIN)
#else
    if (fabs(fullTT-tnow) > MSECS_IN_5MIN)
#endif
    {
      badTimeTagCount++;
      return 0;
    }
  }
  lastTT = tt;
  return fullTT;
}

