//
//              Copyright 2004 (C) by UCAR
//

#ifndef ATDISFF_RAWSAMPLE_H
#define ATDISFF_RAWSAMPLE_H

#include <atdISFF/RawSamplePool.h>

#include <string>

#ifdef TEST_INTERFACE

class RawSampleInterface {
  
  typedef xxx tt_type;

  virtual void setTimeTag(isff_sys_time_t t) = 0;
  tt_type getTimeTag() const;	// problem, data loss on set

  virtual void setChannelId(int val) = 0;
  int getChannelId() const;

  virtual const char* getHeaderPtr() const = 0;

  virtual const char* getDataPtr() const = 0;

  // virtual void allocateData(unsigned int length) = 0;

  unsigned int getDataLength() const;

};

/* If we want to use templates AND pooled reference counting
 * then each type will have to have its own RawSamplePool. Hmmm.
 */
template<class Ttype,Dtype>
class BasicRawSample<Ttype,Dtype> : public RawSampleInterface {

#endif


namespace atdISFF {
class RawSample {
public:
  static int headLen;
  /**
   * Global count of the number of samples in use by a process.
   * Incremented in the constructor, decremented in the destructor.
   * Useful for development debugging to track leaks.
   */
  static int nsamps;

public:
  unsigned long tt;
  unsigned char chan;
  unsigned char len;
  char *data;

  const char* getHeaderPtr() const { return (const char *) &tt; }

  /**
   * Increment the reference count for this sample.
   * RawSample supports a form of reference counting.
   * See freeReference.  We have to use a hack-like
   * cast of the this pointer to a const so that holdReference
   * can be used on a const RawSample.
   */
  void holdReference() const { ((RawSample*)this)->refCount++; }

  /**
   * Decrement the reference count for this sample.
   * If the reference count is zero, then put the RawSample
   * into the RawSamplePool.
   * freeReference() can be performed on a const RawSample.
   * However if at any moment a user of a const RawSample has called
   * freeReference more times than they have called holdReference,
   * then they have violated the contract of a const RawSample.
   * Once the sample goes back into the sample pool it may be altered.
   *
   * RawSamples can be used like normal variables without reference
   * counting:
   *  {
   *	RawSample samp;		// automatic variable
   *	samp.tt = 99;
   *    ...
      }				// automatic variable destroyed
   *    
   * Typical reference count usage:
   *   void pushSample(const RawSample* samp) 
   *   {
   *     samp->holdReference();
   *     buffer.push_back(samp);
   *   }
   *   const RawSample* popSample()
   *   {
   *     const RawSample* samp = buffer.back();
   *     buffer.pop_back();
   *     samp->freeReference();
   *     return samp;
   *   }
   *
   *   // Get sample from RawSamplePool.
   *   // The reference count will be one after the sample
   *   // is fetched from the pool - holdReference is
   *   // called for you by RawSamplePool.
   *   RawSample* samp =
   *	  RawSamplePool::getReference()->getRawSample(100);
   *
   *   pushSample(samp);
   *   ...
   *   samp = popSample();
   *   // When you're done with it, call freeReference. This 
   *   // should decrement the reference count to zero, and
   *   // cause the sample to be returned to the RawSamplePool.
   *   samp->freeReference();
   *
   * TODO: write a test program and see if the RawSamplePool
   * actually speeds things up - is it faster than new()?
   */
  void freeReference() const {
    // if refCount is 0, put it back in the Pool.
    // Would be cool to have some sort of introspection to
    // detect the situation of returning a const RawSample back
    // to the RawSamplePool.
    if (! --(((RawSample*)this)->refCount))
    	RawSamplePool::getInstance()->putRawSample((RawSample*)this);
  }

  RawSample() : data(0),refCount(1),allocLen(0) { nsamps++; }
  virtual ~RawSample() { delete [] data; nsamps--; }

  short getAllocLen() const { return allocLen; }
  void setAllocLen(short val) {
    if (allocLen < val) {
      delete [] data;
      data = new char[val];
      allocLen = val;
    }
  }

  std::string toString() const;

protected:
  /**
   * The reference count.  Shouldn't be more than 32767 users.
   */
  short refCount;
  short allocLen;
};
}

#endif
