//
//              Copyright 2004 (C) by UCAR
//

#include <unistd.h>	// getopt
#include <config.h>
#include <atdISFF/NDaqSampler.h>

#include <atdISFF/AsciiSensorPort.h>
#include <atdISFF/BinarySensorPort.h>

#ifdef ENABLE_DSCUD
#include <dsc/ExtClockedPrometheusA2DSampler.h>
#include <dsc/ExtClockedMM16AT_A2DSampler.h>
#include <dsc/ExtClockedMM32XAT_A2DSampler.h>
#include <dsc/DiamondUD.h>
#endif

#include <atdISFF/RPCSerialPortConfiguration.h>
#include <atdISFF/RPCA2DChannelConfiguration.h>
#include <atdTermio/SerialPort.h>
#include <atdISFF/DaqMonitorRPC.h>
#include <atdISFF/RawSampleSocket.h>
#include <atdISFF/RawSampleOutputStream.h>
#include <atdISFF/ChannelDevNameMap.h>

#include <atdISFF/relays/Opto22Relay.h>

#if defined(ENABLE_DSCUD)
#include <dsc/IR104.h>
#endif

#include <atdUtil/Logger.h>

#include <errno.h>

#ifdef GPP_2_95_2
#include <strstream>
#else
#include <sstream>
#endif

// for mlockall
#include <sys/mman.h>
#include <sys/resource.h>


using namespace std;
using namespace atdISFF;
using namespace atdUtil;
using namespace atdTermio;

/* static */
const string NDaqSampler::_version = "v1";

/* static */
const string& NDaqSampler::getVersion() { return _version; }

/* static */
time_t NDaqSampler::_startTime = 0;

/* static */
NDaqSampler* NDaqSampler::getNDaqInstance()
{
  return (NDaqSampler*) Sampler::_instance;
}

NDaqSampler::NDaqSampler() : 
  Sampler(),
  _name("ndaq"),
  _rsoutput(0),
  _rssocket(0),
  _rsbuffer(0),
  _archiver(0),
  _sphandler(0),
  _a2dsampler(0),
  _monitor(0),
  _rrelayListener(0),
  _relayFormatter(0),
  _timer(0),
  _quit(false)
#if defined(ENABLE_DSCUD)
  ,_dostepper(false)
  ,_stepper(0)
#endif
{
  _privileged = (geteuid() == 0);	// if effective userid is root
  time(&_restartTime);
}

void NDaqSampler::init()
  throw(UnknownHostException,IOException,InvalidParameterException)
{
  _sensorClients.clear();

  if (getDataServerName().size() > 0) {
    _rsoutput = _rssocket = new RawSampleSocket(getDataServerName());
    _rssocket->open();
  }
  else
    _rsoutput = new RawSampleOutputStream(cout);

  if (archiveDir.size() > 0) createArchiver();

  if (buflenMsec > 0) {
    createRawSampleBuffer(buflenMsec);
    _rsbuffer->addRawSampleClient(_rsoutput);
    if (_archiver) _rsbuffer->addRawSampleClient(_archiver);
    _sensorClients.push_back(_rsbuffer);
  }
  else {
    _sensorClients.push_back(_rsoutput);
    if (_archiver) _sensorClients.push_back(_archiver);
  }

  startSensorPortHandler();
  setupSensorPorts();

  createA2D();
  if (_a2dsampler) setupAnalogSampling();

  startRelays();

#ifdef ENABLE_DSCUD
  // create a stepper
  if (_dostepper) {
    _stepper = new Stepper(0x280, 4);
    _stepper->start();
  }
#endif

  // create monitor last
  _monitor = DaqMonitorRPC::createInstance();
  _monitor->start();
}

NDaqSampler::~NDaqSampler() {

  // cerr << "in ~NDaqSampler, deleting _monitor" << endl;
  delete _monitor;

  // cerr << "in ~NDaqSampler, deleting _rrelayListener" << endl;
  delete _rrelayListener;

  // cerr << "in ~NDaqSampler, deleting relay threads" << endl;
  for (list<atdUtil::ThreadRunnable*>::iterator it =
	 _relayThreads.begin(); it != _relayThreads.end(); ++it)
    delete *it;

  // relayPrograms don't belong to me, don't delete them

  // cerr << "in ~NDaqSampler, deleting relays" << endl;
  for (list<Relay*>::iterator it =
	 _relays.begin(); it != _relays.end(); ++it)
    delete *it;

  // cerr << "in ~NDaqSampler, deleting _relayFormatter" << endl;
  delete _relayFormatter;

  // cerr << "in ~NDaqSampler, deleting timer" << endl;
  delete _timer;

  // cerr << "in ~NDaqSampler, deleting _sphandler" << endl;
  delete _sphandler;

  // cerr << "in ~NDaqSampler, deleting _a2dsampler" << endl;
  delete _a2dsampler;

  // cerr << "in ~NDaqSampler, deleting _rsbuffer" << endl;
  delete _rsbuffer;

  // cerr << "in ~NDaqSampler, deleting _rsoutput" << endl;
  delete _rsoutput;

  // cerr << "in ~NDaqSampler, deleting _archiver" << endl;
  delete _archiver;

  _instance = 0;
  // cerr << "end of ~NDaqSampler" << endl;
}

void NDaqSampler::wait() throw(Exception) {

  // cerr << "in NDaqSampler::wait(), sphander->join()" << endl;
  _sphandler->join();

  // cerr << "in NDaqSampler::wait(), timer->join()" << endl;
  if (_timer) _timer->join();

  // cerr << "in NDaqSampler::wait(), relayThreads->join()" << endl;
  for (list<atdUtil::ThreadRunnable*>::iterator it =
	 _relayThreads.begin(); it != _relayThreads.end(); ++it)
    (*it)->join();

  // cerr << "in NDaqSampler::wait(), a2sampler->join()" << endl;
  if (_a2dsampler) _a2dsampler->join();

  // cerr << "in NDaqSampler::wait(), rsbuffer->join()" << endl;
  if (_rsbuffer) _rsbuffer->join();

#ifdef ENABLE_DSCUD
  // cerr << "in NDaqSampler::wait(), stepper join" << endl;
  if (_stepper) _stepper->join();
#endif

  // cerr << "in NDaqSampler::wait(), monitor join" << endl;
  if (_monitor) _monitor->join();

  // cerr << "end of NDaqSampler::wait()" << endl;
}
  
void NDaqSampler::interrupt() {
  // cerr << "in NDaqSampler::interrupt()" << endl;

  if (_monitor) {
    // cerr << "cancelling monitor" << endl;
    _monitor->cancel();
  }

  if (_rrelayListener) {
    // cerr << "doing rrelayListener->cancel()" << endl;
    _rrelayListener->cancel();
  }

  for (list<RelayProgram*>::iterator it =
	 _relayPrograms.begin(); it != _relayPrograms.end(); ++it) {
    for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	 si != _sensorClients.end(); si++)
      (*it)->removeRawSampleClient(*si);
    // OK to remove _relayFormatter even if its not a client
    (*it)->removeRawSampleClient(_relayFormatter);
  }

  // cerr << "in ~NDaqSampler, interrupting relayThreads" << endl;
  for (list<atdUtil::ThreadRunnable*>::iterator it =
	 _relayThreads.begin(); it != _relayThreads.end(); ++it)
    (*it)->interrupt();

  if (_timer) {
    // cerr << "doing _timer->cancel()" << endl;
    _timer->cancel();
  }

  if (_a2dsampler) {
    // cerr << "interrupting a2dsampler" << endl;
    for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	 si != _sensorClients.end(); si++)
      _a2dsampler->removeRawSampleClient(*si);
    _a2dsampler->cancel();
  }

  // cerr << "removing RawSampleBuffer from sensor ports" << endl;
  for (unsigned int i = 0; i < _sensors.size(); i++)
    for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	 si != _sensorClients.end(); si++)
      _sensors[i]->removeRawSampleClient(*si);

  // cerr << "cancelling sphandler" << endl;
  _sphandler->interrupt();
  _sphandler->cancel();

  // cerr << "doing rsbuffer->interrupt()" << endl;
  if (_rsbuffer) {
    if (_rsoutput) _rsbuffer->removeRawSampleClient(_rsoutput);
    if (_archiver) _rsbuffer->removeRawSampleClient(_archiver);
    _rsbuffer->interrupt();
  }

#ifdef ENABLE_DSCUD
  if (_stepper) {
    _stepper->cancel();
  }
#endif

  // cerr << "end of NDaqSampler::interrupt()" << endl;

}

void NDaqSampler::setupSensorPorts() throw(UnknownHostException,IOException,InvalidParameterException) {

  if (getSensorTestDevName().size() > 0) {
    SerialPortConfig cfg;		// default settings
    cfg.setChannel(200);
    cfg.setBaud(9600);
    cfg.setDataBits(8);
    cfg.setStopBits(1);
    cfg.setParity(atdTermio::SerialPort::NONE);
    cfg.setFlowControl(atdTermio::SerialPort::NOFLOWCONTROL);
    // cfg.setRecordLength(0);
    //fg.setRecordDelimiter(string(1,'\n'));
    cfg.setRecordLength(10);
    cfg.setDelimiterAtEOM(true);
    cfg.setRecordDelimiter(string("\x55\xaa"));	// csat3 sonic

    SensorPort *sensor = _sphandler->createSensorPort(getSensorTestDevName(),cfg);
    for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	 si != _sensorClients.end(); si++)
      sensor->addRawSampleClient(*si);
    _sensors.push_back(sensor);
  }
  else {
    RPCSerialPortConfiguration sconfig;
    sconfig.getConfiguration(getDataServerName(),getNodeName());

    ChannelDevNameMap* devs = ChannelDevNameMap::getInstance();

    for (int is = 0; is < sconfig.getNumConfigs(); is++) {
      sleep(1);
      const SerialPortConfig& cfg = sconfig.getConfig(is);

      ChannelDevNameMap::iterator itr = devs->find(cfg.getChannel());

      if (itr == devs->end()) {
#ifdef GPP_2_95_2
	ostrstream os;
#else
	ostringstream os;
#endif
	os << "device for channel " << cfg.getChannel() << " not found" << ends;
	throw InvalidParameterException(os.str());
      }
      string& devname = itr->second;

      SensorPort *sensor = _sphandler->createSensorPort(devname,cfg);
      for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	   si != _sensorClients.end(); si++)
	sensor->addRawSampleClient(*si);
      _sensors.push_back(sensor);
    }
  }
}

void NDaqSampler::setupAnalogSampling() throw(UnknownHostException,IOException,InvalidParameterException) {

  if (getDataServerName().size() == 0) {
    _a2dsampler->addChannel(100,0,1);
  }
  else {

    RPCA2DChannelConfiguration aconfig;
    aconfig.getConfiguration(getDataServerName(),getNodeName());

    if (aconfig.getNumConfigs() == 0) {
      delete _a2dsampler;
      _a2dsampler = 0;
      return;
    }

    const std::vector<float>& rates = _a2dsampler->getAvailableSamplingRates();
    const std::vector<const float*>& ranges = _a2dsampler->getAvailableVoltageRanges();
    
    for (int ia = 0; ia < aconfig.getNumConfigs(); ia++) {
      const A2DChannelConfig& cfg = aconfig.getConfig(ia);
      int rateIndex;
      for (rateIndex = 0; rateIndex < (signed) rates.size(); rateIndex++)
	if (rates[rateIndex] == cfg.getSamplingRate()) break;

      int rangeIndex;
      for (rangeIndex = 0; rangeIndex < (signed) ranges.size(); rangeIndex++)
	if (ranges[rangeIndex][0] == cfg.getMinV() &&
	    ranges[rangeIndex][1] == cfg.getMaxV()) break;

      cerr << "rangeIndex=" << rangeIndex << endl;
      _a2dsampler->addChannel(cfg.getChannel(),rangeIndex,rateIndex);
    }
  }
  _a2dsampler->init();
  for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
       si != _sensorClients.end(); si++)
    _a2dsampler->addRawSampleClient(*si);
  _a2dsampler->start();
}

void NDaqSampler::createRawSampleBuffer(int buflenMsec) {
  // parameter is the sorting buffer length in milliseconds
  _rsbuffer = new RawSampleBuffer(buflenMsec);
  if (_privileged) _rsbuffer->setRealTimeFIFOPriority(40);
  _rsbuffer->start();
}

void NDaqSampler::setArchiveValues(std::string& aDir,
				   std::string& aFormat,
				   int aLengthSecs)
{
  archiveDir = aDir;
  archiveFormat = aFormat;
  archiveLengthSecs = aLengthSecs;
}

void NDaqSampler::createArchiver()
{
  _archiver = new RawSampleArchiver();
  _archiver->setDirectoryName(archiveDir);
  _archiver->setFileNameFormat(archiveFormat);
  _archiver->setFileLength(archiveLengthSecs);
}

void NDaqSampler::startSensorPortHandler() {

  _sphandler = new SensorPortHandler();
  if (_privileged) _sphandler->setRealTimeFIFOPriority(50);
  _sphandler->start();

}

void NDaqSampler::createA2D() throw(IOException) { 

#ifdef ENABLE_DSCUD

  int irq;

  switch (a2d) {
  case MM16AT_A2D:
    irq = 7;
    _a2dsampler = dsc::ExtClockedMM16AT_A2DSampler::createInstance(
								   0x300,irq,differential,bipolar);
    break;
  case MM32XAT_A2D:
    cerr << "MM32XAT in switch" << endl;
    irq = 7;
    _a2dsampler = dsc::ExtClockedMM32XAT_A2DSampler::createInstance(
								    0x300,irq,differential,bipolar);
    break;
  case PROM_A2D:
  default:
    irq = 4;
    _a2dsampler = dsc::ExtClockedPrometheusA2DSampler::createInstance(
								      0x280,irq,differential,bipolar);
    break;
  }

  _a2dsampler->open();
  if (_privileged) _a2dsampler->setRealTimeFIFOPriority(45);
  _a2dsampler->blockSignal(SIGINT);
  _a2dsampler->blockSignal(SIGHUP);
  _a2dsampler->blockSignal(SIGTERM);
#endif

}

void NDaqSampler::addOpto22Relay(RelayProgram* relayProg,
				 const std::string& devName)
{
  Opto22Relay* relay = new Opto22Relay(devName);
  _relays.push_back(relay);
  relayProg->setRelay(relay);
  _relayPrograms.push_back(relayProg);
}

#if defined(ENABLE_DSCUD)
void NDaqSampler::addIR104Relay(RelayProgram* relayProg, int ioport)
  throw(IOException)
{
  // ioport = 0x260;
  int irq = 4;
  dsc::IR104* relay = new dsc::IR104(ioport,irq);
  _relays.push_back(relay);
  relayProg->setRelay(relay);
  _relayPrograms.push_back(relayProg);
}
#endif

void NDaqSampler::startRelays() throw(IOException) {

  if (_relayPrograms.size() > 0) {
    if (!_timer) {
      _timer = new TimerThread("Relay timer");
      if (_privileged) _timer->setRealTimeFIFOPriority(50);
      _timer->setWakeupIntervalMsec(1000);
      _timer->start();
    }

    for (list<RelayProgram*>::iterator it = _relayPrograms.begin();
	 it != _relayPrograms.end(); ++it) {
      RelayProgram* prog = *it;
      prog->setTimerThread(_timer);
      for (list<RawSampleClient*>::iterator si = _sensorClients.begin();
	   si != _sensorClients.end(); si++)
	prog->addRawSampleClient(*si);

      atdUtil::ThreadRunnable* relayThread =
      	new ThreadRunnable("Relay",prog);
      if (_privileged) relayThread->setRealTimeFIFOPriority(45);
      relayThread->blockSignal(SIGINT);
      relayThread->blockSignal(SIGHUP);
      relayThread->blockSignal(SIGTERM);
      relayThread->start();
      _relayThreads.push_back(relayThread);
    }

    RelayProgram* prog = *_relayPrograms.begin();
    _relayFormatter = new SampleFormatter();
    _relayFormatter->addVariable(SampleFormatter::RELAY,"relay",164);

    prog->addRawSampleClient(_relayFormatter);

    _rrelayListener = new RrelayListener(*prog,_relayFormatter);
    _rrelayListener->start();
  }
}

void NDaqSampler::quit() {
  // cerr << "in NDaqSampler::quit()" << endl;
  _quit = true;
  interrupt();
  // cerr << "end of NDaqSampler::quit()" << endl;
}
  
struct RelayInfo {
  string relayProgFile;
  string relayProgName;
  string relayDevice;
  string channelNum;
};


class Runstring {
public:
  Runstring(int argc, char** argv): syslogit(true),
				    dostepper(false),
				    bipolar(false),
				    differential(false),
				    a2d(NDaqSampler::PROM_A2D),
				    buflenMsec(2000) {
    extern char *optarg;       /* set by getopt() */
    extern int optind;       /* "  "     "     */
    int opt_char;     /* option character */
    while ((opt_char = getopt(argc, argv, "a:Bb:c:Dd:eMr:s:SXT")) != -1) {
      switch (opt_char) {
      case 'a':
	{
	  archiveDir = optarg;
	  // archive file name format must be next argument
	  if (optind == argc || argv[optind][0] == '-')
	    usage(argv[0]);
	  archiveFormat = argv[optind++];
	  // file length in secs is next arg
	  if (optind == argc || argv[optind][0] == '-')
	    usage(argv[0]);
	  archiveLengthSecs = atoi(argv[optind++]);
	}
	break;
      case 'b':
	{
	  istringstream ist(optarg);
	  ist >> buflenMsec;
	  if (ist.fail() || buflenMsec < 0) usage(argv[0]);
	}
	break;
      case 'B':
	bipolar = true;
	break;
      case 'c':
	serialportmap = optarg;
	break;
      case 'd':
	devname = optarg;
	break;
      case 'D':
	differential = true;
	break;
      case 'M':
	a2d = NDaqSampler::MM16AT_A2D;
	break;
      case 'X':
	a2d = NDaqSampler::MM32XAT_A2D;
	break;
      case 'e':
        syslogit = false;
	break;
      case 'r':
	{
	  struct RelayInfo ri;
	  ri.relayProgFile = optarg;
	  // relay program name must be next argument
	  if (optind == argc || argv[optind][0] == '-')
	    usage(argv[0]);
	  ri.relayProgName = argv[optind++];
	  // opto22 device name or IR104 ioport is optional next argument
	  if (optind < argc && argv[optind][0] != '-')
	    ri.relayDevice = argv[optind++];
	  // channel number
	  if (optind < argc && argv[optind][0] != '-')
	    ri.channelNum = argv[optind++];
	  relays.push_back(ri);
	}
	break;
      case 'S':
        syslogit = true;
	break;
      case 'T':
        dostepper = true;
	break;
      case 's':
	nodename = optarg;
	// server name must be next argument
	if (optind == argc || argv[optind][0] == '-')
	  usage(argv[0]);
	servername = argv[optind++];
	break;
      case '?':
	usage(argv[0]);
	break;
      }
    }
    if (optind != argc) usage(argv[0]);
  }

  string serialportmap;
  string servername;
  string nodename;
  string devname;
  string archiveDir;
  string archiveFormat;
  int archiveLengthSecs;
  vector<struct RelayInfo> relays;
  bool syslogit;
  bool dostepper;
  bool bipolar;
  bool differential;
  NDaqSampler::A2D_t a2d;
  int buflenMsec;

  static void usage(const char* argv0) {
    cerr << "Usage: " << argv0 << "[-a adir afile alen] [-B] [-D] [-T] [-S] [-e]\n\
        [-s nodename servername] [-d devname] [-c serialportmap]\n\
        [-r relayFileName relayProgName [dev] [chan] ]\n\n\
  -a adir afile alen: archive directory, file and file length(secs)\n\
      file can contain strftime time format descriptors\n\
      Example: -a /data cme_%Y%m%d_%H%M%S.dat 86400\n\
        creates day-long files named like /data/cme_20050519_000000.dat\n\
  -B: bipolar A2D (must match jumper setting). Default: unipolar.\n\
  -D: differential A2D (must match jumper setting). Default: single-ended.\n\
  -T: enable stepper motor control\n\
  -S: enable syslog logging\n\
  -e: status messages to stderr, otherwise to syslog\n\
  -s: fetch data-acq configuration for nodename from servername, then\n\
      open socket for data transmission to servername\n\
  -d: open a sensor port, e.g. /dev/ttyS0 for testing of data acq code\n\
      Either -s or -d option should be specified\n\
  -c: name of a file containing a mapping between serial port channel\n\
      numbers and the corresponding /dev/ttySX device name.\n\
      Example:\n\
      200	/dev/ttyS0\n\
      201	/dev/ttyS2\n\
  -M: use DSC MM16AT A2D on irq=7, ioport=0x300.\n\
      Default: prometheus A2D on irq=4,ioport=0x280.\n\
  -X: use DSC MM32XAT A2D on irq=7, ioport=0x300.\n\
      Default: prometheus A2D on irq=4,ioport=0x280.\n\
  -r: parse relayFileName for relay programs, then execute program\n\
      named relayProgName.  Optional dev is ioport address of IR104\n\
      relay board (default: 0x260) or serial port name (/dev/ttyX) of Opto22\n\
      relay controller. chan is optional channel number for\n\
      relay data sent to server (default: 164)\n\
      Zero or more -r relay options can be specified.\n\
" << endl;
    exit(1);
  }
};

/* static */
int NDaqSampler::main(int argc, char** argv) {

  time(&NDaqSampler::_startTime);
  Runstring runstring(argc,argv);

  if (runstring.syslogit)
    Logger::createInstance("ndaq",LOG_CONS | LOG_PID, LOG_LOCAL4,(char *)0);
  else
    Logger::createInstance(stderr);

  int status = 0;
  try {
    bool quit = false;
    while (!quit) {
      NDaqSampler* sampler = new NDaqSampler();
      sampler->setA2D(runstring.a2d,
		      runstring.bipolar,runstring.differential);

      sampler->setBuflenMsec(runstring.buflenMsec);

      vector<RelayParser*> relayParsers;

      if (runstring.serialportmap.size() > 0) {
	ChannelDevNameMap::getInstance()->read(runstring.serialportmap);
	if (ChannelDevNameMap::getInstance()->size() == 0)
	  throw IOException(runstring.serialportmap,"open","file not found");
      }

      for (unsigned int i = 0; i < runstring.relays.size(); i++) {
	// RelayParser owns the RelayProgram, so we
	// don't want to delete the parsers until the
	// Sampler is finished.
	RelayParser* relayParser = new RelayParser();
	relayParsers.push_back(relayParser);

	relayParser->parse(runstring.relays[i].relayProgFile);
	RelayProgram* relayProgram 
	  = &relayParser->getProgram(runstring.relays[i].relayProgName);

#ifdef ENABLE_DSCUD
	if (runstring.relays[i].relayDevice.size() == 0)
	  sampler->addIR104Relay(relayProgram,0x260);
	else
#endif
	  if (runstring.relays[i].relayDevice[0] == '/')
	    sampler->addOpto22Relay(relayProgram,runstring.relays[i].relayDevice);
#ifdef ENABLE_DSCUD
	  else {
	    istringstream istr(runstring.relays[i].relayDevice);
	    unsigned int ioport;
	    istr >> hex >> ioport;
	    if (!istr) Runstring::usage(argv[0]);
	    sampler->addIR104Relay(relayProgram,ioport);
	  }
#else
	else Runstring::usage(argv[0]);
#endif
	if (runstring.relays[i].channelNum.size() > 0) {
	  istringstream istr(runstring.relays[i].channelNum);
	  int channel;
	  istr >> channel;
	  if (!istr) Runstring::usage(argv[0]);
	  relayProgram->setChannel(channel);
	}
      }

      if (runstring.servername.size() > 0) {
        sampler->setDataServerName(runstring.servername);
        sampler->setNodeName(runstring.nodename);
      }
      else if (runstring.devname.size() > 0)
	sampler->setSensorTestDevName(runstring.devname);
      else Runstring::usage(argv[0]);

      if (runstring.archiveDir.size() > 0)
        sampler->setArchiveValues(runstring.archiveDir,
				  runstring.archiveFormat,
				  runstring.archiveLengthSecs);

#ifdef ENABLE_DSCUD
      if (runstring.dostepper)
	sampler->enableStepper();
#endif

      sampler->init();

      struct timespec stime;
      stime.tv_sec = 20;
      stime.tv_nsec = 0;
      nanosleep(&stime,0);

#ifdef  DO_MLOCK
      if (mlockall(MCL_CURRENT) < 0) {
	Exception e("mlockall",Exception::errnoToString(errno));
	Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());
	struct rlimit mlimit;
	if (getrlimit(RLIMIT_MEMLOCK,&mlimit) < 0) {
	  Exception e("getrlimit",Exception::errnoToString(errno));
	  Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());
	}
	else
	  Logger::getInstance()->log(LOG_INFO,"getrlimit RLIMIT_MEMLOCK: soft limit=%d, hard limit=%d",mlimit.rlim_cur,mlimit.rlim_max);
      }
#endif

      sampler->wait();
      quit = sampler->getQuit();

      cerr << "deleting sampler" << endl;
      delete sampler;

      cerr << "deleting relayParsers" << endl;
      for (unsigned int i = 0; i < relayParsers.size(); i++)
        delete relayParsers[i];

      if (!quit) {
	cerr << "sleeping 20" << endl;
        sleep(20);
      }
      if (munlockall() < 0) {
	Exception e("munlockall",Exception::errnoToString(errno));
	Logger::getInstance()->log(LOG_WARNING,"Warning: %s ",e.what());
      }

    }
#if defined(ENABLE_DSCUD)
    dsc::DiamondUD::freeUD();
#endif
  }
  catch (UnknownHostException &uhe) {
    cerr << "UnknownHostException in main: " << uhe.toString() << endl;
    status = 1;
  }
  catch (IOException &se) {
    cerr << "IOException in main: " << se.toString() << endl;
    status = 1;
  }
  catch (Exception &e) {
    cerr << "Exception in main: " << e.toString() << endl;
    status = 1;
  }
  return status;
}

