/*
 ******************************************************************
    Copyright 2005 UCAR, NCAR, All Rights Reserved

    $LastChangedDate: 2005-09-22 12:59:20 -0600 (Thu, 22 Sep 2005) $

    $LastChangedRevision: 2965 $

    $LastChangedBy: maclean $

    $HeadURL: http://svn/svn/nids/trunk/dsm/class/DSMSerialSensor.h $

 ******************************************************************
*/
#ifndef ATDISFF_DIGITALIO_H
#define ATDISFF_DIGITALIO_H

#include <atdUtil/BitArray.h>
#include <atdUtil/IOException.h>

namespace atdISFF {

/**
 * Abstract class for controlling digital output lines.
 */
class DigitalOut
{
public:

    virtual ~DigitalOut() {}

    /**
     * Set the OS device name for the digital out device.
     */
    virtual void setDeviceName(const std::string& val) = 0;

    virtual const std::string& getDeviceName() const = 0;

    /**
     * Open the device.
     */
    virtual void open() throw(atdUtil::IOException) = 0;

    /**
     * Set the name and open.
     */
    virtual void open(const std::string& devname) throw(atdUtil::IOException)
    	= 0;

    virtual void close() = 0;

    /**
     * Query the number of lines available on the device.
     */
    virtual int getNumberOfOutputs() throw(atdUtil::IOException) = 0;

    /**
     * Request exclusive control of specified lines. Other processes
     * may open the device and control other lines.
     * @param mask A BitArray indicating which lines are to be
     *      controlled.
     * @return BitArray of requested lines.
     */
    virtual void reserveOutputs(const atdUtil::BitArray& lines) throw(atdUtil::IOException)
    	= 0;

    /**
     * Query requested lines.
     * @return atdUtil::BitArray of requested lines.
     */
    virtual const atdUtil::BitArray& getReservedOutputs() const throw(atdUtil::IOException) = 0;

    /**
     * Set specified output lines high (1) or low(0).
     * @param lines A atdUtil::BitArray specifiying which lines to set.
     * @param state A atdUtil::BitArray specifying desired state
     *      of lines specified in lines.  A bit value of 0
     *      means set low, 1=set high.
     */
    virtual void write(const atdUtil::BitArray& lines,const atdUtil::BitArray& state)
    	throw(atdUtil::IOException) = 0;

    /**
     * Set specified output lines high (1).
     * @param lines A atdUtil::BitArray specifiying which lines to set.
     */
    virtual void setHigh(const atdUtil::BitArray& lines) throw(atdUtil::IOException) 
    {
        atdUtil::BitArray newstate = lines;
	write(lines,newstate);
    }

    /**
     * Set specified output lines low (0).
     * @param lines A atdUtil::BitArray specifiying which lines to set.
     */
    virtual void setLow(const atdUtil::BitArray& lines) throw(atdUtil::IOException) 
    {
        atdUtil::BitArray newstate(lines.getLength());
	write(lines,newstate);
    }

    /**
     * Set all requested lines high.
     */
    virtual void setHigh() throw(atdUtil::IOException)
    {
        atdUtil::BitArray newstate = getReservedOutputs();
	write(newstate,newstate);
    }

    /**
     * Set all requested lines low.
     */
    virtual void setLow() throw(atdUtil::IOException) 
    {
        atdUtil::BitArray newstate(getReservedOutputs().getLength());
	write(getReservedOutputs(),newstate);
    }

    /**
     * Set a line high.
     */
    virtual void setHigh(int line) throw(atdUtil::IOException)
    {
        atdUtil::BitArray state = read(getReservedOutputs());
	state.setBit(line,true);
	write(getReservedOutputs(),state);
    }

    /**
     * Set a line high.
     */
    virtual void setLow(int line) throw(atdUtil::IOException)
    {
        atdUtil::BitArray state = read(getReservedOutputs());
	state.setBit(line,false);
	write(getReservedOutputs(),state);
    }

    /**
     * Get state of specified output lines.
     * @param mask A atdUtil::BitArray of which lines to get.
     * @return A atdUtil::BitArray indicating state of of lines.
     */
    virtual atdUtil::BitArray read(const atdUtil::BitArray& lines)
    	throw(atdUtil::IOException) = 0;

    virtual atdUtil::BitArray read() throw(atdUtil::IOException)
    {
        return read(getReservedOutputs());
    }

};

/**
 * Abstract class for reading digital input lines.
 */
class DigitalIn
{
public:

    virtual ~DigitalIn() {}

    virtual void setDeviceName(const std::string& val) = 0;

    virtual void open() throw(atdUtil::IOException) = 0;

    virtual void close() = 0;

    virtual void open(const std::string& devname) throw(atdUtil::IOException) = 0;

    /**
     * Query the number of lines available on the device.
     */
    virtual int getNumberOfInputs() throw(atdUtil::IOException) = 0;

    /**
     * Request input of specified lines.
     * @param mask A atdUtil::BitArray indicating which lines are to be
     *      controlled.
     * @return atdUtil::BitArray of requested lines.
     */
    virtual void reserveInputs(const atdUtil::BitArray& lines) throw(atdUtil::IOException) = 0;

    /**
     * Query requested lines.
     * @return atdUtil::BitArray of requested lines.
     */
    virtual const atdUtil::BitArray& getReservedInputs() const throw(atdUtil::IOException) = 0;

    /**
     * Get state of specified input lines.
     * @param mask A atdUtil::BitArray of which lines to get.
     * @return A atdUtil::BitArray indicating state of of lines.
     */
    virtual atdUtil::BitArray read(const atdUtil::BitArray& lines) throw(atdUtil::IOException) = 0;

    virtual atdUtil::BitArray read() throw(atdUtil::IOException)
    {
        return read(getReservedInputs());
    }


};

}

#endif
