// Constants for daq motor routines

#ifndef DAQMOTOR_H_
#define DAQMOTOR_H_

#include <string>
#include <atdISFF/Stepper.h>

namespace atdISFF {

  /// DaqMotor represents an NDAQ stepper motor, from the networked
  /// client side. It manages the RPC calls used for controlling
  /// a Stepper on NDAQ.
  class DaqMotor {
  public: 

    /// param host The remote NDAQ host
    DaqMotor(std::string host);

    ///
    ~DaqMotor();

    /// Move the motor the specified number of steps.
    /// param steps The number of steps to move. Positive means clockwise,
    /// negative means counter clockwise.
    /// @param motorNumber The motor to be commanded.
    /// @return True if command was accepted, false otherwise
    bool move(int motorNumber, double steps);

    /// Move the motor to the zero position.
    /// returns 0 if the RPC call succeeded, -1 otherwise.
    /// @param motorNumber The motor to be commanded.
    /// @return True if command was accepted, false otherwise
    bool zero(int motorNumber);

    /// reset the stepper
    /// @param motorNumber The motor to be commanded.
    /// @return True if command was accepted, false otherwise
    bool reset(int motorNumber);

    /// Get the status.
    /// @param motorNumber The motor to be commanded.
    /// @param status The current motor status, from atdISFF::Stepper::StepperStatus,
    /// is returned through this variable.
    /// @param position The current position is returned through this variable.
    /// @return True if command was accepted, false otherwise
    bool status(int motorNumber,
		atdISFF::Stepper::StepperStatus& status, 
		double& position);

  protected:
    /// The NDAQ host
    std::string _host;
    /// Execute the RPC.
    /// param motorNumber The motor to be commanded.
    /// param cmd The command code that will be sent by RPC. 
    /// param arg1 A possible argument to the command
    bool daqMotorCall(int motorNumber,
		      atdISFF::Stepper::StepperCommand cmd, 
		      double arg);
    /// The status as returned by a status request
    atdISFF::Stepper::StepperStatus _status;
    /// The position, as returned by a status request
    double _position;
  };
}

#endif
