/*              Copyright 2004 (C) by UCAR
 *
 * File       : DaqMonitorRPC.cpp,v
 * Revision   : 1.19
 * Directory  : /code/cvs/isa/src/lib/atdISFF/DaqMonitorRPC.cpp,v
 * System     : PAM
 * Date       : 2006/02/27 00:17:03
 *
 * Description:
 *   Class containing information (error counters, event counters, sample rates)
 *   about the status of the data acquisition system
 *
 * Things to report:
 *   current time (client can then check the difference)
 *   time of initial start
 *   time of last restart
 *   software version
 *   for each sensor port: minRead, maxRead, samples/sec, cumulative errors
 *   select timeouts (5 minute and cumulative), select errors(cumulative)
 *   rserial errors
 *   socket: samples/sec, errors (in last 5 minutes and cumulative),
 *      5 minute maxWriteLen, minWriteLen, and cumulative
 *	
 * DaqMonitorRPC.cpp,v
 * Revision 1.19  2006/02/27 00:17:03  martinc
 * added motor number to Stepper control system
 *
 * Revision 1.18  2006/01/28 23:12:03  maclean
 * more fiddles for non-dscud builds
 *
 * Revision 1.17  2006/01/28 22:27:53  maclean
 * changes so that things build if not using Diamond System universal driver
 *
 * Revision 1.16  2006/01/28 04:05:53  martinc
 * Log a message on each stepper command request
 *
 * Revision 1.15  2006/01/28 03:45:03  martinc
 * completely reworked to place stepper functions in a thread so that they wdo not block the rpc handling in DaqMonitorRPC.
 *
 * Revision 1.14  2006/01/26 04:57:43  martinc
 * major refactoring of motor control. Created Stepper class for instatiating a motor on ndaq, and DaqMotor class for the remote manifestation of the Stepper. Stepper is created by DaqMonitor. The motor rpc calls are forwarded onto Stepper.
 *
 * Revision 1.13  2006/01/25 20:13:41  martinc
 * Added parameter passing for daqMotor_1_svc
 *
 * Revision 1.12  2006/01/25 05:36:13  martinc
 * Added stub for motor control.
 *
 * Fixed calls to A2DSampler::getSamplingRate(i) and gA2DSampler::etVoltageRange(i).
 * A2DSampler.h says that the parameter should be a zero based index, but the
 * derived class DscA2DSampler shows 100 being subtracted, and so I guess that
 * it should be the actual channel number (100 based). This was causing the
 * daqStatus RPC to segv.
 *
 * Revision 1.11  2005/11/16 18:17:49  maclean
 * fixed a bunch of compiler warnings
 *
 * Revision 1.10  2005/01/16 01:34:21  maclean
 * added copyright
 *
 * Revision 1.9  2004/10/13 16:03:12  maclean
 * derived thread classes block signals in their own constructor
 *
 * Revision 1.8  2004/10/07 18:56:47  maclean
 * use amInterrupted within a run method
 *
 * Revision 1.7  2004/08/30 05:14:03  maclean
 * *** empty log message ***
 *
 * Revision 1.6  2004/08/13 21:07:32  maclean
 * added support for RawSampleArchiver
 *
 * Revision 1.5  2004/08/12 20:20:09  maclean
 * added support for report overflow counts
 *
 * Revision 1.4  2004/06/26 05:37:04  maclean
 * *** empty log message ***
 *
 * Revision 1.3  2004/06/23 14:25:27  maclean
 * reorg. Added RawSampleClient and RawSampleSource, and a source can have multiple clients
 *
 * Revision 1.2  2004/05/25 16:45:01  maclean
 * Initial version for Brit Stephens, OHATS test
 *
 * Revision 1.1.1.1  2004/03/18 14:49:51  maclean
 * first import
 *
 */

#include <atdISFF/DaqMonitorRPC.h>
#include <atdISFF/NDaqSampler.h>
#include <atdISFF/A2DSampler.h>

#ifdef ENABLE_DSCUD
#include <atdISFF/DaqMotor.h>
#endif

#include <atdUtil/Logger.h>
#include <errno.h>
#include <signal.h>
#include <rpc/pmap_clnt.h>

using namespace atdISFF;
using namespace atdUtil;
using namespace std;

//-----------------------------------------------------------------------------
extern "C" {
  void daqmonitor_prog_1(svc_req*, SVCXPRT*);
}

//-----------------------------------------------------------------------------
daqStatus_res* daqstatus_1_svc(struct svc_req* req) {
  return DaqMonitorRPC::getInstance()->doStatus();
}

//-----------------------------------------------------------------------------
void* daqrestart_1_svc(struct svc_req* req) {
  return DaqMonitorRPC::getInstance()->doRestart();
}

//-----------------------------------------------------------------------------
void* daqstop_1_svc(struct svc_req* req) {
  return DaqMonitorRPC::getInstance()->doStop();
}

#ifdef ENABLE_DSCUD
//-----------------------------------------------------------------------------
daqMotor_res* daqmotor_1_svc(daqMotorCmd cmd, struct svc_req* req) {
  return DaqMonitorRPC::getInstance()->doMotor(&cmd);
}
#else
//-----------------------------------------------------------------------------
daqMotor_res* daqmotor_1_svc(daqMotorCmd cmd, struct svc_req* req) {
  return 0;
}
#endif

//-----------------------------------------------------------------------------
/* static */
DaqMonitorRPC* DaqMonitorRPC::_instance = 0;

//-----------------------------------------------------------------------------
/* static */
DaqMonitorRPC* DaqMonitorRPC::createInstance() throw(IOException)
{
  if (_instance) return _instance;
  _instance = new DaqMonitorRPC();
  return _instance;
}

//-----------------------------------------------------------------------------
DaqMonitorRPC::DaqMonitorRPC() 
  throw(IOException): Thread("DaqMonitorRPC"),
		      _udptransport(0),
		      _tcptransport(0)
{
  blockSignal(SIGINT);
  blockSignal(SIGHUP);
  blockSignal(SIGTERM);
}

//-----------------------------------------------------------------------------
DaqMonitorRPC::~DaqMonitorRPC()
{
  _instance = 0;
}

//-----------------------------------------------------------------------------
int 
DaqMonitorRPC::run() throw(Exception)
{

  /* apparently you must create these transports
   * in the same thread that svc_run runs in.
   */
  (void) pmap_unset(DAQMONITOR_PROG, DAQMONITOR_VERS);
                                                                                
  _udptransport = svcudp_create(RPC_ANYSOCK);
  if (_udptransport == NULL) {
    IOException e("DaqMonitorRPC service","svcudp_create",errno);
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str());
    throw e;
  }
  
  if (!svc_register(_udptransport, DAQMONITOR_PROG, DAQMONITOR_VERS,
		    daqmonitor_prog_1, IPPROTO_UDP)) {
    IOException e("DaqMonitorRPC service","svc_register",errno);
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str());
    throw e;
  }

#ifdef DO_DAQMONITOR_TCP
  _tcptransport = svctcp_create(RPC_ANYSOCK,0,0);
  if (_tcptransport == NULL) {
    IOException e("DaqMonitorRPC service","svctcp_create",errno);
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str());
    throw e;
  }
  
  if (!svc_register(_tcptransport, DAQMONITOR_PROG, DAQMONITOR_VERS,
		    daqmonitor_prog_1, IPPROTO_TCP)) {
    IOException e("DaqMonitorRPC service","svc_register",errno);
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str());
    throw e;
  }
#endif

  // svc_run() ignores errors, like EINTR.  We don't want that
  // since we want to quit on receipt of a signal, so
  // we do our own select.

  int selectn = 0;
  // RPC file descriptors
  for (int i = 0; i < FD_SETSIZE; i++)
    if (FD_ISSET(i, &svc_fdset) && i > selectn) selectn = i;
  selectn++;

  for (;;) {
    if (amInterrupted()) break;
    fd_set rset = svc_fdset;
    int nfdsel = ::select(selectn,&rset,0,0,0);
    if (nfdsel < 0) {
      // Report IOException.
      IOException e("DaqMonitorRPC","select",errno);
      Logger::getInstance()->log(LOG_INFO,"%s",e.toString().c_str());
    }
    else svc_getreqset(&rset);
  }
  svc_unregister(DAQMONITOR_PROG, DAQMONITOR_VERS);
  if (_udptransport) svc_destroy(_udptransport);
  if (_tcptransport) svc_destroy(_tcptransport);
  return RUN_OK;
}

//-----------------------------------------------------------------------------
daqStatus_res* DaqMonitorRPC::doStatus() {
  static daqStatus_res res;
  xdr_free((bool_t(*)(XDR*,void*,...))xdr_daqStatus_res,(char *)&res);

  NDaqSampler *sampler = NDaqSampler::getNDaqInstance();
  RawSampleSocket *socket = sampler->getRawSampleSocket();	// could be null
  RawSampleBuffer *buffer = sampler->getRawSampleBuffer();
  // SensorPortHandler *sph = sampler->getSensorPortHandler();
  A2DSampler *a2d = sampler->getA2DSampler();	// could be null
  RawSampleArchiver *archiver = sampler->getRawSampleArchiver();	// could be null

  struct timeval tval;
  ::gettimeofday(&tval,0);
  res.daqStatus_res_u.status.currentTimeSecs = tval.tv_sec;
  res.daqStatus_res_u.status.currentTimeUsecs = tval.tv_usec;
  res.daqStatus_res_u.status.startTime = sampler->getStartTime();
  res.daqStatus_res_u.status.restartTime = sampler->getRestartTime();

  if (socket) {
    res.daqStatus_res_u.status.lostSamples = socket->getNumLostSamples();
    res.daqStatus_res_u.status.numWriteErrors =
      socket->getNumWriteErrors();
    res.daqStatus_res_u.status.numWriteTempUnavailable =
      socket->getNumWriteTempUnavailable();
    res.daqStatus_res_u.status.maxWriteLength =
      socket->getMaxWriteLength();
    res.daqStatus_res_u.status.minWriteLength =
      socket->getMinWriteLength();
    res.daqStatus_res_u.status.samplesPerSec =
      socket->getSamplesPerSec();
    res.daqStatus_res_u.status.bytesPerSec =
      socket->getBytesPerSec();
  }
  else {
    res.daqStatus_res_u.status.lostSamples = 0;
    res.daqStatus_res_u.status.numWriteErrors = 0;
    res.daqStatus_res_u.status.numWriteTempUnavailable = 0;
    res.daqStatus_res_u.status.maxWriteLength = 0;
    res.daqStatus_res_u.status.minWriteLength = 0;
    res.daqStatus_res_u.status.samplesPerSec = 0;
    res.daqStatus_res_u.status.bytesPerSec = 0;
  }
  res.daqStatus_res_u.status.rawSampleSetSize =
    buffer->getRawSampleSetSize();
  res.daqStatus_res_u.status.rawSamplesInMemory =
    buffer->getRawSamplesInMemory();

  if (archiver)
    res.daqStatus_res_u.status.badArchiveTimeTags =
      archiver->getBadTimeTagCount();
  else
    res.daqStatus_res_u.status.badArchiveTimeTags = 0;

  res.daqStatus_res_u.status.softversion =
    (char*)malloc(strlen(sampler->getVersion().c_str())+1);
  strcpy(res.daqStatus_res_u.status.softversion,sampler->getVersion().c_str());

  if (a2d) {
    res.daqStatus_res_u.status.a2d.missedSamples = a2d->getMissedSamples();
    res.daqStatus_res_u.status.a2d.cumulativeMissedSamples =
      a2d->getCumulativeMissedSamples();
    res.daqStatus_res_u.status.a2d.totalSamplesPerSec = a2d->getTotalObservedSamplingRate();
    res.daqStatus_res_u.status.a2d.userInterruptRate = a2d->getUserInterruptRate();
    std::vector<int> a2dchans = a2d->getUserChannelNumbers();
    a2dChannelStatus** a2dpp = &res.daqStatus_res_u.status.a2ds;
    for (unsigned int i = 0; i < a2dchans.size(); i++) {
      a2dChannelStatus* a2dp;
      *a2dpp = a2dp = (a2dChannelStatus*) malloc(sizeof(a2dChannelStatus));
      a2dp->channelNum = a2dchans[i];
      a2dp->sampleRate = a2d->getSamplingRate(a2dp->channelNum);
      const float* vrange = a2d->getVoltageRange(a2dp->channelNum);
      a2dp->minV = vrange[0];
      a2dp->maxV = vrange[1];
      a2dp->uVresolution = vrange[2];
      a2dpp = &a2dp->next;
    }
    *a2dpp = 0;
  }
  else {
    res.daqStatus_res_u.status.a2d.missedSamples = 0;
    res.daqStatus_res_u.status.a2d.cumulativeMissedSamples = 0;
    res.daqStatus_res_u.status.a2d.totalSamplesPerSec = 0.0;
    res.daqStatus_res_u.status.a2d.userInterruptRate = 0.0;
    res.daqStatus_res_u.status.a2ds = 0;
  }
  
  sensorPortStatus** spspp = &res.daqStatus_res_u.status.ports;
  for (int i = 0; i < sampler->getNumSensors(); i++) {
    string tmp;
    sensorPortStatus* spsp;
    *spspp = spsp = (sensorPortStatus*) malloc(sizeof(sensorPortStatus));
    SensorPort* sp = sampler->getSensor(i);

    spsp->channelNum = sp->getChannel();
    spsp->devname = (char *)malloc(sp->getName().size() + 1);
    strcpy(spsp->devname,sp->getName().c_str());

    spsp->channelNum = sp->getChannel();
    spsp->baudRate = sp->getBaudRate();

    tmp = sp->getParityString();
    spsp->parity = (char*) malloc(tmp.size() + 1);
    strcpy(spsp->parity,tmp.c_str());

    spsp->dataBits = sp->getDataBits();
    spsp->stopBits = sp->getStopBits();

    tmp = sp->getFlowControlString();
    spsp->flowcontrol = (char*) malloc(tmp.size() + 1);
    strcpy(spsp->flowcontrol,tmp.c_str());

    // user should make this printable
    tmp = sp->getMessageSeparator();
    spsp->messageSeparator = (char *)malloc(tmp.size() + 1);
    strcpy(spsp->messageSeparator,tmp.c_str());

    spsp->separatorAtEOM = sp->getSeparatorAtEOM();

    spsp->messageLength = sp->getMessageLength();

    spsp->samplesPerSecond = sp->getObservedSamplingRate();
    spsp->readsPerSecond = sp->getReadRate();
    spsp->minReadLength = sp->getMinReadLength();
    spsp->maxReadLength = sp->getMaxReadLength();
    spsp->readErrorCount = sp->getReadErrorCount();
    spsp->writeErrorCount = sp->getWriteErrorCount();
    spsp->cumReadErrorCount = sp->getCumulativeReadErrorCount();
    spsp->cumWriteErrorCount = sp->getCumulativeWriteErrorCount();
    spsp->bufferOverflowCount = sp->getBufferOverflowCount();
    spsp->cumBufferOverflowCount = sp->getCumulativeBufferOverflowCount();

    spspp = &spsp->next;
  }
  *spspp = 0;

  res.ierr = 0;
  return &res;
}

//-----------------------------------------------------------------------------
void* DaqMonitorRPC::doRestart() {
  cerr << "doRestart" << endl;
  try {
    Sampler::getInstance()->kill(SIGHUP);
  }
  catch (Exception& e) {
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str()); 
  }
  return 0;
}

//-----------------------------------------------------------------------------
void* DaqMonitorRPC::doStop() {
  cerr << "doStop" << endl;

  try {
    Sampler::getInstance()->kill(SIGTERM);
  }
  catch (Exception& e) {
    Logger::getInstance()->log(LOG_ERR,"%s",e.toString().c_str()); 
  }
  return 0;
}

#ifdef ENABLE_DSCUD
//-----------------------------------------------------------------------------
daqMotor_res* 
DaqMonitorRPC::doMotor(daqMotorCmd* cmd) {

  int ierr = 0;

  Logger* logger = Logger::getInstance();
  NDaqSampler *sampler = NDaqSampler::getNDaqInstance();

  atdISFF::Stepper* stepper = sampler->getStepper();

  bool cmdSuccess = false;
  atdISFF::Stepper::StepperStatus stepperStatus = atdISFF::Stepper::STEPPERSTATUS_FAULT;
  double stepperPosition = 0.0;

  if (stepper) {
    switch (cmd->command) {
    case atdISFF::Stepper::STEPPERCMD_RESET: {
      logger->log(LOG_INFO, "DaqMonitorRPC::doMotor() stepper reset");
      cmdSuccess = stepper->reset(cmd->motorNumber);
      break;
    }
    case atdISFF::Stepper::STEPPERCMD_ZERO: {
      logger->log(LOG_INFO, "DaqMonitorRPC::doMotor() stepper zero");
      cmdSuccess = stepper->zero(cmd->motorNumber);
      break;
    }
    case atdISFF::Stepper::STEPPERCMD_MOVE: {
      logger->log(LOG_INFO, "DaqMonitorRPC() stepper move");
      cmdSuccess = stepper->move(cmd->motorNumber, cmd->arg1);
      break;
    }
    case atdISFF::Stepper::STEPPERCMD_STATUS: {
      logger->log(LOG_INFO, "DaqMonitorRPC() stepper status");
      stepperStatus   = stepper->status(cmd->motorNumber);
      stepperPosition = stepper->position(cmd->motorNumber);
      cmdSuccess      = true;
      break;
    }
    default: {
      Logger::getInstance()->log(LOG_ERR,"DaqMonitor::doMotor() Invalid motor command received (%d)", 
				 cmd->command); 
      ierr = 1;
      break;
    }
    }
  } else {
    Logger::getInstance()->log(LOG_ERR,
			       "DaqMonitor::doMotor() Motor command received, but motor is not enabled"); 
    ierr = 1;
  }

  static daqMotor_res res;
  xdr_free((bool_t(*)(XDR*,void*,...))xdr_daqMotor_res,(char *)&res);
  res.daqMotor_res_u.status.cmdSuccess  = cmdSuccess;
  res.daqMotor_res_u.status.status      = stepperStatus;
  res.daqMotor_res_u.status.position    = stepperPosition;
  res.ierr = ierr;

  return &res;
}
#endif

//-----------------------------------------------------------------------------
