//
//              Copyright 2004 (C) by UCAR
//

// #include <vector>

#include <atdISFF/CSAT3AsciiOutput.h>
#include <atdUtil/IOException.h>
#include <math.h>
#include <iostream>
#include <iomanip>

using namespace atdISFF;
using namespace std;

CSAT3AsciiOutput::CSAT3AsciiOutput(ostream& ostr) :
    ost(ostr),
    // csat data is little endian
    conv(atdUtil::NumConverter::getNumConverter(
    	atdUtil::NumConverter::NC_LITTLE_ENDIAN)),
    lastcnt(-1),
    FLOAT_NAN(nanf(""))
{
}

bool CSAT3AsciiOutput::receive(const RawSample *sample)
	throw(atdUtil::IOException) {

    // csat wind scaling factors
    static const float scale[]={0.002,0.001,0.0005,0.00025};
  
    // get csat3 diagnostics word
    unsigned short diag = 
	conv->ushortValue(sample->data+(4*sizeof(short)));
    // decode scale factors
    int range[3];
    range[0] = (diag & 0x0c00) >> 10;
    range[1] = (diag & 0x0300) >> 8;
    range[2] = (diag & 0x00c0) >> 6;

    // diagnostic bits
    unsigned char flags = (diag & 0xf000) >> 12;
    unsigned char counter = (diag & 0x003f);
    
    // set bit 4 in flags if count field isn't last+1
    int tmpcnt;
    if ((tmpcnt = lastcnt) >= 0 && (++tmpcnt % 64) != counter)
    	flags |= 0x10;
    lastcnt = counter;

    short sval[4];
    for (int i = 0; i < 4; i++)
	sval[i] = conv->shortValue(sample->data+(i*sizeof(short)));

    float uvwt[4];
    int iscale;
    for (int i = 0; i < 3; i++) {
	uvwt[i] = sval[i] * scale[iscale = range[i]];
	if (iscale == 0) {
	    switch (sval[i]) {
	    case -32768:
	    case 0:
		uvwt[i] = FLOAT_NAN;
		break;
	    default:
		break;
	    }
	}
    }

    // speed of sound
    float ss = (sval[3] * 0.001) + 340.;
    if (sval[3] == -32768) ss = FLOAT_NAN;
    ss /= 20.067;
    uvwt[3] = (ss * ss) - 273.15;

    ost << fixed << setprecision(2) << setw(9) << setfill(' ') <<
    	sample->tt / 1000.0;		// seconds since GMT midnight

    for (int i = 0; i < 4; i++) {
	if (isnan(uvwt[i])) ost << setw(8) << setfill(' ' ) << "NaN ";
        else ost <<  setprecision(3) << setw(8) << setfill(' ') <<
		uvwt[i] << ' ';
    }
    ost << setfill('0') << hex << setw(2) << (unsigned int)flags << dec << endl;
    return true;
}
