//
//              Copyright 2004 (C) by UCAR
//

#include <atdISFF/BinarySensorPort.h>
#include <atdISFF/RawSample.h>

#include <iomanip>
#include <math.h>

using namespace atdISFF;
using namespace std;

BinarySensorPort::BinarySensorPort(const std::string portName,int channel)
    throw (atdUtil::IOException) :
    	SensorPort(portName,channel),
	syncstr(0),lsync(0),nsync(0) {
  setMessageLength(10);		// default
}

BinarySensorPort::~BinarySensorPort() {
  delete [] syncstr;
}

void BinarySensorPort::setMessageLength(int val) {
  messageLength = val;
  setRawLength(val + lsync);
}

void BinarySensorPort::init() throw(atdUtil::IOException) {
  SensorPort::init();
  if (getSeparatorAtEOM()) currentSampleLen = messageLength;
  else currentSampleLen = 0;
}

void BinarySensorPort::setMessageSeparator(const std::string& val) throw(atdUtil::IOException) {
  SensorPort::setMessageSeparator(val);
  delete [] syncstr;
  lsync = val.size();
  syncstr = new char[lsync + 1];
  strcpy(syncstr,val.c_str());
  nsync = 0;
  setRawLength(getMessageLength() + lsync);
}


void BinarySensorPort::processInput(isff_sys_time_t rtime,const char *charP)
	throw(atdUtil::IOException)
{
  int charOffset = 0;
  if (getSeparatorAtEOM()) {
    scanInputSepAtEOM(charP);
  }
  else {
    scanInputSepAtBOM(charP);
    charOffset = lsync;
  }

  /* rtime is the receipt time of the last character in the buffer.
   */

  size_t nsamp = samplePointers.size();

  /* Compute time of first byte of last sample, which
   * may be a partial sample. Assume characters
   * are evenly spaced at msecsPerChar apart.
   * If the sync bytes are at the beginning of the record,
   * charOffset is the length of the sync bytes.
   */

  /* scanPtr is point to beginning of last, possibly
   * partial sample in buffer.
   */

  bool lastIsPartial = currentSampleLen > 0;
  rtime = isff_sys_time_add(rtime,
      	-(eodP - scanPtr + charOffset) * msecsPerChar);

  // How many sample deltaTs between rtime and first sample
  // that we will distribute.
  int nsampDt = nsamp;
  if (!lastIsPartial) nsampDt--;	// last is complete sample

  deltatms = 0.0;
  if (nsampDt > 0) {
    // since we have to roll rtime back over one or more samples
    // we need to know the sample deltat.  We compute that
    // from the last sample time in the previous buffer.
    // compute how many deltaT's back to that sample.
    if (lastSampleTime == 0) {
      for (int i = 0; i < nsampDt; i++)
        samplePointers.erase(samplePointers.begin());
      lastSampleTime = rtime;
    }
    else {
      int ndt = nsamp;
      if (lastIsPartial) ndt++;
      if (partialLastSample) ndt--;
      deltatms = (rtime - lastSampleTime) / ndt;

      /* deltatms can be negative if the system clock
       * is reset backwards - by user or ntp.
       */
      if (deltatms < 0.0) deltatms = 0.0;

      // if ((int)deltatms < 14 || (int)deltatms > 17) 
      	// cerr << "deltatms=" << deltatms << endl;

      lastSampleTime = rtime;

      /* Compute time of initial sample */
      rtime = isff_sys_time_add(rtime,-(nsampDt * deltatms));
    }
  }
  else lastSampleTime = rtime;

  partialLastSample = lastIsPartial;

  int is = 0;
  for (list<const char*>::const_iterator sitr = samplePointers.begin();
      sitr != samplePointers.end(); sitr++) {
    RawSample *sample = 
	    RawSamplePool::getInstance()->getRawSample(messageLength);

    sample->tt = sample_time(rtime,(is++ * deltatms));
    sample->chan = chan;
    sample->len = messageLength;
    ::memcpy(sample->data,*sitr,messageLength);
    distribute(sample);
    nsamples++;
  }
  shiftBuffer();
}

void BinarySensorPort::scanInputSepAtBOM(const char *charP)
{
  register const char *cp;

#ifdef DEBUG
  std::cerr << " buffer=" << std::hex << std::setw(8) << (int)buffer << 
      " charP=" << (int)charP << 
      " currentSamplep=" << std::setw(8) << (int)currentSamplep << 
      " eodP-buffer=" << std::dec << eodP - buffer << 
      " nsync=" << nsync << " lsync=" << lsync <<
      " messageLength=" << messageLength << " currentSampleLen=" << currentSampleLen <<
      std::dec << std::endl;
#endif

  for(cp = charP; cp < eodP; cp++) {
#ifdef DEBUG
    std::cerr << "c=" << std::hex << std::setw(2) << ((unsigned int)*cp & 0xff)<< std::dec <<
      	" nsync=" << nsync << " lsync=" << lsync <<
      	" messageLength=" << messageLength << " currentSampleLen=" << currentSampleLen <<
	" syncstr[nsync]=" << std::setfill('0') << std::hex << std::setw(2) << ((unsigned int)syncstr[nsync] & 0xff) << std::dec <<
	std::endl;
#endif
    if (nsync < lsync) {
      char c = *cp;
      // start over if no match
      if (c == syncstr[nsync] || c == syncstr[nsync=0]) nsync++;
    }
    else {
      if (currentSampleLen++ == 0) scanPtr = cp;
      if (currentSampleLen == messageLength) {
        samplePointers.push_back(scanPtr);
	nsync = 0;
	currentSampleLen = 0;
      }
    }
  }
}

void BinarySensorPort::scanInputSepAtEOM(const char *charP)
{
  register const char *cp;

#ifdef DEBUG
  std::cerr << " buffer=" << std::hex << std::setw(8) << (int)buffer << 
      " charP=" << (int)charP << 
      " scanPtr=" << std::setw(8) << (int)scanPtr << 
      " eodP-buffer=" << std::dec << eodP - buffer << 
      " nsync=" << nsync << " lsync=" << lsync <<
      " messageLength=" << messageLength << " currentSampleLen=" << currentSampleLen <<
      std::dec << std::endl;
#endif

  for(cp = charP; cp < eodP; cp++) {
#ifdef DEBUG
    std::cerr << "c=" << std::hex << std::setw(2) << ((unsigned int)*cp & 0xff)<< std::dec <<
      	" nsync=" << nsync << " lsync=" << lsync <<
      	" messageLength=" << (int)messageLength << " currentSampleLen=" << (int)currentSampleLen <<
	" syncstr[nsync]=" << std::setfill('0') << std::hex << std::setw(2) << ((unsigned int)syncstr[nsync] & 0xff) << std::dec <<
	std::endl;
#endif
    // initial conditions: currentSampleLen=messageLength, nsync=0
    if (currentSampleLen < messageLength) {
      if (currentSampleLen++ == 0) scanPtr = cp;
    }
    else {
      char c = *cp;
      // start over (nsync=0) if no match
      if ((c == syncstr[nsync] || c == syncstr[nsync=0]) && ++nsync == lsync) {
	if (scanPtr)
	  samplePointers.push_back(scanPtr);
	nsync = 0;
	currentSampleLen = 0;
      }
    }
  }
}
