//
//              Copyright 2004 (C) by UCAR
//


#ifndef ATDISFF_A2DSAMPLER_H
#define ATDISFF_A2DSAMPLER_H

#include <vector>
#include <string>

#include <atdUtil/Thread.h>
#include <atdUtil/InvalidParameterException.h>
#include <atdUtil/IOException.h>
#include <atdISFF/RawSampleBuffer.h>
#include <atdISFF/Time.h>

namespace atdISFF {

class A2DSampler : public atdUtil::Thread, public RawSampleSource {
public:

  A2DSampler(const std::string& name);

  virtual ~A2DSampler() {}

  virtual void open() throw(atdUtil::IOException) = 0;

  virtual void close() throw(atdUtil::IOException) = 0;

  /**
   * How many A2D channels on this board?
   */
  virtual int getMaxNumberOfChannels() const { return maxNumberOfChannels; }

  /**
   * Request a channel of this A2D.
   */
  virtual void addChannel(int userChannelNum, int vRangeIndex)
  	throw(atdUtil::IOException,atdUtil::InvalidParameterException) = 0;

  /**
   * initialize function that is called after adding
   * channels, and before start();
   */
  virtual void init() throw(atdUtil::IOException,atdUtil::InvalidParameterException) = 0;

  /**
   * What voltage ranges does this A2D support.
   * Each element in the returned vector is a const pointer to a
   * float array. The array contains the minimum voltage supported
   * by the A2D, the maximum voltage and the minimum voltage resolution.
   * Usage:
   *	const std::vector& ranges = a2d.getAvailableVoltageRanges();
   *	for (int i = 0; i < ranges.size(); i++) {
   *	  float vmin = ranges[i][0];
   *	  float vmax = ranges[i][1];
   *	  float vresolution = ranges[i][2];
   *	  ...
   *    }
   *	
   */
  virtual const std::vector<const float*>& getAvailableVoltageRanges() const = 0;

  /**
   * Which channels are active, meaning they have been requested with
   * addChannel.
   */
  virtual const std::vector<int>& getUserChannelNumbers() const = 0;

  /**
   * What is the voltage range for a channel, by its channel index.
   * The range returned is an array of floats, as with
   * getAvailableVoltageRanges().
   * The channel index is not the same as the channel number.
   * Channel indices are numbered from 0, in the order they
   * were added.  Channel numbers are numbered from 100.
   */
  virtual const float* getVoltageRange(int channelIndex) const = 0;

  /**
   * What is the sample rate for the channel.
   */
  virtual float getSamplingRate(int channelIndex) const = 0;

  /**
   * Provide the slope of the conversion from counts to voltage.
   * Volts = counts * slope + intercept
   */
  virtual float getConversionSlope(int channelIndex) const = 0;

  /**
   * Provide the offset of the conversion from counts to voltage.
   * Volts = counts * slope + intercept
   */
  virtual float getConversionIntercept(int channelIndex) const = 0;

  /**
   * Set the statistics period, in seconds. The A2D maintains
   * counts and other status information which can be queried.
   * Every statistics period these counters are reset to 0.
   */
  void setStatisticsPeriodInSecs(int val) { statisticsPeriod = val * 1000; }

  /**
   * Get the statistics period, in seconds.
   */
  int getStatisticsPeriodInSecs() const { return statisticsPeriod / 1000; }

  /**
   * How may missed samples during the previous statistics period.
   */
  virtual size_t getMissedSamples() const = 0;

  /**
   * How may missed samples since bootup?
   */
  virtual size_t getCumulativeMissedSamples() const = 0;

  /**
   * What is the total (all channels) observed sample rate.
   */
  virtual float getTotalObservedSamplingRate() const { return totalObservedSamplingRate; }

  /**
   * How frequently are interrupts coming from the A2D device?
   */
  virtual float getUserInterruptRate() const { return userInterruptRate; }

protected:

  /**
   * Provided for derived classes to set the observed sample rate.
   */
  virtual void setTotalObservedSamplingRate(float val) {
    totalObservedSamplingRate = val;
  }

  /**
   * Provided for derived classes to set the user interrupt rate.
   */
  virtual void setUserInterruptRate(float val) {
    userInterruptRate = val;
  }

  /**
   * Reset statistics counters, sums, etc.
   * A2DSamplers keep track of some useful status variables,
   * like the actual observed sample rate, how
   * many times the user side of the A2D is interrrupted, 
   * number of missed samples, etc.  These can be queried
   * by external status monitors.  Periodically,
   * based on statisticsPeriod, the statistics are reset.
   */
  virtual void resetStatistics(isff_sys_time_t tnow) = 0;

  /**
   * Next time to recompute/reset the statistics.
   */
  isff_sys_time_t statisticsTime;

  /**
   * length of statistics period, in milliseconds.
   */
  unsigned long statisticsPeriod;

  /**
   * Number of available channels on this A2D board.
   */
  int maxNumberOfChannels;

  /**
   * Number of samples distributed.
   */
  size_t nsamples;

  /**
   * Observed number of samples per second.
   */
  float totalObservedSamplingRate;

  /**
   * Number of interrupts from A2D to user space processes, per second.
   */
  float userInterruptRate;

};

}

#endif
