//
//              Copyright 2004 (C) by UCAR
//

/*
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <signal.h>
*/

#include <atdISFF/relays/RrelayListener.h>
#include <atdUtil/UnknownHostException.h>

#include <string>

#include <unistd.h>
#include <termios.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <poll.h>

using namespace std;
// using namespace atdISFF;
using namespace atdUtil;

class Rrelay {

public:

  Rrelay();
  ~Rrelay();

  static void signalCatcher(int s);
  static Rrelay* instance;
  static Rrelay* getInstance();
  static int usage(const char* argv0);
  static int main(int argc, char **argv);

  void openSocket(const string& hostName) throw(IOException,UnknownHostException);
  void run() throw(IOException);

private:

  string hostName;

  void setupStdin() throw(IOException);
  void restoreStdin() throw(IOException);

  int sockfd;;

  bool outahere;
  struct termios term_io_old;

};

Rrelay::Rrelay(): sockfd(-1),outahere(false)
{
  struct sigaction action;
  action.sa_handler = &signalCatcher;
  memset(&action.sa_mask,0,sizeof action.sa_mask);
  memset(&action.sa_flags,0,sizeof action.sa_flags);

  sigaction(SIGINT,&action,0);
  sigaction(SIGTERM,&action,0);
  sigaction(SIGHUP,&action,0);

}

Rrelay::~Rrelay()
{
  if (sockfd >= 0) ::close(sockfd);
}

/* static */
Rrelay* Rrelay::instance = 0;

/* static */
Rrelay* Rrelay::getInstance() 
{
  if (!instance) instance = new Rrelay();
  return instance;
}

/***************************************************************/
/*
 * catch SIGINT
 * 
 */
/* static */

void Rrelay::signalCatcher(int s)
{
  const char *signame;
  switch (s) {
  case SIGHUP:
    signame = "hangup";
    break;
  case SIGINT:
    signame = "interrupt";
    break;
  case SIGTERM:
    signame = "terminate";
    break;
  default:
    signame = "";
    break;
  }
  cerr << "received " << signame << " signal (#" << s << ")\n" << endl;
  instance->outahere = 1;
}

void Rrelay::openSocket(const string& hostName)
	throw(IOException,UnknownHostException)
{
  struct sockaddr_in serverAddr;	/* server's address */
  struct hostent *host_ent;

  /*
   * Zero out the sock_addr structures.
   */

  memset(&serverAddr, 0, sizeof(serverAddr));

  /*
   * Open the socket. Use ARPA Internet address format and stream sockets.
   * Format described in "socket.h".
   */

  sockfd = socket(AF_INET, SOCK_STREAM, 0);

  if (sockfd < 0)
    throw IOException("rrelay socket","open",errno);

  serverAddr.sin_family = AF_INET;
  // serverAddr.sin_port = htons(RrelayListener::RRELAY_PORT);
  serverAddr.sin_port = htons(RRELAY_PORT);

  /* get server's Internet address */

  if ((host_ent = gethostbyname(hostName.c_str())) == 0)
    throw UnknownHostException(hostName);
  
  memcpy((char *) &serverAddr.sin_addr.s_addr,
      (char *) (host_ent->h_addr),
      host_ent->h_length);

  if (connect(sockfd, (struct sockaddr *) & serverAddr, sizeof(serverAddr)) < 0)
    throw IOException("rrelay socket","connect",errno);
}
void Rrelay::setupStdin() throw(IOException) {
  struct termios term_io_new;
  /* get the termio characterstics for stdin */
  if (isatty(0)) {
    if (tcgetattr(0,&term_io_old) < 0)
      throw IOException("stdin","tcgetattr(0,)",errno);

    /* copy and turn off local echo */
    memcpy( &term_io_new, &term_io_old, sizeof term_io_old);
    term_io_new.c_lflag &= ~ECHO & ~ICANON;

    term_io_new.c_iflag &= ~ICRNL & ~INLCR; 

    // term_io_new.c_oflag &= ~OPOST;
    term_io_new.c_oflag |= OPOST | ONLCR;

/* termio man page:
     Case A: MIN > 0, TIME > 0
          In this case, TIME serves as  an  intercharacter  timer
          and is activated after the first character is received.
          Since it is an intercharacter timer, it is reset  after
          a  character  is received.  The interaction between MIN
          and TIME is as follows:  as soon as  one  character  is
          received,  the intercharacter timer is started.  If MIN
          characters are received before the intercharacter timer
          expires  (note  that the timer is reset upon receipt of
          each character), the read is satisfied.  If  the  timer
          expires before MIN characters are received, the charac-
          ters received to that point are returned to  the  user.
          Note  that if TIME expires, at least one character will
          be returned because  the  timer  would  not  have  been
          enabled  unless a character was received.  In this case
          (MIN > 0, TIME > 0), the read sleeps until the MIN  and
          TIME  mechanisms  are  activated  by the receipt of the
          first character.  If the number of characters  read  is
          less than the number of characters available, the timer
          is not reactivated and the subsequent read is satisfied
          immediately.
*/
    term_io_new.c_cc[VMIN] = 1;
    term_io_new.c_cc[VTIME] = 1;		// 1/10 second

    if (tcsetattr(0,TCSAFLUSH,&term_io_new) < 0)
      throw IOException("stdin","tcsetattr(0,TCSAFLUSH...)",errno);
  }
}
void Rrelay::restoreStdin() throw(IOException) {
  /* reset terminal to earlier state */
  if (isatty(0) && tcsetattr(0,TCSAFLUSH,&term_io_old) < 0)
      throw IOException("stdin","tcsetattr(0,)",errno);
}


/* static */
int Rrelay::usage(const char* argv0) {
  cerr << "Usage: " << argv0 << " data_system_host_name" << endl;
  return 1;
}

void Rrelay::run() throw(IOException) {

  struct pollfd pollfds[2];
  pollfds[0].fd = 0;		// stdin
  pollfds[0].events = POLLIN | POLLHUP | POLLERR;
  pollfds[1].fd = sockfd;	// receive socket
  pollfds[1].events = POLLIN | POLLHUP | POLLERR;

  // int POLLING_TIMEOUT = 300000; 	// milliseconds
  char buffer[1000];
  int pollres,nwrite,nread;

  // the polling loop
  while (!outahere) {
    switch (pollres = poll(pollfds,2,300000)) {
    case -1:
      if (errno != EINTR) throw IOException("rrelay","poll",errno);
      outahere = 1;
      break;
    case 0:	// polling timeout
      cerr << "Timeout waiting for input" << endl;
      outahere = 1;
      break;
    default:
      if (pollfds[0].revents & POLLIN) {	// data on stdin
	/* read a character from standard input */
	if ((nread = read(0, buffer, sizeof buffer)) <= 0) {
	  if (nread < 0) throw IOException("stdin","read",errno);
	    /* client read an EOF; exit the loop. */
	  outahere = 1;
	  break;
	}
	// ctrl-D = \004
	if (nread == 1 && buffer[0] == '\004') {
	  outahere = 1;
	  break;
	}

	/* send bytes to socket */
	for (char *cp = buffer; cp < buffer + nread; ) {
	  int n = buffer + nread - cp;
	  // cerr << "sending " << n << " bytes on socket" << endl;
	  if ((nwrite = send(sockfd, cp, n, 0)) < 0)
	    throw IOException("socket","send",errno);
	  cp += nwrite;
	}
      }
      if (pollfds[0].revents & POLLERR) {
	cerr << "poll() POLLERR on stdin" << endl;
	outahere = 1;
      }
      if (pollfds[0].revents & POLLHUP) {
	cerr << "poll() POLLHUP on stdin" << endl;
	outahere = 1;
      }
      if (pollfds[1].revents & POLLIN) {
	if ((nread = recv(sockfd,(char *)buffer, sizeof buffer, 0)) <= 0) {
	  /* error or client has disappeared */
	  if (nread < 0) throw IOException("rrelay socket","recv",errno);
	  outahere = 1;
	  break;
	}
	/* send bytes to stdout */
	for (char *cp = buffer; cp < buffer + nread; ) {
	  int n = buffer + nread - cp;
	  // cerr << "sending " << n << " bytes to stdout" << endl;
	  if ((nwrite = write(1, cp, n)) < 0)
	    throw IOException("stdin","write",errno);
	  cp += nwrite;
	}
      }
      if (pollfds[1].revents & POLLERR) {
	cerr << "poll() POLLERR on input socket" << endl;
	outahere = 1;
      }
      if (pollfds[1].revents & POLLHUP) {
	cerr << "poll() POLLHUP on input socket" << endl;
	outahere = 1;
      }
    }
  }

  restoreStdin();
}

/* static */
int Rrelay::main(int argc, char **argv)
{
  try {
    Rrelay rrelay;

    if (argc < 2) return rrelay.usage(argv[0]);

    rrelay.openSocket(argv[1]);
    rrelay.setupStdin();
    rrelay.run();
  }
  catch (Exception& e) {
    cerr << e.what() << endl;
  }
  return 0;
}


int main(int argc, char **argv) {
  return Rrelay::main(argc,argv);
}
