# Program reads in standard and flight specific input, estimates flight times and locations, and outputs graphics, kml, and csv

#### fix so longitude goes -180 to 180 instead of -360 to 0

fpnn = function(flightname='EXAMPLE_FLIGHT', 
                netdir='/net/www/docs/homes/stephens/ORCAS/FLTPLAN',
                infile='config_files/fltplan_input.txt',
                wdir=315,
                tstep=1/3,
                svstep=10,
                hdstep=60,
                gtdistco=0.5){

# flightin has resolution of flight specific input file
# flight has resolution of svstep seconds
# segment has resolution of tstep seconds
# headings adjusted at hdstep seconds

# wdir = wind direction, for uw, dw, cw, and revcw calls (time and course enroute assumes no wind)
# tstep = time step of calculation, 1/3 sec corresponds to 1 degree in 2 min turns (when turn = 1 or -1)
# svstep = save step, time interval at which to save position data
# hdstep = heading step, time interval at which to recalculate heading
# gtdistco = distance criterion in nmi to determine if a specified location is reached

# to do:

##### flight time of 1:60 in ORCAS_TF01
##### for ar and for extended time, collect added rows for flightin and stick in before writing out

### ability to specify a zoom region to plot
### ability to specify a turn of a certain radius or to precisely hit a certain alt change?
###    e.g. 270 turn over 20000 ft. (turn would have to be other than 1)
### ability to specify e.g. 90 degree right turn? (and a 90/270)
### allow specloc to be in deg/min
### ability to read in set sequences (e.g. upstream fence, RS grid)
### warn when ar, gt, or approach does not have enough time or distance to reach alt2
### warn if alt1 and alt2 differ, and min < (alt2-alt1)/rate
### warn if heading change required, and min < (delta-heading)/360*2
### warn if alt1=alt2, dir=NA, and min=NA (or would such a row just be skipped)
### warn if 2 waypoints have the same name
### warn if waypoint name is 'Y' bc Waypoint name left blank
### warn if nothing in zoom area
### add diagrams to instructions
### waypoint locs/names to KML and varied track colors
### control view settings on KML
### add ability to add points not on flight track, eg. ASPAs
### write sec not min to flight
### spiraling from 43000 to 500 followed by gt so same pt crashes, but from 30000 to 500 does not (bc inbound to point)
###  from 43500 also does not work

### other to dos as ### throughout

# someday:
### use winds to modify ground speeds
### improve fuel burn calculation (up vs. down, heavy vs. light, non-linear with height) - get output from pilots code, also report in flight?
### group flight plans by type?

library(maps)
library(mapdata)
library(mapproj)

# define wind-relative headings
cwhdg=wdir+180+90
if(cwhdg>360) cwhdg=cwhdg-360
revcwhdg=cwhdg+180
if(revcwhdg>360) revcwhdg=revcwhdg-360
uwhdg=wdir
dwhdg=wdir+180
if(dwhdg>360) dwhdg=dwhdg-360

# read in standard input
fpinput=read.table(infile,col.names=c('val','var'),flush=T,stringsAsFactors=F,comment.char='#')
for(i in 1:nrow(fpinput)){ assign(fpinput$var[i],fpinput$val[i]) }
# "maxalt"   "pblalt"   "maxgs"    "pblgs"    "fullfuel" "maxfb"    "pblfb"    "rate"
# All other ground speeds and fuel burns linearly interpolated by altitude

# read in pre-defined waypoints
specloc=read.table('config_files/specloc.txt',header=T,comment.char='#')

# read in flight specific input
flightin=read.table(paste(flightname,'.txt',sep=''),skip=1,stringsAsFactors=F,fill=T,comment.char='#')
if(ncol(flightin)==4){
	names(flightin)=c('alt2','min','dir','phase')
}else if(ncol(flightin)==5){
	names(flightin)=c('alt2','min','dir','phase','wpname')
}else if(ncol(flightin)==6){
	names(flightin)=c('alt2','min','dir','phase','wpname','sel')
}

# insert any RS legs
flightin2=flightin
rsrows=0
rsinfiles=NULL
for(i in c(1:nrow(flightin))){
if(!is.na(flightin$dir[i])){
if(substr(flightin$dir[i],1,2)=='rs'){
	rsinfile=paste(substr(flightin$dir[i],4,nchar(flightin$dir[i])),'.utm_flt_line.in.py',sep='')
	rsoutfile=paste(substr(flightin$dir[i],4,nchar(flightin$dir[i])),'.utm_flt_line.out.py',sep='')
	system(paste('python utm_flt_lines.py',rsinfile,rsoutfile))
	rslegs=read.table(rsoutfile,stringsAsFactors=F,fill=T,comment.char='#')
	names(rslegs)=c('alt2','min','dir','phase','wpname','sel')
	flightin2=rbind(flightin2[1:(i-1+rsrows),],rslegs,flightin2[(i+1+rsrows):nrow(flightin2),])
	rsrows=rsrows+nrow(rslegs)
	rsinfiles=c(rsinfiles,rsinfile)
}
}
}
flightin=flightin2

## specify alt, min, dir, phase, wpname, and whether a named waypoint should be output for pilots
## 'dir' is a 'directive', not a heading direction
## 1) if alt new, then min must be NA or > alt/rate
## 2) if heading new, then min must be NA or > change in deg / 180
## 3) if heading new, turn starts immediately
## 4) if alt new and dir is an 'sc' then both turn and climb/descent start immediately
## 5) if alt new and dir is an 'ar' then turn starts immediately but climb/descent starts when needed to end at waypoint
## 6) if min = NA, then duration calculated from time to climb/descend or time to turn (whichever is longer)
## 7) if phase = NA, then phase set to be same as previous row
## 8) min must be specified if alt1=alt2 and dir is NA
## 9) dir = one of:
## 10) file must start with a 'sc' or a fixed heading (specified as xxx in dir)
### 'lh_xxx' = left hand turn of specified degrees (xxx = 3 digit degrees)
### 'rh_xxx' = right hand turn of specified degrees (xxx = 3 digit degrees)
### 'lh2_xxx' = left hand turn to new heading (xxx = 3 digit degrees or cw, revcw, uw, or dw)
### 'rh2_xxx' = right hand turn to new heading (xxx = 3 digit degrees or cw, revcw, uw, or dw)
### 't2_xxx' = shortest direction turn to new heading (xxx = 3 digit degrees or cw, revcw, uw, or dw)
### 'lhspiral' = left hand circle (while climbing or descending dep on alt)
### 'rhspiral' = right hand circle (while climbing or descending dep on alt)
### 'scDDMMSDDDMMW' = set course (great circle) towards waypoint (start any climb/descent immediately)
### 'arDDMMSDDDMMW' = arrive (same as sc but hold alt until time to ascend/descend to arrive at waypoint)
### 'gtDDMMSDDDMMW' = go to (same as sc but continues until destination reached)
### 'sc_****' = same as sc, but using a location name with lat/lon specified in specloc.txt
### 'ar_****' = same as ar, but using a location name with lat/lon specified in specloc.txt
### 'gt_****' = same as gt, but using a location name with lat/lon specified in specloc.txt
### 'inbound' = same as sc, but location set to initial waypoint
### 'approach' = same as ar but location set to initial waypoint
### 'inbound' and 'set course' directives propagated until next non-NA directive, so only need to be set once and great circle heading adjustments will continue, all other NA dir left as is
## 11) on first row, sc does not add any time, but on other rows, adds in a standard (shortest direction) turn
#14) the Waypoint and Selected columns can be left blank, but if Waypoint is blank Selected must be too.  All other columns need either a value or NA.
#15) the Waypoint name on the first row must reference a specified location in specloc.txt
#16) to change altitude and then fly level for specified time, need to set min to total of time for alt change and level time
#17) to change heading and then fly straight for specified time, need to set min to total of time for hdg change and straight time
flightin$dir[flightin$dir=='lh2_cw']=paste('lh2_',round(cwhdg,0),sep='')
flightin$dir[flightin$dir=='rh2_cw']=paste('rh2_',round(cwhdg,0),sep='')
flightin$dir[flightin$dir=='t2_cw']=paste('t2_',round(cwhdg,0),sep='')
flightin$dir[flightin$dir=='lh2_revcw']=paste('lh2_',round(revcwhdg,0),sep='')
flightin$dir[flightin$dir=='rh2_revcw']=paste('rh2_',round(revcwhdg,0),sep='')
flightin$dir[flightin$dir=='t2_revcw']=paste('t2_',round(revcwhdg,0),sep='')
flightin$dir[flightin$dir=='lh2_uw']=paste('lh2_',round(uwhdg,0),sep='')
flightin$dir[flightin$dir=='rh2_uw']=paste('rh2_',round(uwhdg,0),sep='')
flightin$dir[flightin$dir=='t2_uw']=paste('t2_',round(uwhdg,0),sep='')
flightin$dir[flightin$dir=='lh2_dw']=paste('lh2_',round(dwhdg,0),sep='')
flightin$dir[flightin$dir=='rh2_dw']=paste('rh2_',round(dwhdg,0),sep='')
flightin$dir[flightin$dir=='t2_dw']=paste('t2_',round(dwhdg,0),sep='')
for(i in 2:nrow(flightin)){
	if(is.na(flightin$phase[i])){ flightin$phase[i]=flightin$phase[i-1] }
}
# change phase names to numbers
phasenames=flightin$phase
allphase=unique(flightin$phase)
flightin$phase=match(flightin$phase,allphase)
flightin$wpnum=c(1:nrow(flightin))

# propagate 'inbound' and 'set course' directives until next non-NA directive
for(i in 2:nrow(flightin)){
	if(!is.na(flightin$dir[i-1])){
		if(is.na(flightin$dir[i])&(flightin$dir[i-1]=='inbound'|substr(flightin$dir[i-1],1,2)=='sc')){
			flightin$dir[i]=flightin$dir[i-1]
		}
	}
}

# set initial location
if(any(specloc$name==flightin$wpname[1])){
initlat=specloc$lat[specloc$name==flightin$wpname[1]]
initlon=specloc$lon[specloc$name==flightin$wpname[1]]
initalt=specloc$alt[specloc$name==flightin$wpname[1]]
} else {
print(paste('NAMED WAYPOINT',flightin$wpname[1],'NOT FOUND IN specloc.txt'))
initlat=0
initlon=0
initalt=0
}

# set initial heading
flight=c(flightin[1,c(1:2,4)],rep(NA,7)) # 'flight' will be similar to 'flightin' but have extra columns and svstep resolution (nominally 10 sec)
names(flight)=c(names(flightin[c(1:2,4)]),"gs","dist","fb","fuel","lat","lon","hdg") # have to leave 'dir' out to use cbind below without forcing numbers to characters
flight=data.frame(flight)
# "alt2"  "min"   "phase" "gs"    "dist"  "fb"    "fuel"  "lat"  "lon"   "hdg"
if(substr(flightin$dir[1],1,2)=='sc'){ # must start with "set course" or a fixed heading
	lat1=initlat
	lon1=initlon
        lat=lat1
        lon=lon1
	if(substr(flightin$dir[1],3,3)=='_'){ # specified location
		locname=substr(flightin$dir[1],4,nchar(flightin$dir[1]))
	if(any(specloc$name==locname)){
		lat2=specloc$lat[specloc$name==locname]
		lon2=specloc$lon[specloc$name==locname]
		} else {
			print(paste('NAMED WAYPOINT',locname,'NOT FOUND IN specloc.txt'))
		}
	} else {
		lat2=as.numeric(substr(flightin$dir[1],3,4))+as.numeric(substr(flightin$dir[1],5,6))/60
		if(substr(flightin$dir[1],7,7)=='S') lat2=lat2*-1
		lon2=as.numeric(substr(flightin$dir[1],8,10))+as.numeric(substr(flightin$dir[1],11,12))/60
		if(substr(flightin$dir[1],13,13)=='W') lon2=lon2*-1
	}
	flight$hdg[1]=round((atan2(sin((lon2-lon1)*pi/180)*cos(lat2*pi/180),cos(lat1*pi/180)*sin(lat2*pi/180)-sin(lat1*pi/180)*cos(lat2*pi/180)*cos((lon2-lon1)*pi/180))/pi*180+360) %% 360) # this is spherical geometry
} else {
	flight$hdg[1]=as.numeric(flightin$dir[1])
}
flight$lat[1]=initlat
flight$lon[1]=initlon
flight$alt2[1]=initalt
flight$gs[1]=0
flight$dist[1]=0
flight$fb[1]=0
flight$fuel[1]=0

# add columns to fill in flightin
flightin$endlat=rep(NA,nrow(flightin))
flightin$endlon=rep(NA,nrow(flightin))
flightin$endalt=rep(NA,nrow(flightin))
flightin$enddist=rep(NA,nrow(flightin))
flightin$endfuel=rep(NA,nrow(flightin))
flightin$endhdg=rep(NA,nrow(flightin))
flightin$endlat[1]=initlat
flightin$endlon[1]=initlon
flightin$endalt[1]=initalt
flightin$enddist[1]=0
flightin$endfuel[1]=0
flightin$endhdg[1]=flight$hdg[1]

# step through flight plan, estimating time to climb and turn as needed
print(flightin[1,])
for(i in c(2:nrow(flightin))){
	segmin=0
	segdist=0
	segfuel=0
	updown=0 # -1, 0, or 1
	turn=0 # -1, 0, or 1
	hdg1=tail(flight$hdg,1)
	lat1=tail(flight$lat,1)
	lon1=tail(flight$lon,1)
	alt1=tail(flight$alt2,1)
	alt2=flightin$alt2[i]
	# check if altitude change needed
	if(flightin$alt2[i]>flightin$alt2[i-1]){ updown=1 } else if(flightin$alt2[i]<flightin$alt2[i-1]) { updown=-1 }
#print(c(i,flightin$alt2[i],flightin$alt2[i-1],alt1,alt2,updown))
	# check if turn needed
	if(!is.na(flightin$dir[i])){ # need to turn (and possibly determine direction and or heading)
		if(substr(flightin$dir[i],1,2)=='lh'){
			turn=-1
			if(substr(flightin$dir[i],3,8)=='spiral'){
				hdg2=-1000
			} else {
				if(substr(flightin$dir[i],3,4)=='2_'){
					hdg2=as.numeric(substr(flightin$dir[i],5,7))
				} else if(substr(flightin$dir[i],3,3)=='_'){
					hdg2=hdg1-as.numeric(substr(flightin$dir[i],4,6))
				}
				hdg2=hdg2%%360
			}
		} else if(substr(flightin$dir[i],1,2)=='rh'){
			turn=1
			if(substr(flightin$dir[i],3,8)=='spiral'){
				hdg2=1000
			} else {
				if(substr(flightin$dir[i],3,4)=='2_'){
					hdg2=as.numeric(substr(flightin$dir[i],5,7))
				} else if(substr(flightin$dir[i],3,3)=='_'){
					hdg2=hdg1+as.numeric(substr(flightin$dir[i],4,6))
				}
				hdg2=hdg2%%360
			}
		} else if(substr(flightin$dir[i],1,2)=='t2'){ # turn in shortest direction
			hdg2=as.numeric(substr(flightin$dir[i],4,6))
			hdg2=hdg2%%360
			#> dir=matrix(0,360,360)
			#> for(i in 1:360){for(j in 1:360){if(i-j>=180){dir[i,j]=1}else if(i-j<(-180)){dir[i,j]=-1} else if(i>j){dir[i,j]=-1}else if(i<j){dir[i,j]=1}}}
			#> image(1:360,1:360,dir,zlim=c(-1.5,1.5),col=rainbow(3))
			if(hdg1-hdg2>=180){ turn=1 } else if(hdg1-hdg2<(-180)){ turn=-1 } else if(hdg1>hdg2){ turn=-1 }else if(hdg1<hdg2){ turn=1 } # 1 = turn right, -1 = turn left
#			if(hdg2>hdg1&hdg2-hdg1<=180|hdg2<hdg1&hdg1-hdg2>180){ turn=1 # turn right
#                        } else if(hdg2>hdg1&hdg2-hdg1>180|hdg2<hdg1&hdg1-hdg2<=180){ turn=-1 } # turn left
		} else if(substr(flightin$dir[i],1,2)=='sc'|substr(flightin$dir[i],1,2)=='gt'|substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='inbound'|flightin$dir[i]=='approach'){ # turn in shortest direction to required heading
			# assumes bearing to destination does not change during turn, but then updates heading every minute to maintain great circle course
			if(flightin$dir[i]=='inbound'|flightin$dir[i]=='approach'){
				lat2=initlat
				lon2=initlon
			} else {
				if(substr(flightin$dir[i],3,3)=='_'){ # specified location
					locname=substr(flightin$dir[i],4,nchar(flightin$dir[i]))
					if(any(specloc$name==locname)){
						lat2=specloc$lat[specloc$name==locname]
						lon2=specloc$lon[specloc$name==locname]
					} else {
						print(paste('NAMED WAYPOINT',locname,'NOT FOUND IN specloc.txt'))
					}
				} else {
			        	lat2=as.numeric(substr(flightin$dir[i],3,4))+as.numeric(substr(flightin$dir[i],5,6))/60
				        if(substr(flightin$dir[i],7,7)=='S') lat2=lat2*-1
				        lon2=as.numeric(substr(flightin$dir[i],8,10))+as.numeric(substr(flightin$dir[i],11,12))/60
				        if(substr(flightin$dir[i],13,13)=='W') lon2=lon2*-1
				}
#print(c(lat2,lon2))
			}
			hdg2=round((atan2(sin((lon2-lon1)*pi/180)*cos(lat2*pi/180),cos(lat1*pi/180)*sin(lat2*pi/180)-sin(lat1*pi/180)*cos(lat2*pi/180)*cos((lon2-lon1)*pi/180))/pi*180+360) %% 360) # this is spherical geometry
			hdg2=hdg2%%360
			if(hdg1-hdg2>=180){ turn=1 } else if(hdg1-hdg2<(-180)){ turn=-1 } else if(hdg1>hdg2){ turn=-1 }else if(hdg1<hdg2){ turn=1 } # 1 = turn right, -1 = turn left
#			if(hdg2>hdg1&hdg2-hdg1<=180|hdg2<hdg1&hdg1-hdg2>180){ turn=1 # turn right
#			} else if(hdg2>hdg1&hdg2-hdg1>180|hdg2<hdg1&hdg1-hdg2<=180){ turn=-1 } # turn left
#print(c(lat1,lat2,lon1,lon2,hdg1,hdg2,turn))
			# else no turn needed (hdg2=hdg1)
			if(substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='approach'){ # needed for arhold
				alttime=abs(flightin$alt2[i]-flightin$alt2[i-1])/rate # minutes to make alt change
				altspeed=((flightin$alt2[i]+flightin$alt2[i-1])/2-pblalt)/(maxalt-pblalt)*(maxgs-pblgs)+pblgs # average ground speed during climb/descent
				altdist=altspeed*alttime/60
			}
		}
	} else { # dir = NA
		hdg2=hdg1
	}
	# fly segment
	endseg=F
	hdsec=0
	cumsec=0
	cumdist=0
	cumfuel=0
	arhold=0
	gthold=0
	for(sec in c(1:1E5)*tstep){ # step by tstep seconds for a maximum of 1E5*tstep, for tstep = 1/3 seconds, this is 1 degree in 2 min turn, 8 feet in climb, for a maximum of 9 hours per segment
		hdg=hdg1+turn*360/120*tstep
		avehdg=(hdg+hdg1)/2
		hdg=hdg%%360
		avehdg=avehdg%%360
		# check if turn done
		if(turn!=0&round(hdg)%%360==round(hdg2)%%360&substr(flightin$dir[i],3,8)!='spiral'){ turn=0 } # stop turning
#		if(turn==-1&round(hdg)%%360==round(hdg2)%%360){ turn=0 } # stop turning
#		if(hdg<0){ hdg=hdg+360 }
#		if(hdg>360){ hdg=hdg-360 }
#		if(hdg==0){ hdg=360 } # not sure if this is needed, but will plan to use 360 instead of zero for input (### test if needed, and if both 0 and 360 work)
		# if location specified, and turn=0 for 1 minute, check/adjust heading
		if(!is.na(flightin$dir[i])){
			if(substr(flightin$dir[i],1,2)=='sc'|substr(flightin$dir[i],1,2)=='gt'|substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='inbound'|flightin$dir[i]=='approach'){
				if(turn==0){
					hdsec=hdsec+tstep
				}
#print(c(hdg1,hdg2,turn))
#if(sec%%10==0) print(c(hdg1,hdg2,turn))
				if(hdsec>=hdstep){
       			        	hdg2=round((atan2(sin((lon2-lon1)*pi/180)*cos(lat2*pi/180),cos(lat1*pi/180)*sin(lat2*pi/180)-sin(lat1*pi/180)*cos(lat2*pi/180)*cos((lon2-lon1)*pi/180))/pi*180+360) %% 360) # this is spherical geometry
					hdg2=hdg2%%360
					if(hdg1-hdg2>=180){ turn=1 } else if(hdg1-hdg2<(-180)){ turn=-1 } else if(hdg1>hdg2){ turn=-1 }else if(hdg1<hdg2){ turn=1 } # 1 = turn right, -1 = turn left
#print(c(hdg1,hdg2,turn))
#       			               	if(hdg2>hdg1&hdg2-hdg1<=180|hdg2<hdg1&hdg1-hdg2>180){ turn=1 # turn right
#       			               	} else if(hdg2>hdg1&hdg2-hdg1>180|hdg2<hdg1&hdg1-hdg2<=180){ turn=-1 } # turn left
					hdsec=0
				}
#if(flightin$phase[i]==3) print(c(hdg1,hdg2,turn))
			}
		}
		if(!is.na(flightin$dir[i])){
		if(substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='approach'){ # set updown to 0 until needed
			distance=acos(sin(lat1*pi/180)*sin(lat2*pi/180)+cos(lat1*pi/180)*cos(lat2*pi/180)*cos(lon2*pi/180-lon1*pi/180))*6371000/1852 # nmi, this is spherical geometry
#			alttime=abs(flightin$alt2[i]-flightin$alt2[i-1])/rate # minutes to make alt change
#			altspeed=((flightin$alt2[i]+flightin$alt2[i-1])/2-pblalt)/(maxalt-pblalt)*(maxgs-pblgs)+pblgs # average ground speed during climb/descent
#			altdist=altspeed*alttime/60
			if(distance<=altdist){
#print(c(flightin$alt2[i],flightin$alt2[i-1],flightin$alt2[i]-flightin$alt2[i-1],abs(flightin$alt2[i]-flightin$alt2[i-1])/rate,distance,alttime,altspeed,altdist))
				# start climb/descent
				arhold=0
				if(flightin$alt2[i]>flightin$alt2[i-1]){ updown=1 } else if(flightin$alt2[i]<flightin$alt2[i-1]) { updown=-1 }
			} else {
				# hold off on climb/descent
				updown=0
				arhold=1
			}
#print(c(distance,altdist,arhold,updown,alt))
		}
		}
		alt=alt1+updown*rate/60*tstep
		# check if climb/descent done
		if(updown==1&alt>=alt2){ updown=0 } # stop climbing
		if(updown==-1&alt<=alt2){ updown=0 } # stop descending
#print(c(alt,alt1,updown,rate,tstep))
#if(updown==-1&alt<0) print(c(alt,alt1,alt2,updown,rate,tstep))
		gs=((alt+alt1)/2-pblalt)/(maxalt-pblalt)*(maxgs-pblgs)+pblgs # ground speed in kts
		dist=gs*tstep/60/60
		fb=((alt+alt1)/2-pblalt)/(maxalt-pblalt)*(maxfb-pblfb)+pblfb # fuel burn in lbs/hr
		fuel=fb*tstep/60/60
		if(!is.na(flightin$dir[i])){
			# calculate initial distance to specified point
			if(substr(flightin$dir[i],1,2)=='gt'|substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='approach'){
				gtdist1=acos(sin(lat*pi/180)*sin(lat2*pi/180)+cos(lat*pi/180)*cos(lat2*pi/180)*cos(lon2*pi/180-lon*pi/180))*6371000/1852 # nmi, this is spherical geometry
			}
		}
        	londist=sin(avehdg/180*pi)*dist # nmi, (60*cos(lat)) nm per degree
	        latdist=cos(avehdg/180*pi)*dist # nmi, 60 nm per degree
        	lat=lat1+latdist/60 # deg
        	lon=lon1+londist/(60*cos((lat+lat1)/2/180*pi)) # deg
#print(c(dist,londist,lon1,lon,latdist,lat1,lat))
		# check if specified location reached
		if(!is.na(flightin$dir[i])){
			# calculate updated distance to specified point and check if point reached
			if(substr(flightin$dir[i],1,2)=='gt'|substr(flightin$dir[i],1,2)=='ar'|flightin$dir[i]=='approach'){
				gtdist2=acos(sin(lat*pi/180)*sin(lat2*pi/180)+cos(lat*pi/180)*cos(lat2*pi/180)*cos(lon2*pi/180-lon*pi/180))*6371000/1852 # nmi, this is spherical geometry
#print(c(gtdist1,gtdist2))
				if(gtdist2>gtdist1&gtdist1<gtdistco){ endseg=T } else { gthold=1 } # assume if getting further from waypoint, have arrived ### avoid overshoot? ### allow for initial turn?  ### use distance too
#print(c(gtdist1,gtdist2))
### need to do an sc before a gt to avoid getting further away while turning?
				gtdist1=gtdist2
			}
		}
		# check if segment end reached
		if(is.na(flightin$min[i])){
			if(turn==0&updown==0&arhold==0&gthold==0){ endseg=T }
			if(!is.na(flightin$dir[i])){
                        	if(substr(flightin$dir[i],3,8)=='spiral'&updown==0){ endseg=T }
			}
		} else {
			if(sec/60>=flightin$min[i]){ endseg=T }
		}
		# every 10 sec and at end save values
		cumsec=cumsec+tstep
		cumdist=cumdist+dist
		cumfuel=cumfuel+fuel
		if(cumsec>=svstep|endseg){
			flight=rbind(flight,c(alt,cumsec/60,flightin$phase[i],gs,cumdist,fb,cumfuel,lat,lon,hdg))
			# "alt2"  "min"   "phase" "gs"    "dist"  "fb"    "fuel"  "lat"   "lon"   "hdg"
			segmin=segmin+cumsec/60
			segdist=segdist+cumdist
			segfuel=segfuel+cumfuel
			cumsec=0
			cumdist=0
			cumfuel=0
#print(c(distance,altdist,arhold,updown,alt))
#print(sec)
#print(tail(flight,1))
		}
		# at end, save cumulative values and exit loop
		if(endseg){
        		flightin$endlat[i]=tail(flight$lat,1)
		        flightin$endlon[i]=tail(flight$lon,1)
		        flightin$endalt[i]=alt
		        flightin$enddist[i]=segdist
		        flightin$endfuel[i]=segfuel
        		flightin$endhdg[i]=hdg
		        if(is.na(flightin$min[i])) flightin$min[i]=segmin ### otherwise leave min as originally specified (### check for agreement and warn if not)
			break
		}
		# else, update values and loop
		hdg1=hdg
		alt1=alt
		lat1=lat
		lon1=lon
	}
	print(flightin[i,])
}

# output flight data at svstep (nominally 10 sec) resolution
### add rounding?
write(names(flight),paste(flightname,'.out',sep=''),ncol=ncol(flight))
write(t(flight),paste(flightname,'.out',sep=''),ncol=ncol(flight),append=T)

# output unique lat/lon points from 1-min resolution at 0.25 deg resolution for model interpolation
xyres=0.25
flightxy=flight[seq(1,nrow(flight),60/svstep),] # for svstep = 10 sec (resolution of flight), flightxy pulls every 6th one
#xylat=round(flightxy$lat,1)
#xylon=round(flightxy$lon,1)
xylat=round(flightxy$lat/xyres,0)*xyres
xylon=round(flightxy$lon/xyres,0)*xyres
xylatlon=paste(xylat,xylon)
xyphase=flightxy$phase[!duplicated(xylatlon)]
xylatlon=xylatlon[!duplicated(xylatlon)]
write('lat lon phase',paste(flightname,'_xy.txt',sep=''))
write(paste(xylatlon,xyphase),paste(flightname,'_xy.txt',sep=''),ncol=1,append=T)

# output formatted waypoint file
flightout=flightin
flightout$latdeg=trunc(flightout$endlat)
flightout$latmin=round((flightout$endlat-trunc(flightout$endlat))*60)
flightout$latdeg[abs(flightout$latmin)==60]=flightout$latdeg[abs(flightout$latmin)==60]+flightout$latmin[abs(flightout$latmin)==60]/60 ### work at 0 or 180 deg lon?
flightout$latmin[abs(flightout$latmin)==60]=0
flightout$latmin=abs(flightout$latmin)
flightout$lathem=rep('s',nrow(flightout)); flightout$lathem[flightout$endlat>=0]='n'
flightout$londeg=trunc(flightout$endlon)
flightout$lonmin=round((flightout$endlon-trunc(flightout$endlon))*60)
flightout$londeg[abs(flightout$lonmin)==60]=flightout$londeg[abs(flightout$lonmin)==60]+flightout$lonmin[abs(flightout$lonmin)==60]/60 ### work at 0 or 180 deg lon?
flightout$lonmin[abs(flightout$lonmin)==60]=0
flightout$lonmin=abs(flightout$lonmin)
flightout$lonhem=rep('w',nrow(flightout)); flightout$lonhem[flightout$endlon>=0]='e'
flightout$hours=trunc(cumsum(flightout$min)/60)
flightout$mins=round(cumsum(flightout$min)-trunc(cumsum(flightout$min)/60)*60)
flightout$tdist=round(cumsum(flightout$enddist))
flightout$tfuel=round(fullfuel-cumsum(flightout$endfuel))
formattedout=cbind(flightout$wpnum,flightout$wpname[flightout$wpnum],flightout$alt2,paste(flightout$lathem,formatC(abs(flightout$latdeg),width=2,flag=0),formatC(flightout$latmin,width=2,flag=0),sep=''),paste(flightout$lonhem,formatC(abs(flightout$londeg),width=3,flag=0),formatC(flightout$lonmin,width=2,flag=0),sep=''),round(flightout$min,1),paste(formatC(flightout$hours,width=2,flag=0),':',formatC(flightout$mins,width=2,flag=0),sep=''),round(flightout$enddist),flightout$tdist,round(flightout$endfuel),flightout$tfuel,allphase[flightout$phase])
write('Waypoint #, Waypoint Name, Altitude (ft), Latitude (hDDMM), Longitude (hDDDMM), Segment Time (Min), Flight Time (HH:MM), Segment Distance (nmi), Flight Distance (nmi), Segment Fuel (lbs), Fuel Remaining (lbs), Phase',paste(flightname,'.csv',sep=''))
#formattedout<<-formattedout
#flightin<<-flightin
write(t(formattedout),paste(flightname,'.csv',sep=''),ncol=ncol(formattedout),sep=',',append=T)
selectedout=formattedout[flightin$sel=='Y',c(2,4,5)]
#selectedout<<-selectedout
write('Waypoint Name, Latitude (hDDMM), Longitude (hDDDMM)',paste(flightname,'_sel.csv',sep=''))
#print('x')
if(!is.null(nrow(selectedout))){
	write(t(selectedout),paste(flightname,'_sel.csv',sep=''),ncol=ncol(selectedout),sep=',',append=T)
} else { # 1 or fewer selected
#print(selectedout)
	write(selectedout,paste(flightname,'_sel.csv',sep=''),ncol=length(selectedout),sep=',',append=T)
}

waypoints=flightin[flightin$wpname!='',]
waypoints$min=cumsum(flightin$min)[flightin$wpname!='']

# output modified input file
### add rounding?
write(names(flightin),paste(flightname,'_mod.txt',sep=''),ncol=ncol(flightin))
write(t(flightin),paste(flightname,'_mod.txt',sep=''),ncol=ncol(flightin),append=T)

# output kml file
write(rbind(flight$lon,flight$lat,flight$alt2),paste(flightname,'.tmp',sep=''),ncol=3,sep=',',)
system(paste('cat config_files/header.kml ',flightname,'.tmp config_files/footer.kml > ',flightname,'_GV.kml',sep=''))

# print totals
print(paste('total minutes =',round(sum(flight$min),1)))
print(paste('total hours =',round(sum(flight$min)/60,1)))
print(paste('total dist =',round(sum(flight$dist),1)))
print(paste('total fuel =',round(sum(flight$fuel),1)))
print(paste('remaining fuel =',round(fullfuel-sum(flight$fuel),1)))

print(paste('dim(flight)',dim(flight)))

# make plots
collist=c('red','green','blue','violet','cyan','orange','dark green','dark blue','brown','yellow')
collist=c(rep(collist,5)) # up to 50 segments colored

print('plot1')
png(paste(flightname,'_%d.png',sep=''),width=700,height=700)
par(mar=c(5,5,4,2))
plot(flight$lat,flight$alt2,xlab='Latitude',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2)
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lat[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlat[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$endlat[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')
print('plot2')
plot(flight$lon,flight$alt2,xlab='Longitude',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2)
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lon[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlon[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$endlon[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')
xlm=range(flight$lon);ylm=range(flight$lat)
if(abs(cos(mean(flight$lat)/180*pi))*diff(xlm)>diff(ylm)){ ylm=mean(ylm)+c(-1,1)*abs(cos(mean(flight$lat)/180*pi))*diff(xlm)/2 } # x distance > y distance, set xlim to be square wrt ylim
if(abs(cos(mean(flight$lat)/180*pi))*diff(xlm)<diff(ylm)){ xlm=mean(xlm)+c(-1,1)*1/abs(cos(mean(flight$lat)/180*pi))*diff(ylm)/2 } # x distance < y distance, set xlim to be square wrt ylim
print('plot3')
plot(flight$lon,flight$lat,xlab='Longitude',ylab='Latitude',type='n',cex.axis=2,cex.lab=2,xlim=xlm,ylim=ylm)
map.grid(c(-360,0,-90,90),72,36,col='black')
map('worldHires',add=T,resolution=1,fill=T,col='white',wrap=c(-360,0))
box()
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lon[flight$phase==phs],flight$lat[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlon[waypoints$phase==phs],waypoints$endlat[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=2,col=collist[colnum])
points(waypoints$endlon[waypoints$phase==phs],waypoints$endlat[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')

print('plot4')
plot(cumsum(flight$min)/60,flight$alt2,xlab='Hours',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2)
colnum=1
for(phs in 1:length(allphase)){
lines((cumsum(flight$min)/60)[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$min[waypoints$phase==phs]/60,waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$min[waypoints$phase==phs]/60,waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')

dev.off()

# make zoom plots
zm=F
if(file.exists(paste(flightname,'.zm',sep=''))){
zm=T

zoom=scan(paste(flightname,'.zm',sep=''),skip=1,comment.char='#')

print('zoom plot1')
png(paste(flightname,'_zm_%d.png',sep=''),width=700,height=700)
par(mar=c(5,5,4,2))
plot(flight$lat,flight$alt2,xlab='Latitude',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2,xlim=c(zoom[1],zoom[2]),ylim=c(zoom[5],zoom[6]))
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lat[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlat[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$endlat[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')
print('zoom plot2')
plot(flight$lon,flight$alt2,xlab='Longitude',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2,xlim=c(zoom[3],zoom[4]),ylim=c(zoom[5],zoom[6]))
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lon[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlon[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$endlon[waypoints$phase==phs],waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')
xlm=range(flight$lon);ylm=range(flight$lat)
print('zoom plot3')
plot(flight$lon,flight$lat,xlab='Longitude',ylab='Latitude',type='n',cex.axis=2,cex.lab=2,xlim=c(zoom[3],zoom[4]),ylim=c(zoom[1],zoom[2]))
map.grid(c(-360,0,-90,90),72,36,col='black')
map('worldHires',add=T,resolution=1,fill=T,col='white',wrap=c(-360,0))
box()
colnum=1
for(phs in 1:length(allphase)){
lines(flight$lon[flight$phase==phs],flight$lat[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$endlon[waypoints$phase==phs],waypoints$endlat[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=2,col=collist[colnum])
points(waypoints$endlon[waypoints$phase==phs],waypoints$endlat[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')

print('zoom plot4')
tsel=flight$lat>=zoom[1]&flight$lat<=zoom[2]&flight$lon>=zoom[3]&flight$lon<=zoom[4]&flight$alt2>=zoom[5]&flight$alt2<=zoom[6]

plot((cumsum(flight$min)/60)[tsel],flight$alt2[tsel],xlab='Hours',ylab='Altitude (feet)',type='n',cex.axis=2,cex.lab=2,ylim=c(zoom[5],zoom[6]))
colnum=1
for(phs in 1:length(allphase)){
lines((cumsum(flight$min)/60)[flight$phase==phs],flight$alt2[flight$phase==phs],col=collist[colnum],lwd=4)
if(any(waypoints$phase==phs)){
text(waypoints$min[waypoints$phase==phs]/60,waypoints$alt2[waypoints$phase==phs],waypoints$wpname[waypoints$phase==phs],pos=1,col=collist[colnum])
points(waypoints$min[waypoints$phase==phs]/60,waypoints$alt2[waypoints$phase==phs],pch=19,col=collist[colnum],cex=1.5)
}
colnum=colnum+1
}
legend('topleft',allphase,col=collist,lwd=4,cex=1.5,bty='n')

dev.off()

system(paste('cp ',flightname,'.zm ',netdir,sep=''))

} # if zoom file exists

system(paste('cp ',flightname,'.txt ',netdir,sep=''))
system(paste('cp ',flightname,'.utm_flt_line.in.py ',netdir,sep=''))
system(paste('tar -zcvf ',
             'fltpln_tool.tar.gz config_files fpnn.r all_fpnn.r',
             'utm_flt_lines.py EXAMPLE_FLIGHT.utm_flt_line.in.py ',
             'mk_plan.sh EXAMPLE_FLIGHT.txt EXAMPLE_FLIGHT.zm'))
system(paste('mv ','fltpln_tool.tar.gz ',netdir,sep=''))

system(paste('mv ',flightname,'*.png ',netdir,sep=''))
system(paste('mv ',flightname,'_sel.csv ',netdir,sep=''))
system(paste('mv ',flightname,'.csv ',netdir,sep=''))
system(paste('mv ',flightname,'_GV.kml ',netdir,sep=''))
system(paste('mv ',flightname,'.out ',netdir,sep=''))
system(paste('mv ',flightname,'_xy.txt ',netdir,sep=''))
system(paste('mv ',flightname,'_mod.txt ',netdir,sep=''))
system(paste('rm ',flightname,'.tmp',sep=''))

# make flight html
write(paste('<HTML><TITLE> Flight Planning Page for ',flightname,'</TITLE><BODY TEXT="#000000" BGCOLOR="#FFFFFF">',sep=''),paste(flightname,'.html',sep=''))
write(paste('<H2>Flight Planning Page for flight ',flightname,'</H2>',sep=''),paste(flightname,'.html',sep=''),append=T)
dstamp<-system('date',intern=T)
write(paste('<H3>CREATED ',dstamp,'</H3>',sep=''),paste(flightname,'.html',sep=''),append=T)

write('<table style="width: 950px; text-align: center;" border="0" cellpadding="2" cellspacing="0"><tbody><tr>',paste(flightname,'.html',sep=''),append=T)

write('<td><td align="left">',paste(flightname,'.html',sep=''),append=T)

write('<H3>Output Files</H3>',paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_sel.csv"> Selected Waypoints</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'.csv"> All Waypoints</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_GV.kml"> KML Output</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'.txt"> Input File</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
if(zm) write(paste('<A HREF="',flightname,'.zm"> Zoom File</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
if(!is.null(rsinfiles)){
	for(i in c(1:length(rsinfiles))){
		write(paste('<A HREF="',rsinfiles[i],'"> Remote Sensing Input ',i,'</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
	}
}
write(paste('<A HREF="',flightname,'_mod.txt"> Modified Input File</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'.out"> Full Output File</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_xy.txt"> Unique Lat/Lon Pairs</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)

write('</td><td><td align="left">',paste(flightname,'.html',sep=''),append=T)

write('<H3>Totals</H3>',paste(flightname,'.html',sep=''),append=T)
#write(paste('Total Minutes = ',round(sum(flight$min),1),'<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Total Hours = ',round(sum(flight$min)/60,1),'<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Total Dist = ',round(sum(flight$dist),1),' nmi<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Average Speed = ',round(sum(flight$dist)/sum(flight$min)*60,1),' kts<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Total Fuel = ',round(sum(flight$fuel),1),' lbs<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Remaining Fuel = ',round(fullfuel-sum(flight$fuel),1),' lbs<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Average Burn = ',round(sum(flight$fuel)/sum(flight$min)*60,1),' lbs/hr<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)

write('</td><td><td align="left">',paste(flightname,'.html',sep=''),append=T)

write('<H3>Assumptions</H3>',paste(flightname,'.html',sep=''),append=T)
write(paste('Ascent/Descent Rate = ',rate,' ft/min<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste(maxalt,' Ft. Alt. Ground Speed = ',maxgs,' kts<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste(pblalt,' Ft. Alt. Ground Speed = ',pblgs,' kts<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('Full Fuel = ',fullfuel,' lbs.<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste(maxalt,' Ft. Alt. Fuel Burn = ',maxfb,' lbs/hr<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste(pblalt,' Ft. Alt. Fuel Burn = ',pblfb,' lbs/hr<BR>',sep=''),paste(flightname,'.html',sep=''),append=T)

write('</td></tr></table><BR>',paste(flightname,'.html',sep=''),append=T)

write('<H3>Graphics</H3>',paste(flightname,'.html',sep=''),append=T)

write('<table style="width: 950px; text-align: center;" border="0" cellpadding="2" cellspacing="0"><tbody><tr>',paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_3.png"> Map</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_3.png"><img src="',flightname,'_3.png" title="Map" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_4.png"> Alt Timeseries</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_4.png"><img src="',flightname,'_4.png" title="Alt Timeseries" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write('<tr>',paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_1.png"> Alt / Lat Cross Section</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_1.png"><img src="',flightname,'_1.png" title="Alt / Lat Cross Section" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_2.png"> Alt / Lon Cross Section</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_2.png"><img src="',flightname,'_2.png" title="Alt / Lon Cross Section" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write('</tr></table><BR>',paste(flightname,'.html',sep=''),append=T)

if(zm){

write('<H3>Zoomed Graphics</H3>',paste(flightname,'.html',sep=''),append=T)

write('<table style="width: 950px; text-align: center;" border="0" cellpadding="2" cellspacing="0"><tbody><tr>',paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_zm_3.png"> Zoomed Map</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_zm_3.png"><img src="',flightname,'_zm_3.png" title="Zoomed Map" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_zm_4.png"> Zoomed Alt Timeseries</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_zm_4.png"><img src="',flightname,'_zm_4.png" title="Zoomed Alt Timeseries" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write('<tr>',paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_zm_1.png"> Zoomed Alt / Lat Cross Section</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_zm_1.png"><img src="',flightname,'_zm_1.png" title="Zoomed Alt / Lat Cross Section" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<td><A HREF="',flightname,'_zm_2.png"> Zoomed Alt / Lon Cross Section</A><BR>',sep=''),paste(flightname,'.html',sep=''),append=T)
write(paste('<A HREF="',flightname,'_zm_2.png"><img src="',flightname,'_zm_2.png" title="Zoomed Alt / Lon Cross Section" style="border: 0 px solid ; width: 700px; "></A></td>',sep=''),paste(flightname,'.html',sep=''),append=T)
write('</tr></table><BR>',paste(flightname,'.html',sep=''),append=T)


} # if(zm)

write('</BODY></HTML>',paste(flightname,'.html',sep=''),append=T)

system(paste('mv ',flightname,'*.html ',netdir,sep=''))
system(paste('chmod 664 ',netdir,'/',flightname,'.*',sep=''))

# make index html
write('<HTML><TITLE> Flight Planning Page</TITLE><BODY TEXT="#000000" BGCOLOR="#FFFFFF">','index.html')
write('<H2>Flight Planning Page</H2>','index.html',append=T)
dstamp<-system('date',intern=T)
write(paste('<H3>UPDATED ',dstamp,'</H3>',sep=''),'index.html',append=T)

write('<H3>Output and images for individual plans:</H3>','index.html',append=T)

pages=list.files(netdir,pattern='*\\.html$',full.names=T)

# find created date and list in reverse order ### working?
prodate=NULL
for(i in c(1:length(pages))){
#	prodate<-c(prodate,file.info(pages[i])$mtime)
	prodate<-c(prodate,file.mtime(pages[i]))
}

pages=pages[order(prodate,decreasing=T)]
prodate=prodate[order(prodate,decreasing=T)]
for(i in c(1:length(pages))){
	htmlname=tail(unlist(strsplit(pages[i],'/')),1)
	if(htmlname!='index.html'){
		linkname=unlist(strsplit(htmlname,"[.]"))[1]
		wpfile=read.table(paste(netdir,'/',linkname,'.csv',sep=''),stringsAsFactors=F,sep=',',skip=1,comment.char='#')
		hours=round(sum(wpfile[,6]/60),1)
		dist=round(tail(wpfile[,9],1))
		fuel=round(tail(wpfile[,11],1))
		write(paste('<A HREF="',htmlname,'">',linkname,'</A>&nbsp;&nbsp;',hours,' hours,  ',dist,' naut mi,  ',fuel,' lbs rem,  ',sep=''),'index.html',append=T)
		write(paste('Processed on ',as.POSIXlt(prodate[i],origin='1970-01-01'),'<BR>',sep=''),'index.html',append=T)
	}
}

write('<H4><A HREF=http://www.eol.ucar.edu/homes/stephens/ORCAS/FLTPLAN> Main Page </A></H4>','index.html',append=T)

if(netdir=='/net/www/docs/homes/stephens/ORCAS/FLTPLAN'|substr(netdir,1,5)=='/colm'){
write("<H4><A HREF=http://www.cgd.ucar.edu/staff/mclong/orcas/fltpln> Matt Long's Planning Page </A></H4>",'index.html',append=T)
}
if(netdir==''|substr(netdir,1,5)=='/colm'){
    write("<H4><A HREF=http://www.eol.ucar.edu/homes/stephens/ORCAS/FLTPLAN> Britt Stephens' Planning Page </A></H4>",'index.html',append=T)
}
if(netdir==''|netdir=='/net/www/docs/homes/stephens/ORCAS/FLTPLAN'){
    write("<H4><A HREF=http://www.eol.ucar.edu/homes/stephens/ORCAS/FLTPLAN/CS> Colm Sweeney's Planning Page </A></H4>",'index.html',append=T)
}

    write('<H4><A HREF=http://www.eol.ucar.edu/homes/stephens/ORCAS/FLTPLAN/EXTRAS> Extra Flight Plans </A></H4>','index.html',append=T)
    write('<H4><A HREF=instructions.txt> Instructions </A></H4>','index.html',append=T)

#write('<H4><A HREF=all_fpnn.r> All Flight R Code (all_fpnn.r) </A></H4>','index.html',append=T)

write('<H4><A HREF=fltpln_tool.tar.gz> fltpln_tool.tar.gz </A></H4>','index.html',append=T)
write('</BODY></HTML>','index.html',append=T)

system(paste('mv index.html',netdir))

} # end of fpnn function
