#! /usr/bin/env python
from math import pi,cos,sin,sqrt,tan,floor
import numpy as np
import sys,os

#------------------------------------------------------------
#--- projection settings, defaults set to PALTER grid
class utm_prj():
    def __init__(self,
                 angle = -50.,
                 centereast  = 433820.404,
                 centernorth = 2798242.817,
                 zone = -20.):
        self.__dict__['angle'] = angle * pi / 180.
        self.__dict__['centereast']  = centereast
        self.__dict__['centernorth'] = centernorth
        self.__dict__['zone']        = zone

    def __setattr__(self,name,value):
        if name in self.__dict__:
            if name == 'angle':
                value = value * pi / 180.
            self.__dict__[name] = value
        else:
            raise AttributeError('%s has no attribute %s' %
                                 (self.__class__.__name__,name))

#------------------------------------------------------------
#---- class
#------------------------------------------------------------
class flight():
    def __init__(self,
                 spcng1 = 5.75,         # 1.852(km/nm)*3.(nm) ~ 6 km
                 spcng2 = 5.75,
                 alt   = 41000,        # altitude
                 turn_x_km = 41.,       # 1.852(km/nm)*22(nm) ~ 40 km
                 nlines = 6,           # number of lines (must be even)
                 north_km_0 = 610.,     # starting line (grid X)
                 east_km  = [290.,70.]): # Y limits of lines

        #--- instrument properties
        # line spacing for grid
        self.__dict__['spcng1'] = spcng1
        self.__dict__['spcng2'] = spcng2
    
        self.__dict__['alt']   = alt

        # min turn distance for GV
        self.__dict__['turn_x_km'] = turn_x_km # 1.852(km/nm)*22(nm) ~ 40 km

        # first line (UTM x coord)
        self.__dict__['nlines'] = nlines  # number of lines (must be even)

        self.__dict__['north_km_0'] = north_km_0

        # line length = [offshore,inshore]
        self.__dict__['east_km']  = east_km

    def __setattr__(self,name,value):
        if name in self.__dict__:
            self.__dict__[name] = value
        else:
            raise AttributeError('%s has no attribute %s' %
                                 (self.__class__.__name__,name))

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def stn2utm(N,E,prj = utm_prj() ):
    cntrN = (N - 600.) * 1000.
    cntrE = (E - 40.) * 1000.

    v1 = cos(-prj.angle)
    v2 = sin(-prj.angle)
    x = v1 * cntrN - v2 * cntrE + prj.centereast
    y = v2 * cntrN + v1 * cntrE + prj.centernorth

    return x,y

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def utm2stn(x,y,prj = utm_prj() ):
    uu = x - prj.centereast
    vv = y - prj.centernorth
    z1 = cos(prj.angle);
    z2 = sin(prj.angle);
    N = (z1 * uu - z2 *vv) / 1000. + 600.;
    E = (z2 * uu + z1 * vv) / 1000. + 40.;

    return N,E

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def cntrl_merid(iz):
    iutz = 0.;
    cm = 0.;
    if abs(iz) > 30:
	iutz = abs(iz) - 30;
	cm = ((iutz * 6.0) -3.0) * -3600.;
    else:
	iutz = 30 - abs(iz);
	cm = ((iutz * 6.0) +3.0) * +3600.;
    return(cm)

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def utm2geo(x,y,prj = utm_prj() ):

    axis = np.array([6378206.4,6378249.145,6377397.155,
	    6378157.5,6378388.,6378135.,6377276.3452,
	    6378145.,6378137.,6377563.396,6377304.063,
	    6377341.89,6376896.0,6378155.0,6378160.,
	    6378245.,6378270.,6378166.,6378150.] )
    bxis = np.array([6356583.8,6356514.86955,6356078.96284,
	    6356772.2,6356911.94613,6356750.519915,6356075.4133,
	    6356759.769356,6356752.31414,6356256.91,6356103.039,
	    6356036.143,6355834.8467,6356773.3205,6356774.719,
	    6356863.0188,6356794.343479,6356784.283666,
	    6356768.337303])

    ak0    = 0.9996;
    radsec = 206264.8062470964;
    sphere = 8;

    a  = axis[sphere-1]; # major axis size
    b  = bxis[sphere-1]; # minior axis size
    es = (np.power(a,2)  \
          - np.power(b,2)) / np.power(a,2); # eccentricity squared

    if prj.zone < 0:
        yy = y - 1.E7;
    else:
        yy = y;

    xx = x - 5.e5;

    #--- compute central meridian
    cm = cntrl_merid(prj.zone);
    em = yy/ak0;
    um = em/(a * (1.-(es/4.) - (3.*es*es/64.) - (5.*es*es*es/256.)));
    e1 = (1. - sqrt(1.-es))/(1.+sqrt(1.-es));
    phi1 = um + ((3. * e1/2.) - (27.* np.power(e1,3.)/32.))*sin(2.*um) +  \
           ((21.*e1*e1/16.) - (55. * np.power(e1,4.)/32.))*sin(4.*um) +  \
	   (151*np.power(e1,3.)/96) * sin(6.*um);
    en = a/sqrt(1. - es * np.power(sin(phi1),2.));
    t = np.power(tan(phi1),2.);
    epri = es/(1. - es);
    c = epri*cos(phi1);
    r = (a*(1.-es))/(np.power((1.-es* np.power(sin(phi1),2.)),1.5));
    d = xx/(en*ak0);
    phi = phi1 - (en * tan(phi1)/r) * ((d * d/2.) -  \
		                       (5.+3.*t + 10.*c - 4.*c*c - 9.*epri)*np.power(d,4.)/240. +  \
		                       (61.+90.*t +298.*c + 45.*t*t - 252*epri-3.*c*c) *  \
		                       np.power(d,6.) / 720.);
    alam = (cm / radsec) - (d - (1. + 2.*t+c)*np.power(d,3.) /  \
		            6.0 + (5.0 - 2.*c + 28.*t - 3.*c*c+8.*epri +  \
		                   24.*t*t)*np.power(d,5.)/120.) / cos(phi1);
    seclat = phi*radsec;
    seclon = alam*radsec;
    lat = seclat/3600.;
    lon = -seclon/3600.;

    return lat,lon



#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def geo2utm(lat,lon,prj = utm_prj()):
    axis = np.array([6378206.4,6378249.145,6377397.155,
	             6378157.5,6378388.,6378135.,6377276.3452,
	             6378145.,6378137.,6377563.396,6377304.063,
	             6377341.89,6376896.0,6378155.0,6378160.,
	             6378245.,6378270.,6378166.,6378150.])
    bxis = np.array([6356583.8,6356514.86955,6356078.96284,
	             6356772.2,6356911.94613,6356750.519915,6356075.4133,
	             6356759.769356,6356752.31414,6356256.91,6356103.039,
	             6356036.143,6355834.8467,6356773.3205,6356774.719,
	             6356863.0188,6356794.343479,6356784.283666,
	             6356768.337303])

    ak0    = 0.9996;
    radsec = 206264.8062470964;
    sphere = 8;

    #
    a = axis[sphere-1] # major axis size
    b = bxis[sphere-1] # minior axis size

    es = (np.power(a,2.) - np.power(b,2.))/np.power(a,2.); # eccentricity squared
    slat = lat * 3600.   # latitude in seconds
    slon = -lon * 3600.; # longitude in seconds
    cm= 0;               # central meridian in sec
    iutz=0;

    cm = cntrl_merid(prj.zone);

    phi = slat/radsec;
    dlam = -(slon - cm)/radsec;
    epri = es/(1.0 - es);
    en = a/sqrt(1.0 - es * np.power(sin(phi), 2.));
    t = np.power(tan(phi),2.);
    c = epri * np.power(cos(phi),2.);
    aa = dlam * cos(phi);
    s2 = sin(2.0 * phi);
    s4 = sin(4.0 * phi);
    s6 = sin(6.0 * phi);
    f1 = (1.0 - (es/4.)-(3.0*es*es/64.)-(5.0*es*es*es/256.));
    f2 = ((3.*es/8.)+(3.0*es*es/32.)+(45.*es*es*es/1024.));
    f3 = ((15.*es*es/256.)+(45.*es*es*es/1024.));
    f4 = (35.*es*es*es/3072.);
    em = a*(f1*phi - f2*s2 + f3*s4 - f4*s6);
    xx = ak0 * en * (aa + (1.-t+c) * np.power(aa,3.)/6. + \
	             (5. - 18.*t + t*t + 72.*c-58.*epri) \
                     * np.power(aa,5.)/120.) + 5.e5;
    yy = ak0 * (em + en * tan(phi) \
	        *((aa*aa/2.) + \
	          (5.-t+9.*c+4.*c*c)*np.power(aa,4.)/24. + \
	          (61.-58.*t +t*t +600.*c - 330.*epri)* np.power(aa,6.)/720.));
    if prj.zone < 0 or slat < 0:
	yy = yy + 1.e7;

    return xx,yy


#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def grid2ll(N,E,prj = utm_prj() ):
    (x,y) = stn2utm(N,E,prj)
    (lat,lon) = utm2geo(x,y,prj)
    return lat,lon

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def ll2grid(lat,lon,prj = utm_prj() ):
    (x,y) = geo2utm(lat,lon,prj)
    (N,E) = utm2stn(x,y,prj)
    return N,E

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def decdeg2degmin(decdeg):
    sgn = 1.0
    if decdeg < 0:
        sgn = -1.0
    decdeg = abs(decdeg)

    dd = sgn * floor(decdeg)
    mm = round( (decdeg - abs(dd)) * 60. )
    return dd,mm

#------------------------------------------------------------
#---- FUNCTION
#------------------------------------------------------------
def fly_grid(flt,fileo,prj = utm_prj()):

    f = open(fileo,'w')
    f.write('#'+'-'*40+'\n')
    f.write('#--- survey legs\n')
    f.write('#'+'-'*40+'\n')

    #--- report settings
    f.write('# '+'Projection:'+'\n')
    att = vars(prj)
    for k,v in att.items():
        f.write('# '+k+': '+str(v)+'\n')
    f.write('#'+'\n')

    f.write('# '+'Flight:'+'\n')
    att = vars(flt)
    for k,v in att.items():
        f.write('# '+k+': '+str(v)+'\n')
    f.write('#'+'\n')

    #--- compute lines
    north_km = [flt.north_km_0]
    for i in range(0,flt.nlines/2):
        north_km.append(flt.north_km_0
                        + (i+1)*flt.spcng2 + flt.turn_x_km)
        if i < flt.nlines/2-1:
            north_km.append(flt.north_km_0
                                + (i+1)*flt.spcng1)

    f.write('# '+'Flight lines (in X):'+'\n')
    f.write('# '+'\t'.join([str(x) for x in north_km])+'\n')
    f.write('#'+'\n')

    line = 0

    east_km = flt.east_km

    Ytxt = ['S','N']
    if flt.east_km[0] > flt.east_km[1]:
        Ytxt.reverse()
     
    for N in north_km:
        if line == 0:
            leg = 'grid'
        else:
            leg = 'NA'

        #--- ocean side of transect
        lat,lon = grid2ll(N,east_km[0],prj)
        ddS,mmS   = decdeg2degmin(lat)
        ddW,mmW   = decdeg2degmin(lon)
        y0 = '%02d'%abs(ddS)+'%02d'%mmS+'S'+'%03d'%abs(ddW)+'%02d'%mmW+'W'

        #--- peninsula side of transect
        lat,lon = grid2ll(N,east_km[1],prj)
        ddS,mmS   = decdeg2degmin(lat)
        ddW,mmW   = decdeg2degmin(lon)
        y1 = '%02d'%abs(ddS)+'%02d'%mmS+'S'+'%03d'%abs(ddW)+'%02d'%mmW+'W'
            
        f.write('\t'.join([str(flt.alt),'NA','ar'+y0,leg,Ytxt[0]+str(line),'Y'])+'\n')
        f.write('\t'.join([str(flt.alt),'NA','ar'+y1,'NA',Ytxt[1]+str(line),'Y'])+'\n')

        Ytxt.reverse()
        east_km.reverse()
        line += 1

    f.write('#\n')
    f.close()
    os.system('cat '+fileo)

#------------------------------------------------------------
#---- MAIN
#------------------------------------------------------------
if __name__ == "__main__":

    if len(sys.argv) != 3:
        print 'incorrect input'
        print 'Usage: '+sys.argv[0]+' file_in file_out'
        print
        exit(1)


    filei = sys.argv[1]
    fileo = sys.argv[2]            
    
    if not os.path.exists(filei):
        print 'Input file not found: '+filei
        exit(1)

    #--- define default flight
    prj = utm_prj()
    fly = flight()

    #--- possibly over write flight/project settings
    execfile(filei)

    #--- fly it
    fly_grid(fly,fileo,prj)

