/*********************************************************************
 *
 * file name : discoverd.c
 * summary   : program to reply received armadillo-J discover packet
 * coded by  : mori
 * copyright : Atmark Techno
 * date(init): 2003.12.04
 * 
 ********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/uio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netdb.h>
#include <net/route.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <errno.h>

#include <config/autoconf.h>
#include "discover_proto.h"

/* MACRO define */
#define TO_INADDR(x)	(*(struct in_addr*)&x)

/* constant value */
#define DEVICE_NAME				"eth0"
#define DEVICE_NAME_LEN			4
#define PROC_ROUTE_PATH			"/proc/net/route"
#ifndef CONFIG_USER_FLATFSD_FLATFSD
#define NETWORK_FILE_NAME		"/etc/network.d/interface.eth0"
#else
#define NETWORK_FILE_NAME		"/etc/config/interface.eth0"
#endif

/* save network config file to flash */
#define FLATFS_COMMAND			"killall -USR1 flatfsd"
/* notify ip was changed to seri2eth */
#define NOTIFY_CHANGED			"killall -USR1 seri2eth"

#define DEFAULT_IP				0
#define DEFAULT_RT_METRIC		3
#define MAC_ADDRESS_SIZE		6

#define DHCP_CMD				"/bin/dhcpcd"
char* const DHCP_UP_PARAM[]		= {DHCP_CMD, "-t", "30", NULL};
char* const DHCP_DOWN_PARAM[]	= {DHCP_CMD, "-k", NULL};

/* network config file format */
#define	DHCP_KEY_STR			"DHCP"
#define	IPADDRESS_KEY_STR		"IPADDRESS"
#define	NETMASK_KEY_STR			"NETMASK"
#define	GATEWAY_KEY_STR			"GATEWAY"

static pid_t g_dhcp_pid = 0;

#ifdef __DEBUG__
	#define DUMP_PACKET(x)	dump_packet(x)
	#define DUMP_NETWORK_INFO(x)	dump_network_info(x)	

/**********************************************************************
 * summary : dump packet data
 * return  : -
 *********************************************************************/
static void dump_network_info(const network_info_s* net_info)
{
	unsigned long broadcast;

	LOG("--------------- NETWORK START --------------\n\n");
	/* use dhcp */
	if(net_info->use_dhcp){
		LOG("DHCP\tUSING DHCP\n");
	}
	else{
		LOG("DHCP\tSTATIC IP\n");
	}

	/* ip, broadcast, netmask, gateway */
	LOG("ip address\t: %s\n", inet_ntoa(TO_INADDR(net_info->ip_address)));
	LOG("net mask\t: %s\n", inet_ntoa(TO_INADDR(net_info->netmask)));
	broadcast = (net_info->ip_address | ~net_info->netmask);
	LOG("broad cast\t: %s\n", inet_ntoa(TO_INADDR(broadcast)));
	LOG("gate way\t: %s\n", inet_ntoa(TO_INADDR(net_info->gateway)));
	LOG("--------------- NETWORK END ----------------\n\n");
}

static void dump_packet(const discover_packet_s* packet)
{
	char* str;
	const network_info_s* net_info = &packet->net_info;

	LOG("--------------- PACKET START --------------\n\n");

	/* message type */
	switch(ntohl(packet->message_type)){
	case DISCOVER_REQ:
		str = "DISCOVER_REQ";
		break;
	case DISCOVER_REP:
		str = "DISCOVER_REP";
		break;
	case IP_OFFER_REQ:
		str = "IP_OFFER_REQ";
		break;
	case IP_OFFER_REP:
		str = "IP_OFFER_REP";
		break;
	default:
		str = "INVALID_MESSAGE_TYPE";
		break;
	}
	LOG("message type\t: %s\n", str);

	/* transaction ID */
	LOG("transaction ID\t: %d\n", packet->transaction_id);

	/* source mac address */
	LOG("mac addr\t: %02x:%02x:%02x:%02x:%02x:%02x\n",
		(unsigned char)packet->mac_addr[0],
		(unsigned char)packet->mac_addr[1],
		(unsigned char)packet->mac_addr[2],
		(unsigned char)packet->mac_addr[3],
		(unsigned char)packet->mac_addr[4],
		(unsigned char)packet->mac_addr[5]
	);

	/* result */
	LOG("result (reply)\t: %d\n", packet->result);

	dump_network_info(net_info);
	LOG("--------------- PACKET END ----------------\n\n");
}

#else
	#define DUMP_NETWORK_INFO(x)	{}
	#define DUMP_PACKET(x)	{}
#endif

/**********************************************************************
 * summary : return temprary socket descriptor
 * return  : socket desc
 *********************************************************************/
static int get_temp_socket(void)
{
	static int sock = -1;

	if(sock < 0){
		sock = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if(sock < 0){
    		ERR("failed to create socket\n");
		}
    }
	
	return sock;
}

/**********************************************************************
 * summary : return local interface device mac address
 * return  : mac address
 *********************************************************************/
static const char* get_mac_address(void)
{
	static char local_mac_address[6];
	static int init = 0;
	if(!init){
		struct ifreq request;
		strcpy(request.ifr_name, DEVICE_NAME);
    	if(ioctl(get_temp_socket(), SIOCGIFHWADDR, &request) < 0){
        	PERROR(__FUNCTION__);
    	}
		else{
			memcpy(local_mac_address, request.ifr_hwaddr.sa_data, 6);
		}
		init = 1;
	}
	return local_mac_address;
}

/**********************************************************************
 * summary : return broadcast socket address
 * return  : mac address
 *********************************************************************/
static const struct sockaddr_in* get_broadcast_sockaddr()
{
	static int init = 0;
	static struct sockaddr_in sockaddr_broadcast;
	if(!init){
		/* prepare broadcast address, socket, device setting */
		bzero((char *)&sockaddr_broadcast, sizeof(struct sockaddr_in));
		sockaddr_broadcast.sin_family = AF_INET;
		sockaddr_broadcast.sin_port   = htons(CLIENT_PORT);
		sockaddr_broadcast.sin_addr.s_addr = INADDR_BROADCAST;
		init = 1;
	}

	return &sockaddr_broadcast;
}

/**********************************************************************
 * summary : get indicated network address by ioctl
 * return  : network address (failed -> 0)
 *********************************************************************/
static unsigned long get_address(
int cmd /* command of ioctl ex. SIOC... */
)
{
	struct sockaddr_in* addr_info;
	struct ifreq request;
	strcpy(request.ifr_name, DEVICE_NAME);

	/* ip address */
	if(ioctl(get_temp_socket(), cmd, &request) < 0){
		PERROR(__FUNCTION__);
		return DEFAULT_IP;
	}

	addr_info = (struct sockaddr_in*)&request.ifr_addr;
	return addr_info->sin_addr.s_addr;
}

/**********************************************************************
 * summary : set indicated network address by ioctl
 * return  : success -> 0, failed -> -1
 *********************************************************************/
static int set_address(
	int cmd,			/* command of ioctl ex. SIOC... */
	unsigned long addr	/* network address to set */
)
{
	struct ifreq request;
	struct sockaddr_in* addr_info = (struct sockaddr_in*)&request.ifr_addr;
	addr_info->sin_family	= AF_INET;
	addr_info->sin_port		= 0;
	addr_info->sin_addr.s_addr = addr;

	strcpy(request.ifr_name, DEVICE_NAME);

	/* ip address */
    if(ioctl(get_temp_socket(), cmd, &request) < 0){
		PERROR(__FUNCTION__);
		return -1;
    }

	return 0;
}

/**********************************************************************
 * summary : get gateway address from proc file system and return it
 * return  : default gateway address (failed -> 0)
 *********************************************************************/
static unsigned long get_default_gateway(void)
{
	char buff[1024];

	/* open proc file system */
	FILE *fp = fopen(PROC_ROUTE_PATH, "r");
	if(!fp){
		ERR("failed to open proc file : %s\n", PROC_ROUTE_PATH);
		return DEFAULT_IP;
	}

	/* ignore first line (only format string) */
	if(!fgets(buff, sizeof(buff), fp)){
		ERR("EOF found\n");
		fclose(fp);
		return DEFAULT_IP;
	}

	/* find line it discribes default gateway */
	while(fgets(buff, sizeof(buff), fp)){
		const char* fmt = "%s %x %x";
		struct in_addr ip, gw;
		char name[64];
		int num = sscanf(buff, fmt, name, &ip, &gw);
		if(num != 3){
			ERR("sscanf() failed\n");
			continue;
		}

		if(!strncmp(name, DEVICE_NAME, DEVICE_NAME_LEN) &&
		gw.s_addr != 0){
			fclose(fp);
			return gw.s_addr;
		}
    }

	ERR("no default gw on : %s\n", DEVICE_NAME);

	fclose(fp);
	return DEFAULT_IP;
}

/**********************************************************************
 * summary : set default gateway address by ioctl
 * return  : -1 - failed, 0 - success, 1 : already exist
 *********************************************************************/
static int set_default_gateway(
	int cmd,					/* command of ioctl ex. SIOC... */
	unsigned long default_gw	/* default gateway address */
)
{
	int result;
	struct rtentry route;
	struct sockaddr_in address;

	if(!default_gw){
		// not indicated is same as OK
		return 0;
	}

	/* set parameter */
	bzero(&route, sizeof(route));
	route.rt_dev		= DEVICE_NAME;
	route.rt_flags		= RTF_UP | RTF_GATEWAY;
	address.sin_family	= AF_INET;
	address.sin_port	= 0;

	address.sin_addr.s_addr = INADDR_ANY;
	memcpy(&route.rt_dst, &address, sizeof(address));

	address.sin_addr.s_addr = default_gw;
	memcpy(&route.rt_gateway, &address, sizeof(address));

	if(ioctl(get_temp_socket(), cmd, &route) < 0){
		if(errno == EEXIST){
			return 1;
		}
		PERROR(__FUNCTION__);
		return -1;
	}

	return 0;
}

/**********************************************************************
 * summary : wake up dhcp client daemon
 * return  : process id of dhcp
 *********************************************************************/
static pid_t start_dhcp()
{
	pid_t dhcp_pid = vfork();
	if(dhcp_pid < 0){
		ERR("vfork() failed\n");
		return -1;
	}
	if(!dhcp_pid){
		LOG("Child process to wake dhcp started, cmd : %s %s\n",
			DHCP_CMD, DHCP_UP_PARAM);
		execv(DHCP_CMD, DHCP_UP_PARAM);
		_exit(0);
	}

	LOG("Child process ID : %d\n", dhcp_pid);
	return dhcp_pid;
}

/**********************************************************************
 * summary : kill dhcp client process
 * return  :
 *********************************************************************/
static void stop_dhcp(pid_t dhcp_pid)
{
	int ret;
	pid_t kill_pid = vfork();
	if(kill_pid < 0){
		ERR("vfork() failed\n");
		return;
	}
	if(!kill_pid){
		LOG("Child process to kill dhcp started\n");
		execv(DHCP_CMD, DHCP_DOWN_PARAM);
		_exit(0);
	}

	ret = waitpid(dhcp_pid, NULL, 0);
	ret = waitpid(kill_pid, NULL, 0);
}

/**********************************************************************
 * summary : get local network information then set to parameter
 * return  : success -> 0, failed -> -1
 *********************************************************************/
static int load_local_network_info(
	network_info_s* net_info
)
{
	char buff[256];
	FILE *fp;

	LOG("in load_local_network_info()\n");
	/* set default */
	memset(net_info, 0, sizeof(network_info_s));
	net_info->use_dhcp = 1;

	/* load setting from config file */
	fp = fopen(NETWORK_FILE_NAME, "r");
 	if(!fp){
		PERROR(__FUNCTION__);
		return -1;
	}

	while(fgets(buff, sizeof(buff), fp)){
		const char* fmt = "%s %s";
		char key[64];
		char value[64];
		
		int num = sscanf(buff, fmt, key, value);
		if(num != 2){
			continue;
		}

		LOG("setting, key : %s, value : %s\n", key, value);
		if(!strcmp(key, DHCP_KEY_STR)){
			net_info->use_dhcp = strcmp(value, "yes")? 0 : 1;
		}
		else if(!strcmp(key, IPADDRESS_KEY_STR)){
			inet_aton(value, (struct in_addr*)&net_info->ip_address);
		}
		else if(!strcmp(key, NETMASK_KEY_STR)){
			inet_aton(value, (struct in_addr*)&net_info->netmask);
		}
		else if(!strcmp(key, GATEWAY_KEY_STR)){
			inet_aton(value, (struct in_addr*)&net_info->gateway);
		}
	}

	fclose(fp);

	DUMP_NETWORK_INFO((const network_info_s*)net_info);

	return 0;
}

/**********************************************************************
 * summary : get local network information then set to parameter
 * return  : success -> 0, failed -> -1
 *********************************************************************/
static int update_local_network_info(
	network_info_s* net_info
)
{
    /* ip, net mask, default gw */
	net_info->ip_address    = get_address(SIOCGIFADDR);
	net_info->netmask       = get_address(SIOCGIFNETMASK);
	net_info->gateway       = get_default_gateway();

	return 0;
}

/**********************************************************************
 * summary : set default route address
 * return  : success -> 0, failed -> -1
 *********************************************************************/
static int set_default_route(void)
{
	struct rtentry route;
	struct sockaddr_in address;

	/* set parameter */
	bzero(&route, sizeof(route));
	address.sin_family	= AF_INET;
	address.sin_port	= 0;

	address.sin_addr.s_addr = INADDR_ANY;
	memcpy(&route.rt_dst, &address, sizeof(address));
	memcpy(&route.rt_genmask, &address, sizeof(address));

	route.rt_dev	= DEVICE_NAME;
	route.rt_flags	= RTF_UP;
	route.rt_metric = DEFAULT_RT_METRIC; /* default gw must be 0 or 1 */

	if(ioctl(get_temp_socket(), SIOCADDRT, &route) < 0){
		PERROR(__FUNCTION__);
		return -1;
	}

	return 0;
}

/**********************************************************************
 * summary : backup flash rom 
 * return  :
 *********************************************************************/
static void flash_backup(void)
{
	system(NOTIFY_CHANGED);
	system(FLATFS_COMMAND);
	sleep(3);	/* to assure saved */
}
						    
/**********************************************************************
 * summary : save network information to file
 * return  : success -> 0, failed -> -1
 *********************************************************************/
static int save_network_info(
	const network_info_s* net_info	/* network information to save */
)
{
	unsigned long broadcast;
	char work_str[256];
	const char* fmt = "%s %s\n";

	FILE *fp = fopen(NETWORK_FILE_NAME, "w");
 	if(!fp){
		PERROR(__FUNCTION__);
		return -1;
	}

	/** error of fwrite() is not cared... */

	/* DHCP */
	if(net_info->use_dhcp){
		sprintf(work_str, fmt, DHCP_KEY_STR, "yes");
		fwrite(work_str, 1, strlen(work_str), fp);
	}
	else{

		sprintf(work_str, fmt, DHCP_KEY_STR, "no");
		fwrite(work_str, 1, strlen(work_str), fp);

		/* ip address */
		sprintf(work_str, fmt, IPADDRESS_KEY_STR,
					inet_ntoa(TO_INADDR(net_info->ip_address)));
		fwrite(work_str, 1, strlen(work_str), fp);

		/* netmask */
		sprintf(work_str, fmt, NETMASK_KEY_STR,
					inet_ntoa(TO_INADDR(net_info->netmask)));
		fwrite(work_str, 1, strlen(work_str), fp);

		/* gateway address */
		sprintf(work_str, fmt, GATEWAY_KEY_STR,
					inet_ntoa(TO_INADDR(net_info->gateway)));
		fwrite(work_str, 1, strlen(work_str), fp);
	}

	fclose(fp);

	flash_backup();

	return 0;
}

/**********************************************************************
 * summary : wake up network interface
 * return  : -
 *********************************************************************/
static void up_interface_dev(void)
{
	struct ifreq request;
	memset(&request, 0, sizeof(request));
	strcpy(request.ifr_name, DEVICE_NAME);

	/*
	 * up interface
	 * need IFF_PROMISC otherwise cann't reply response
	 * after using DHCP client
	 */
	request.ifr_flags |= IFF_UP | IFF_BROADCAST | IFF_RUNNING |
						IFF_NOTRAILERS | IFF_MULTICAST;
   	if(ioctl(get_temp_socket(), SIOCSIFFLAGS, &request) < 0){
		PERROR(__FUNCTION__);
   	}

	return;
}

/**********************************************************************
 * summary : initialize local network
 * return  : -
 *********************************************************************/
static void init_local_network(const network_info_s* local_net_info)
{
	struct ifreq request;
	strcpy(request.ifr_name, DEVICE_NAME);

	if(local_net_info->use_dhcp){
		/* call dhcp client , set address -> 0.0.0.0 */
		set_address(SIOCSIFADDR, 0);
		g_dhcp_pid = start_dhcp(DHCP_UP_PARAM);
		LOG("dhcp started\n");
	}
	else{
		unsigned long broadcast = 
			local_net_info->ip_address | ~local_net_info->netmask;
		set_address(SIOCSIFADDR, local_net_info->ip_address);
		set_address(SIOCSIFNETMASK, local_net_info->netmask);
		set_address(SIOCSIFBRDADDR, broadcast);
		set_default_gateway(SIOCADDRT, local_net_info->gateway);
	}

	return;
}

/**********************************************************************
 * summary : change local network configuration by indicated info
 * return  : result_e (see discover_proto.h)
 *********************************************************************/
static result_e	change_network_setting(
	const network_info_s* new_net_info,	/* requested network conf */
	network_info_s* local_net_info		/* local network conf (IN/OUT) */
)
{
	unsigned long broadcast;

	if(new_net_info->use_dhcp){
		/*
		 * DHCP (call DHCP client)
		 */
		if(local_net_info->use_dhcp){
			return SUCCESS;
		}
		
		/* set address -> 0.0.0.0 */
		set_address(SIOCSIFADDR, 0);
		g_dhcp_pid = start_dhcp(DHCP_UP_PARAM);
		LOG("dhcp started\n");
	}
	else{
		/*
		 * STATIC IP
		 */
		if(set_address(SIOCSIFADDR, new_net_info->ip_address) < 0){
			ERR("failed to change ip address : %s\n",
				inet_ntoa(TO_INADDR(new_net_info->ip_address)));
			return SET_IP_ADDRESS_FAILED;
		}

		if(set_address(SIOCSIFNETMASK, new_net_info->netmask) < 0){
			ERR("failed to change net mask : %s\n",
				inet_ntoa(TO_INADDR(new_net_info->netmask)));
			init_local_network(local_net_info);
			return SET_SUBNETMASK_FAILED;
		}

		broadcast = new_net_info->ip_address | ~new_net_info->netmask;
		if(set_address(SIOCSIFBRDADDR, broadcast) < 0){
			ERR("failed to broadcast address : %s\n",
				inet_ntoa(TO_INADDR(broadcast)));
			init_local_network(local_net_info);
			return SET_BROAD_CAST_FAILED;
		}

		/* set default gw */
		switch(set_default_gateway(SIOCADDRT, new_net_info->gateway)){
		case 0:
			/* remove old gateway if any */
			if(local_net_info->gateway){
				LOG("removing old gateway\n");
				set_default_gateway(SIOCDELRT, local_net_info->gateway);
			}
			break;
		case 1:
			/* already set */
			LOG("default gateway already exist\n");
			break;
		default:
			ERR("failed to set gw address : %s\n",
				inet_ntoa(TO_INADDR(new_net_info->gateway)));
			init_local_network(local_net_info);
			return SET_DEFAULT_GW_FAILED;
		}

		/* kill if dhcp is exist, but all 'route' has gone after kill dhcp, 
		   need to set again in signal handler */
		if(g_dhcp_pid > 0){
			LOG("send kill message to child process\n");
			stop_dhcp(g_dhcp_pid); /* wait until killed */
			g_dhcp_pid = 0;
			up_interface_dev();
			init_local_network(new_net_info);
		}
	}

	/* set setting and save setting to file */
	memcpy(local_net_info, new_net_info, sizeof(network_info_s));
	save_network_info(local_net_info);

	return SUCCESS;
}

/**********************************************************************
 * summary : reply network configuration data
 * return  : -
 *********************************************************************/
static void discover_req(
	int udp_sock,							/* UDP protocol socket */
	network_info_s* local_net_info,			/* local net work conf */
	const discover_packet_s* recv_packet	/* received packet */
)
{
	int result;
	discover_packet_s send_packet;

	DUMP_PACKET(recv_packet);

	update_local_network_info(local_net_info);

	/* make packet to send */
	bzero((char *)&send_packet, sizeof(send_packet));
	send_packet.message_type = htonl(DISCOVER_REP);
	memcpy(
			send_packet.mac_addr,
			get_mac_address(),
			MAC_ADDRESS_SIZE
			);
	send_packet.transaction_id = recv_packet->transaction_id;
	memcpy(&send_packet.net_info, local_net_info, sizeof(network_info_s));
	
	LOG("SEND PACKET DUMP\n");
	DUMP_PACKET(&send_packet);

	/* send packet */
	set_default_route();
	result = sendto(
		udp_sock,
		&send_packet, sizeof(send_packet),
		0,
	   	(const struct sockaddr *)get_broadcast_sockaddr(),
		sizeof(struct sockaddr_in)
		);
	if(result < 0){
		PERROR(__FUNCTION__);
		ERR("sendto() failed\n");
	}
}

/**********************************************************************
 * summary : change network config by request and send reply
 * return  : -
 *********************************************************************/
static void ip_offer_req(
	int udp_sock,							/* UDP protocol socket */
	network_info_s* local_net_info,			/* local net work conf */
	const discover_packet_s* recv_packet	/* received packet */
)
{
	int result;
	discover_packet_s send_packet;
	memset(&send_packet, 0, sizeof(discover_packet_s));

	DUMP_PACKET(recv_packet);

	/* check if request is mine */
	if(memcmp(
		(void*)recv_packet->mac_addr,
		(void*)get_mac_address(),
		MAC_ADDRESS_SIZE
		)){
		return;
	}

	/* if mac address is same then setting */
	send_packet.result = htonl(change_network_setting(
										&recv_packet->net_info,
										local_net_info
										));
	/* make reply packet */
	send_packet.message_type = htonl(IP_OFFER_REP);
	send_packet.transaction_id = recv_packet->transaction_id;
	memcpy(send_packet.mac_addr, get_mac_address(), MAC_ADDRESS_SIZE);
	memcpy(&send_packet.net_info, local_net_info, sizeof(network_info_s));

	LOG("SEND PACKET DUMP\n");
	DUMP_PACKET(&send_packet);

	/* send packet */
	set_default_route();
	result = sendto(
		udp_sock,
		&send_packet, sizeof(send_packet),
		0,
	   	(const struct sockaddr *)get_broadcast_sockaddr(),
		sizeof(struct sockaddr_in)
		);
	if(result < 0){
		PERROR(__FUNCTION__);
		ERR("sendto() failed\n");
	}
}

/**********************************************************************
 * summary : check received packet type to know request
 * return  : -
 *********************************************************************/
static void analysis_packet(
	int udp_sock,							/* UDP protocol socket */
	network_info_s* local_net_info,			/* local net work conf */
	const discover_packet_s* recv_packet	/* received packet data */
)
{
	switch(ntohl(recv_packet->message_type)){
	case DISCOVER_REQ:
		discover_req(udp_sock,local_net_info, recv_packet);
		break;
	case IP_OFFER_REQ:
		ip_offer_req(udp_sock,local_net_info, recv_packet);
		break;
	default:
		ERR("invalid message type, id : %d\n",
			ntohl(recv_packet->message_type));
		break;
	}
}

/**********************************************************************
 * summary : set up socket desc for udp broadcast
 * return  : socket desc. (failed -> -1)
 *********************************************************************/
static int setup_udp_broadcast_socket(unsigned short local_port)
{
	struct sockaddr_in name;
	int sock;
	int flag;

	/* Set up the address we're going to bind to. */
	name.sin_family = AF_INET;
	name.sin_port = htons(local_port);
	name.sin_addr.s_addr = INADDR_ANY;
	memset (name.sin_zero, 0, sizeof (name.sin_zero));

	/* Make a socket... */
	if((sock = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0){
    	ERR("Can't create socket: \n");
    	return -1;
    }

	/* Set the REUSEADDR option to not fail when restarted. */
	flag = 1;
	if(setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
					(char *)&flag, sizeof(flag)) < 0){
		ERR("Can't set SO_REUSEADDR option on socket\n");
        return -1;
	}

	/* Set the BROADCAST option to send broadcast packet */
	if(setsockopt(sock, SOL_SOCKET, SO_BROADCAST,
					(char *)&flag, sizeof(flag)) < 0){
		ERR("Can't set SO_BROADCAST option on socket\n");
		return -1;
	}

	/* Bind the socket to this interface's IP address. */
	if(bind(sock, (struct sockaddr *)&name, sizeof(name)) < 0){
		ERR ("filed to bind address\n");
		return -1;
	}

	return sock;
}
						    
/**********************************************************************
 * summary : receive signal from dhcpcd 
 * return  :
 *********************************************************************/
void fin_notify(int sig)
{
	LOG("signal received, sig : %d, pid : %d\n", sig, g_dhcp_pid);
	if(sig == SIGCHLD){
		wait(NULL);
		up_interface_dev();
		LOG("child exited\n");
	}
}

/**********************************************************************
 * summary : main routine (run itself as daemon)
 * return  : success -> 0, failed -> !0
 *********************************************************************/
int main(int argc, char **argv)
{
	int udp_sock;
	network_info_s local_net_info;
	struct sockaddr_in sockaddr_sender;
	int len = sizeof(struct sockaddr_in);
	struct timeval select_timeout;

	/* allocate receive buffer more than packet size
	   so that if receive data is bigger than expected,
	   it's gonna be found */
	int buffer_size = sizeof(discover_packet_s) + 1;
	char* buffer = malloc(buffer_size);
	struct sigaction act;

#ifdef __DEBUG__
/*
	pid_t pid = fork();
	if(pid < 0){
		ERR("fork failed");
		return 0;
	}
	if(pid != 0){
		ERR("parent ended now");
		return 0;
	}
		
	setpgrp();
	LOG("run as a daemon now\n");
*/
#endif

	/* set signal to get from dhcpcd */
	act.sa_handler = fin_notify;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	sigaction(SIGCHLD, &act, 0);

	while(1){	/* work infinite */

	if((udp_sock = setup_udp_broadcast_socket(SERVER_PORT)) < 0){
		ERR("setup_udp_broadcast_socket() failed : %d\n", udp_sock);
		continue;
	}
	up_interface_dev();

	/* call after interface up */
	load_local_network_info(&local_net_info);
	init_local_network(&local_net_info);

	/* in wait routine */
	while(1){
		int data_len, ret;
		fd_set readfds;
		FD_ZERO(&readfds);
		FD_SET(udp_sock, &readfds);

		ret = select(udp_sock + 1, &readfds, NULL, NULL, NULL);
		if(ret < 0){
			ERR("select() failed\n");
			up_interface_dev();
			continue;
		}

		data_len = recvfrom(
			udp_sock, buffer, buffer_size, 0,
			(struct sockaddr *)&sockaddr_sender,
			&len
			);
		if(data_len == sizeof(discover_packet_s)){
			analysis_packet(udp_sock, &local_net_info,
							(discover_packet_s*)buffer);
		}
		else{
			ERR("recvfrom() failed, result : %d\n", data_len);
			break;
		}
	}

	close(udp_sock);

	}	/* work infinite loop */

	free(buffer);

	return 0;
}

