/*
 * linux/drivers/mtd/maps/armadilloj.c
 *
 * Copyright (C) 2001-2003 Atmark Techno, Inc.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#define AUTHOR "Atmark Techno, Inc."
#define DESCRIPTION "Armadillo-J map driver"
#define LICENSE "GPL v2"

static inline void disable_write_protect(void)
{
        unsigned int val = readl(NETARM_MEM_MODULE_BASE + NETARM_MEM_CS0_BASE_ADDR);
        writel(val &= ~(NETARM_MEM_BAR_WRT_PROT), NETARM_MEM_MODULE_BASE + NETARM_MEM_CS0_BASE_ADDR);
}

__u8 armadilloj_read8(struct map_info *map, unsigned long ofs)
{
	DEBUG(2, "armadilloj_read8: 0x%lx -> [0x%x]\n", ofs, readb(map->map_priv_1 + ofs));
	return readb(map->map_priv_1 + ofs);
}

__u16 armadilloj_read16(struct map_info *map, unsigned long ofs)
{
	DEBUG(2, "armadilloj_read16: 0x%lx -> [0x%x]\n", ofs, readw(map->map_priv_1 + ofs));
	return readw(map->map_priv_1 + ofs);
}

void armadilloj_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	DEBUG(2, "armadilloj_write16: [0x%x]@%p -> 0x%lx\n", d, &d, adr);
	writew(d, map->map_priv_1 + (adr));
	mb();
}

void armadilloj_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	DEBUG(2, "armadilloj_copy_from: 0x%x@0x%lx -> 0x%p\n", len, from, to);
	memcpy(to, (void *)(map->map_priv_1 + from), len);
}

#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

static struct mtd_info *flash_mtdinfo;

struct map_info armadilloj_flash_map = {
	.name      = "Flash",
	.size      = FLASH_SIZE,
	.buswidth  = FLASH_WIDTH,

	.read8     = armadilloj_read8,
	.read16    = armadilloj_read16,
	.read32    = 0,
	.copy_from = armadilloj_copy_from,

	.write8    = 0,
	.write16   = armadilloj_write16,
	.write32   = 0,
	.copy_to   = 0,
};

/*
 *** verison 1.0.0 Armadillo-J *** (first release version)
 *
 *  0x02000000 +---------------------+
 *             |    Reserved(boot)   | 
 *  0x02020000 +---------------------+
 *             |        Hermit       |
 *  0x02040000 +------------+--------+
 *             | Image      | Kernel |
 *  0x020B0000 |            +--------+
 *             |            |  User  |
 *  0x021E0000 +--------+   |        |
 *             | Backup |   |        |
 *  0x021F0000 +--------+   |        |
 *             | Config |   |        |
 *             +--------+---+--------+
 *
 *** version 2.0.a Armadillo-J *** (memmap changed version [type:2M])
 *
 *  0x02000000 +---------------------+
 *             | BoardParameter      |
 *  0x02010000 +------------+--------+
 *             | Image      | Kernel |
 *  0x02080000 |            +--------+
 *             |            |  User  |
 *  0x021E0000 +--------+   |        |
 *             | Backup |   |        |
 *  0x021F0000 +--------+   |        |
 *             | Config |   |        |
 *             +--------+---+--------+
 *
 *** version 2.0.b Armadillo-J *** (memmap changed version [type:4M])
 *
 *  0x02000000 +---------------------+
 *             | BoardParameter      |
 *  0x02010000 +------------+--------+
 *             | Image      | Kernel |
 *  0x02080000 |            +--------+
 *             |            |  User  |
 *  0x023E0000 +--------+   |        |
 *             | Backup |   |        |
 *  0x023F0000 +--------+   |        |
 *             | Config |   |        |
 *             +--------+---+--------+
 */

struct mtd_partition v100_armadilloj_partitions[] = {
	{
		.name       = "Flash/Reserved",
		.size       = V100_RESERVED_SIZE,
		.offset     = V100_RESERVED_OFFSET,
		.mask_flags = MTD_WRITEABLE, /* force read-only */
	},
	{
		.name       = "Flash/Hermit",
		.size       = V100_HERMIT_SIZE,
		.offset     = V100_HERMIT_OFFSET,
		.mask_flags = MTD_WRITEABLE, /* force read-only */
	},
	{
		.name       = "Flash/Image",
		.size       = V100_IMAGE_SIZE,
		.offset     = V100_IMAGE_OFFSET,
	},
	{
		.name       = "Flash/Kernel",
		.size       = V100_KERNEL_SIZE,
		.offset     = V100_KERNEL_OFFSET,
	},
	{
		.name       = "Flash/User",
		.size       = V100_USER_SIZE,
		.offset     = V100_USER_OFFSET,
	},
	{
		.name       = "Flash/Backup",
		.size       = V100_BACKUP_SIZE,
		.offset     = V100_BACKUP_OFFSET,
	},
	{
		.name       = "Flash/Config",
		.size       = V100_CONFIG_SIZE,
		.offset     = V100_CONFIG_OFFSET,
	},
};

struct mtd_partition v20a_armadilloj_partitions[] = {
        {
                .name       = "Flash/BoardParameter",
                .size       = V20A_PARAM_SIZE,
                .offset     = V20A_PARAM_OFFSET,
                .mask_flags = MTD_WRITEABLE, /* force read-only */
        },
        {
                .name       = "Flash/Dummy",
                .size       = V20A_DUMMY_SIZE,
                .offset     = V20A_DUMMY_OFFSET,
                .mask_flags = MTD_WRITEABLE, /* force read-only */
        },
	{
		.name       = "Flash/Image",
		.size       = V20A_IMAGE_SIZE,
		.offset     = V20A_IMAGE_OFFSET,
	},
	{
		.name       = "Flash/Kernel",
		.size       = V20A_KERNEL_SIZE,
		.offset     = V20A_KERNEL_OFFSET,
	},
	{
		.name       = "Flash/User",
		.size       = V20A_USER_SIZE,
		.offset     = V20A_USER_OFFSET,
	},
	{
		.name       = "Flash/Backup",
		.size       = V20A_BACKUP_SIZE,
		.offset     = V20A_BACKUP_OFFSET,
	},
	{
		.name       = "Flash/Config",
		.size       = V20A_CONFIG_SIZE,
		.offset     = V20A_CONFIG_OFFSET,
	},
};

struct mtd_partition v20b_armadilloj_partitions[] = {
        {
                .name       = "Flash/BoardParameter",
                .size       = V20B_PARAM_SIZE,
                .offset     = V20B_PARAM_OFFSET,
                .mask_flags = MTD_WRITEABLE, /* force read-only */
        },
        {
                .name       = "Flash/Dummy",
                .size       = V20B_DUMMY_SIZE,
                .offset     = V20B_DUMMY_OFFSET,
                .mask_flags = MTD_WRITEABLE, /* force read-only */
        },
	{
		.name       = "Flash/Image",
		.size       = V20B_IMAGE_SIZE,
		.offset     = V20B_IMAGE_OFFSET,
	},
	{
		.name       = "Flash/Kernel",
		.size       = V20B_KERNEL_SIZE,
		.offset     = V20B_KERNEL_OFFSET,
	},
	{
		.name       = "Flash/User",
		.size       = V20B_USER_SIZE,
		.offset     = V20B_USER_OFFSET,
	},
	{
		.name       = "Flash/Backup",
		.size       = V20B_BACKUP_SIZE,
		.offset     = V20B_BACKUP_OFFSET,
	},
	{
		.name       = "Flash/Config",
		.size       = V20B_CONFIG_SIZE,
		.offset     = V20B_CONFIG_OFFSET,
	},
};

struct resource armadilloj_flash_resource = {
	.name  = "Flash",
	.start = FLASH_START,
	.end   = FLASH_START + FLASH_SIZE - 1,
	.flags = IORESOURCE_IO | IORESOURCE_BUSY,
};

#define FLASH_DID_AMD_AM29LV160DB      (0x22490001)
#define FLASH_DID_FUJITSU_MBM29LV160BE (0x22490004)
#define FLASH_DID_ST_M29W160EB         (0x22490020)
#define FLASH_DID_ST_M29DW323DB        (0x225f0020)

#define AJ_VERSION_V100 (0x414A0100)
#define AJ_VERSION_V20A (0x414A020A)
#define AJ_VERSION_V20B (0x414A020B)

#define FL_WORD(addr) (*(volatile unsigned short*)(addr))

static unsigned int armadilloj_version_check(void){
	unsigned int flash_ID;
	FL_WORD(FLASH_BASE) = 0xF0;
	FL_WORD(FLASH_BASE + (0x555 << 1)) = 0xAA;
	FL_WORD(FLASH_BASE + (0x2AA << 1)) = 0x55;
	FL_WORD(FLASH_BASE + (0x555 << 1)) = 0x90;

	flash_ID=(unsigned int)*(unsigned int *)(FLASH_BASE);

	FL_WORD(FLASH_BASE) = 0xF0;

	switch(flash_ID){
	case FLASH_DID_AMD_AM29LV160DB:
	  printk(KERN_INFO "flash memory device = AM29LV160DB detected\n");
	  return AJ_VERSION_V100;
	case FLASH_DID_FUJITSU_MBM29LV160BE:
	  printk(KERN_INFO "flash memory device = MBM29LV160BE detected\n");
	  return AJ_VERSION_V100;
	case FLASH_DID_ST_M29W160EB:
	  printk(KERN_INFO "flash memory device = M29W160EB detected\n");
	  return AJ_VERSION_V20A;
	case FLASH_DID_ST_M29DW323DB:
	  printk(KERN_INFO "flash memory device = M29DW323DB detected\n");
	  return AJ_VERSION_V20B;
	default:
	  printk(KERN_INFO "warning : unknown flash memory device (%08x)."
		 "using default flash memory device : AM29LV160DB\n",flash_ID);
	  return AJ_VERSION_V100;
	}
}

static int __init init_armadilloj_flash (void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	char *part_type;
	int err;
	
	switch(armadilloj_version_check()){
	case AJ_VERSION_V100:
		parts = v100_armadilloj_partitions;
		nb_parts = NB_OF(v100_armadilloj_partitions);
		armadilloj_flash_map.size = V100_FLASH_SIZE;
		armadilloj_flash_resource.end = \
		  FLASH_START + V100_FLASH_SIZE - 1;
		break;
	case AJ_VERSION_V20A:
		parts = v20a_armadilloj_partitions;
		nb_parts = NB_OF(v20a_armadilloj_partitions);	
		armadilloj_flash_map.size = V20A_FLASH_SIZE;
		armadilloj_flash_resource.end = \
		  FLASH_START + V20A_FLASH_SIZE - 1;
		break;	  
	case AJ_VERSION_V20B:
		parts = v20b_armadilloj_partitions;
		nb_parts = NB_OF(v20b_armadilloj_partitions);
		armadilloj_flash_map.size = V20B_FLASH_SIZE;
		armadilloj_flash_resource.end = \
		  FLASH_START + V20B_FLASH_SIZE - 1;
		break;
	}

	part_type = "static";
	
	if (request_resource (&ioport_resource, &armadilloj_flash_resource)) {
		printk(KERN_NOTICE "Failed to reserve Armadillo-J FLASH space\n");
		err = -EBUSY;
		goto out;
	}
	
	armadilloj_flash_map.map_priv_1 = (unsigned long)ioremap(FLASH_START, FLASH_SIZE);
	if (!armadilloj_flash_map.map_priv_1) {
		printk(KERN_NOTICE "Failed to ioremap Armadillo-J FLASH space\n");
		err = -EIO;
		goto out_resource;
	}

	flash_mtdinfo = do_map_probe("cfi_probe", &armadilloj_flash_map);
	if (!flash_mtdinfo) {
		flash_mtdinfo = do_map_probe("map_rom", &armadilloj_flash_map);
	}
	if (!flash_mtdinfo) {
		printk("FLASH probe failed\n");
		err = -ENXIO;
		goto out_ioremap;
	}

	flash_mtdinfo->module = THIS_MODULE;
	
	if (add_mtd_partitions(flash_mtdinfo, parts, nb_parts)) {
		printk("FLASH partitions addition failed\n");
		err = -ENOMEM;
		goto out_probe;
	}

        /* we have write protect in mtd layer, so don't bother. keep
         * it disabled while kernel is running */
        disable_write_protect();
		
	return 0;

out_probe:
	map_destroy(flash_mtdinfo);
	flash_mtdinfo = 0;
out_ioremap:
	iounmap((void *)armadilloj_flash_map.map_priv_1);
out_resource:
	release_resource (&armadilloj_flash_resource);
out:
	return err;
}

static int __init init_armadilloj_maps(void)
{
	printk(KERN_INFO "Initializing Armadillo-J MTD mappings\n");

	init_armadilloj_flash();
	
	return 0;
}
	

static void __exit cleanup_armadilloj_maps(void)
{
	if (flash_mtdinfo) {
		del_mtd_partitions(flash_mtdinfo);
		map_destroy(flash_mtdinfo);
		iounmap((void *)armadilloj_flash_map.map_priv_1);
		release_resource (&armadilloj_flash_resource);
	}
}

module_init(init_armadilloj_maps);
module_exit(cleanup_armadilloj_maps);

MODULE_AUTHOR(AUTHOR);
MODULE_DESCRIPTION(DESCRIPTION);
MODULE_LICENSE(LICENSE);
MODULE_DESCRIPTION("Armadillo-J map driver");
MODULE_LICENSE("GPL");
