/*
 * linux/drivers/char/nx7520port.c
 *
 * Copyright (C) 2004 Atmark Techno, Inc.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#define NS7520PORT_MAJOR	210
#define CHRDEV			"ns7520port"
#define VERSION			CHRDEV ": port driver, (C) 2004 Atmark Techno, Inc."

#define GPIO_CTRL_REG	0xffb00020
#define GPIO_DEV_CHR	"padr"

#define PORT_DIRECTION	0x10
#define PORT_DATA		0x00

#define PORT_MODE_BASE	0x18
#define PORT_DIR_BASE	0x10

#define SERIAL_MODE		0x02
#define OUTPUT_MODE		0x01
#define INPUT_MODE		0x00

#define DATA_MASK		0x000000ff

inline static unsigned long get_register(void)
{
	return *(volatile unsigned long*)GPIO_CTRL_REG;
}

inline static void set_register(unsigned long val)
{
	*(volatile unsigned long*)GPIO_CTRL_REG = val;
}

/****************************************************************

****************************************************************/
static int get_port_direction (unsigned int minor, char *buf)
{
  unsigned long tmp;
  int offset;

  minor &= 0xf;
  if (minor >= 8) {
    return -ENODEV;
  }

  tmp = get_register();

  offset = minor + PORT_MODE_BASE;
  if((tmp >> offset) & 0x1){
    *buf = SERIAL_MODE;
	return 0;
  }
  
  offset = minor + PORT_DIR_BASE;
  if((tmp >> offset) & 0x1){
    *buf = OUTPUT_MODE;
	return 0;
  }

  *buf = INPUT_MODE;
  return 0;
}

/****************************************************************

****************************************************************/
static int get_port_data (unsigned int minor, char *buf)
{
  minor &= 0xf;
  if (minor > 8) {
    return -ENODEV;
  }

  *buf = (char)(get_register() & DATA_MASK);
  if (minor != 8) {
    *buf = (*buf >> minor) & 0x1;
  }
  
  return 0;
}

/****************************************************************

****************************************************************/
static int set_port_direction (unsigned int minor, const char *buf)
{
  unsigned long tmp;

  minor &= 0xf;
  if (minor >= 8) {
    return -ENODEV;
  }

  tmp = get_register();

  switch(*buf){
  case INPUT_MODE:
    tmp &= ~(0x1 << (PORT_MODE_BASE + minor));
	tmp &= ~(0x1 << (PORT_DIR_BASE + minor));
    break;
  case OUTPUT_MODE:
    tmp &= ~(0x1 << (PORT_MODE_BASE + minor));
	tmp |= 0x1 << (PORT_DIR_BASE + minor);
    break;
  case SERIAL_MODE:
    tmp |= 0x1 << (PORT_MODE_BASE + minor);
	if(minor < 4){
		tmp &= ~(0x1 << (PORT_DIR_BASE + minor));
	}
	else{
		tmp |= 0x1 << (PORT_DIR_BASE + minor);
	}
    break;
  default:
    return -EINVAL;
  }

  set_register(tmp);

  return 0;
}

/****************************************************************

****************************************************************/
static int set_port_data (unsigned int minor, const char *buf)
{
  unsigned long tmp;
  minor &= 0xf;
  if (minor > 8) {
    return -ENODEV;
  }

  tmp = get_register();

  if(minor == 8){
	tmp &= ~DATA_MASK;
    set_register(tmp | (*buf & DATA_MASK));
  }
  else{
  	if(*buf){
    	tmp |= (0x1 << minor) & DATA_MASK;
  	}
  	else{
    	tmp &= ~((0x1 << minor) & DATA_MASK);
  	}
  	set_register(tmp);
  }
  
  return 0;
}

/****************************************************************

****************************************************************/
static ssize_t ns7520port_read (
  struct file *file,
  char *buf,
  size_t count,
  loff_t *ppos
)
{
  char tmp;
  int ret;
  int minor = MINOR(file->f_dentry->d_inode->i_rdev);

  switch(minor & 0xfffffff0){
  case PORT_DIRECTION:
    ret = get_port_direction(minor, &tmp);
    break;
  case PORT_DATA:
    ret = get_port_data(minor, &tmp);
    break;
  default:
    return -ENODEV;
  }
    
  if(ret){
    return ret;
  }

  if (put_user (tmp, buf)) {
    return -EFAULT;
  }

  return 1;
}

/****************************************************************

****************************************************************/
static ssize_t ns7520port_write (
  struct file *file,
  const char *buf,
  size_t count,
  loff_t *ppos
)
{
  int ret;
  int minor = MINOR(file->f_dentry->d_inode->i_rdev);

  switch(minor & 0xfffffff0){
  case PORT_DIRECTION:
    ret = set_port_direction(minor, buf);
    break;
  case PORT_DATA:
    ret = set_port_data(minor, buf);
    break;
  default:
    return -ENODEV;
  }
    
  if(ret){
    return ret;
  }

  return 1;
}

/****************************************************************

****************************************************************/
static struct file_operations ns7520port_fops = {
  owner: THIS_MODULE,
  read:  ns7520port_read,
  write: ns7520port_write,
};

/****************************************************************

****************************************************************/
static int __init ns7520port_init (void)
{
  if (!request_region (GPIO_CTRL_REG, 1, GPIO_DEV_CHR)) {
    printk (KERN_WARNING CHRDEV ": I/O port 0x%x is not free.\n",
     GPIO_CTRL_REG);
    return -EIO;
  }
  
  if (register_chrdev (NS7520PORT_MAJOR, CHRDEV, &ns7520port_fops)) {
    printk (KERN_WARNING CHRDEV ": Unable to get major %d.\n",
	    NS7520PORT_MAJOR);
    return -EIO;
  }
  
  printk (KERN_INFO VERSION "\n");
  return 0;
}

/****************************************************************

****************************************************************/
static void __exit ns7520port_cleanup (void)
{
  unregister_chrdev (NS7520PORT_MAJOR, CHRDEV);
  
  release_region (GPIO_CTRL_REG, 1);
}

module_init (ns7520port_init);
module_exit (ns7520port_cleanup);
