/*
 *  linux/drivers/char/aj_led.c
 *
 *  Copyright (C) 2004 Atmark Techno, Inc.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#define AJ_LED_MAJOR	     240
#define CHRDEV		     "aj_led"
#define VERSION		     CHRDEV ": armadillo-j led control driver, (C) 2004 Atmark Techno, Inc."

#define AJ_LED_DEV_CHR	     "ajled"

#define AJ_LED_CTRL_REG      (0xFFB00028)
#define PTR_LED_CTRL         ((unsigned int *)AJ_LED_CTRL_REG)
#define REG_LED_CTRL         ((unsigned int)*PTR_LED_CTRL)

#define PORTC5_OUT_AND_BIT   (0xDFFFDFFF)
#define PORTC5_OUT_OR_BIT    (0x00200000)

#define LED_ON_AND_BIT       (0xFFFFFFDF)
#define LED_OFF_OR_BIT       (0x00000020)

#define LED_STATUS_MASK      (0x00000020)
#define LED_STATUS_ON        (0x00000000)
#define LED_STATUS_OFF       (0x00000020)

/****************************************************************

****************************************************************/
static ssize_t aj_led_read (
    struct file *file,
    char *buf,
    size_t count,
    loff_t *ppos
){
    unsigned int status;
    if(count<4) return -EFAULT;
    memset(buf,0,4);

    REG_LED_CTRL &= PORTC5_OUT_AND_BIT;
    REG_LED_CTRL |= PORTC5_OUT_OR_BIT;

    status=(REG_LED_CTRL & LED_STATUS_MASK);

    switch(status){
    case LED_STATUS_ON:
        memcpy(buf,"ON",2);
        return 2;
    case LED_STATUS_OFF:
        memcpy(buf,"OFF",3);
	return 3;
    }
    return -1;
}

/****************************************************************

****************************************************************/
static ssize_t aj_led_write (
    struct file *file,
    const char *buf,
    size_t count,
    loff_t *ppos
){
    int flag;
    if(memcmp(buf,"ON",2)==0){
        flag=1;
    }else if(memcmp(buf,"OFF",3)==0){
        flag=0;
    }else{
        return -EINVAL;
    }

    REG_LED_CTRL &= PORTC5_OUT_AND_BIT;
    REG_LED_CTRL |= PORTC5_OUT_OR_BIT;

    switch(flag){
    case 1://ON
        REG_LED_CTRL &= LED_ON_AND_BIT;
	return 0;
    case 0://OFF
        REG_LED_CTRL |= LED_OFF_OR_BIT;
	return 0;
    }
    return -1;
}

/****************************************************************

****************************************************************/
static struct file_operations aj_led_fops = {
    owner: THIS_MODULE,
    read:  aj_led_read,
    write: aj_led_write,
};

/****************************************************************

****************************************************************/
static int __init aj_led_init (void)
{
    if (!request_region (AJ_LED_CTRL_REG, 1, AJ_LED_DEV_CHR)) {
        printk (KERN_WARNING CHRDEV ": I/O port 0x%x is not free.\n",
		AJ_LED_CTRL_REG);
	return -EIO;
    }
  
    if (register_chrdev (AJ_LED_MAJOR, CHRDEV, &aj_led_fops)) {
        printk (KERN_WARNING CHRDEV ": Unable to get major %d.\n",
		AJ_LED_MAJOR);
	return -EIO;
    }
    
    printk (KERN_INFO VERSION "\n");
    return 0;
}

/****************************************************************

****************************************************************/
static void __exit aj_led_cleanup (void)
{
    unregister_chrdev (AJ_LED_MAJOR, CHRDEV);
  
    release_region (AJ_LED_CTRL_REG, 1);
}

module_init (aj_led_init);
module_exit (aj_led_cleanup);
